//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.IntRef;

/**
 * Contains detailed speech recognition model information.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public class SpeechRecognitionModel implements AutoCloseable {

    SpeechRecognitionModel(IntRef modelInfo) {
        Contracts.throwIfNull(modelInfo, "modelInfo");

        this.modelInfoHandle = new SafeHandle(modelInfo.getValue(), SafeHandleType.SpeechRecognitionModel);

        this.name = getName(modelInfoHandle);

        String localesString = getLocalesString(modelInfoHandle);
        if (localesString.isEmpty()) {
            this.locales = new ArrayList<String>();
        } else {
            this.locales = Arrays.asList(localesString.split("\\|"));
        }

        this.path = getPath(modelInfoHandle);
        this.version = getVersion(modelInfoHandle);
    }

    /**
     * Gets the model name.
     * @return The model name.
     */
    public String getName() {
        return this.name;
    }

    /**
     * Gets the locales of the model in BCP-47 format.
     * @return The locales of the model.
     */
    public List<String> getLocales() {
        return this.locales;
    }

    /**
     * Gets the model path (only valid for offline models).
     * @return The model path.
     */
    public String getPath() {
        return this.path;
    }

    /**
     * Gets the model version.
     * @return The model version.
     */
    public String getVersion() {
        return this.version;
    }

    /**
     * Explicitly frees any external resource attached to the object.
     */
    public void close() {
        if (this.modelInfoHandle != null) {
            this.modelInfoHandle.close();
            this.modelInfoHandle = null;
        }
    }

    /*! \cond INTERNAL */

    /**
     * Returns the model info implementation.
     * @return The implementation handle.
     */
    public SafeHandle getImpl() {
        return this.modelInfoHandle;
    }

    /*! \endcond */

    private final native String getName(SafeHandle modelInfoHandle);
    private final native String getLocalesString(SafeHandle modelInfoHandle);
    private final native String getPath(SafeHandle modelInfoHandle);
    private final native String getVersion(SafeHandle modelInfoHandle);

    private String name;
    private List<String> locales;
    private String path;
    private String version;
    private SafeHandle modelInfoHandle = null;
}
