//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//

package com.microsoft.cognitiveservices.speech.dialog;

import java.lang.AutoCloseable;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.audio.PullAudioOutputStream;

/**
 * Class that describes the events of a turn status received event.
 */
public class TurnStatusReceivedEventArgs {

    /*! \cond INTERNAL */

    /**
     * Constructs an TurnStatusReceivedEventArgs object.
     * @param eventArgs The native TurnStatusReceivedEventArgs
     */
    public TurnStatusReceivedEventArgs(long eventArgs) {
        Contracts.throwIfNull(eventArgs, "eventArgs");
        TurnStatusReceivedEventHandle = new SafeHandle(eventArgs, SafeHandleType.TurnStatusReceivedEvent);

        StringRef interactionIdRef = new StringRef("");
        Contracts.throwIfFail(getInteractionId(TurnStatusReceivedEventHandle, interactionIdRef));
        interactionId = interactionIdRef.getValue();

        StringRef conversationIdRef = new StringRef("");
        Contracts.throwIfFail(getConversationId(TurnStatusReceivedEventHandle, conversationIdRef));
        conversationId = conversationIdRef.getValue();

        IntRef statusCodeRef = new IntRef(0);
        getStatusCode(TurnStatusReceivedEventHandle, statusCodeRef);
        statusCode = (int)statusCodeRef.getValue();
    }

    /*! \endcond */

    /**
     * Gets the interaction identifier for the turn reporting status. Interaction identifiers are associated with the
     * input signal (e.g. voice) that started a turn.
     * @return The interaction identifier for the turn.
     */
    public String getInteractionId() {
        return interactionId;
    }

    /**
     * Gets the conversation identifier for the turn reporting status. Conversations can span multiple interactions and
     * a long period of time.
     * @return The conversation identifier for the turn.
     */
    public String getConversationId() {
        return conversationId;
    }

    /**
     * Gets the status code for the turn. These codes reflect standard HTTP error codes such as 200 for success, 400 for
     * a failure, and so on.
     * @return The status code reported for the turn.
     */
    public int getStatusCode() {
        return statusCode;
    }

    private SafeHandle TurnStatusReceivedEventHandle = null;
    private String interactionId = "";
    private String conversationId = "";
    private int statusCode = 404;

    private final native long getInteractionId(SafeHandle TurnStatusReceivedEventHandle, StringRef interactionIdRef);
    private final native long getConversationId(SafeHandle TurnStatusReceivedEventHandle, StringRef conversationIdRef);
    private final native long getStatusCode(SafeHandle TurnStatusReceivedEventHandle, IntRef statusCodeRef);
}