//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.intent;

import java.lang.StringBuilder;
import java.util.concurrent.Future;
import java.util.Collection;

import com.microsoft.cognitiveservices.speech.audio.AudioConfig;
import com.microsoft.cognitiveservices.speech.SpeechConfig;
import com.microsoft.cognitiveservices.speech.intent.IntentRecognitionCanceledEventArgs;
import com.microsoft.cognitiveservices.speech.intent.IntentTrigger;
import com.microsoft.cognitiveservices.speech.intent.LanguageUnderstandingModel;
import com.microsoft.cognitiveservices.speech.intent.PatternMatchingEntity;
import com.microsoft.cognitiveservices.speech.intent.PatternMatchingIntent;
import com.microsoft.cognitiveservices.speech.intent.PatternMatchingModel;
import com.microsoft.cognitiveservices.speech.util.EventHandlerImpl;
import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.JsonBuilder;
import com.microsoft.cognitiveservices.speech.util.JsonBuilderJNI;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.AsyncThreadService;
import com.microsoft.cognitiveservices.speech.PropertyId;
import com.microsoft.cognitiveservices.speech.PropertyCollection;
import com.microsoft.cognitiveservices.speech.KeywordRecognitionModel;

/**
 * Performs intent recognition on the speech input. It returns both recognized text and recognized intent.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class IntentRecognizer extends com.microsoft.cognitiveservices.speech.Recognizer {
    /**
     * The event recognizing signals that an intermediate recognition result is received.
     */
    final public EventHandlerImpl<IntentRecognitionEventArgs> recognizing = new EventHandlerImpl<IntentRecognitionEventArgs>(eventCounter);

    /**
     * The event recognized signals that a final recognition result is received.
     */
    final public EventHandlerImpl<IntentRecognitionEventArgs> recognized = new EventHandlerImpl<IntentRecognitionEventArgs>(eventCounter);

    /**
     * The event canceled signals that the intent recognition was canceled.
     */
    final public EventHandlerImpl<IntentRecognitionCanceledEventArgs> canceled = new EventHandlerImpl<IntentRecognitionCanceledEventArgs>(eventCounter);

    /**
     * Creates a new instance of an intent recognizer.
     * @param speechConfig speech configuration.
     */
    public IntentRecognizer(com.microsoft.cognitiveservices.speech.SpeechConfig speechConfig) {
        super(null);

        Contracts.throwIfNull(speechConfig, "speechConfig");
        Contracts.throwIfNull(super.getImpl(), "recoHandle");
        Contracts.throwIfFail(createIntentRecognizerFromConfig(super.getImpl(), speechConfig.getImpl(), null));
        initialize();
    }

    /**
     * Creates a new instance of an intent recognizer with embedded speech configuration.
     * @param embeddedSpeechConfig embedded speech configuration.
     */
    public IntentRecognizer(com.microsoft.cognitiveservices.speech.EmbeddedSpeechConfig embeddedSpeechConfig) {
        super(null);

        Contracts.throwIfNull(embeddedSpeechConfig, "embeddedSpeechConfig");
        Contracts.throwIfNull(super.getImpl(), "recoHandle");
        Contracts.throwIfFail(createIntentRecognizerFromConfig(super.getImpl(), embeddedSpeechConfig.getImpl(), null));

        initialize();
    }

    /**
     * Creates a new instance of an intent recognizer.
     * @param speechConfig speech configuration.
     * @param audioConfig audio configuration.
     */
    public IntentRecognizer(com.microsoft.cognitiveservices.speech.SpeechConfig speechConfig, AudioConfig audioConfig) {
        super(audioConfig);

        Contracts.throwIfNull(speechConfig, "speechConfig");
        if (audioConfig == null) {
            Contracts.throwIfFail(createIntentRecognizerFromConfig(super.getImpl(), speechConfig.getImpl(), null));
        } else {
            Contracts.throwIfFail(createIntentRecognizerFromConfig(super.getImpl(), speechConfig.getImpl(), audioConfig.getImpl()));
        }

        initialize();
    }

    /**
     * Creates a new instance of an intent recognizer with embedded speech configuration.
     * @param embeddedSpeechConfig embedded speech configuration.
     * @param audioConfig audio configuration.
     */
    public IntentRecognizer(com.microsoft.cognitiveservices.speech.EmbeddedSpeechConfig embeddedSpeechConfig, AudioConfig audioConfig) {
        super(audioConfig);

        Contracts.throwIfNull(embeddedSpeechConfig, "embeddedSpeechConfig");
        Contracts.throwIfNull(super.getImpl(), "recoHandle");

        if (audioConfig == null) {
            Contracts.throwIfFail(createIntentRecognizerFromConfig(super.getImpl(), embeddedSpeechConfig.getImpl(), null));
        } else {
            Contracts.throwIfFail(createIntentRecognizerFromConfig(super.getImpl(), embeddedSpeechConfig.getImpl(), audioConfig.getImpl()));
        }

        initialize();
    }

    /**
     * Gets the spoken language of recognition.
     * @return the spoken language of recognition.
     */
    public String getSpeechRecognitionLanguage() {
        return propertyHandle.getProperty(PropertyId.SpeechServiceConnection_RecoLanguage);
    }

    /**
     * Sets the authorization token used to communicate with the service.
     * Note: The caller needs to ensure that the authorization token is valid. Before the authorization token expires,
     * the caller needs to refresh it by calling this setter with a new valid token.
     * Otherwise, the recognizer will encounter errors during recognition.
     * @param token Authorization token.
     */
    public void setAuthorizationToken(String token) {
        Contracts.throwIfNullOrWhitespace(token, "token");
        propertyHandle.setProperty(PropertyId.SpeechServiceAuthorization_Token, token);
    }

    /**
     * Gets the authorization token used to communicate with the service.
     * @return Authorization token.
     */
    public String getAuthorizationToken() {
        return propertyHandle.getProperty(PropertyId.SpeechServiceAuthorization_Token);
    }

    /**
     * The collection of properties and their values defined for this IntentRecognizer.
     * @return The collection of properties and their values defined for this IntentRecognizer.
     */
    public PropertyCollection getProperties() {
        return propertyHandle;
    }

    /**
     * Starts intent recognition, and returns after a single utterance is recognized. The end of a
     * single utterance is determined by listening for silence at the end or until a maximum of 15
     * seconds of audio is processed.  The task returns the recognition text as result.
     * Note: Since recognizeOnceAsync() returns only a single utterance, it is suitable only for single
     * shot recognition like command or query.
     * For long-running multi-utterance recognition, use startContinuousRecognitionAsync() instead.
     * @return A task representing the recognition operation. The task returns a value of
     * {@code IntentRecognitionResult}
     */
    public Future<IntentRecognitionResult> recognizeOnceAsync() {
        final IntentRecognizer thisReco = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<IntentRecognitionResult>() {
            public IntentRecognitionResult  call() {
                // A variable defined in an enclosing scope must be final or effectively final.
                // The compiler treats an array initialized once as an effectively final.
                final IntentRecognitionResult[] result = new IntentRecognitionResult[1];

                Runnable runnable = new Runnable() { public void run() { result[0] = new IntentRecognitionResult(thisReco.recognize()); }};
                thisReco.doAsyncRecognitionAction(runnable);

                return result[0];
        }});
    }

    /**
     * Performs intent recognition, and generates a result from the text passed
     * in. This is useful for testing and other times when the speech input is
     * not tied to the IntentRecognizer.
     * Note: The Intent Service does not currently support this so it is only
     * valid for offline pattern matching or exact matching intents.
     * @param text The text to be recognized for intent.
     * @return A task representing the recognition operation. The task returns
     * a value of {@link com.microsoft.cognitiveservices.speech.intent.IntentRecognitionResult}.
     */
    public Future<IntentRecognitionResult> recognizeOnceAsync(String text) {
        final IntentRecognizer thisReco = this;
        final String thisText = text;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<IntentRecognitionResult>() {
            public IntentRecognitionResult  call() {
                // A variable defined in an enclosing scope must be final or effectively final.
                // The compiler treats an array initialized once as an effectively final.
                final IntentRecognitionResult[] result = new IntentRecognitionResult[1];

                Runnable runnable = new Runnable() { 
                    public void run() { 
                        // Java does not have pass by reference for primitives. So we must
                        // create an array and pass that.
                        IntRef resultRef = new IntRef(0);
                        Contracts.throwIfNull(thisReco, "Invalid recognizer handle");
                        Contracts.throwIfFail(recognizeTextOnce(thisReco.getImpl(), thisText, resultRef));
                        result[0] = new IntentRecognitionResult(resultRef.getValue());
                    }};
                thisReco.doAsyncRecognitionAction(runnable);

                return result[0];
        }});
    }

    /**
     * Starts speech recognition on a continuous audio stream, until stopContinuousRecognitionAsync() is called.
     * User must subscribe to events to receive recognition results.
     * @return A task representing the asynchronous operation that starts the recognition.
     */
    public Future<Void> startContinuousRecognitionAsync() {
        final IntentRecognizer thisReco = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { thisReco.startContinuousRecognition(getImpl()); }};
                thisReco.doAsyncRecognitionAction(runnable);
                return null;
        }});
    }

    /**
     * Stops a running recognition operation as soon as possible and immediately requests a result based on the
     * the input that has been processed so far. This works for all recognition operations, not just continuous
     * ones, and facilitates the use of push-to-talk or "finish now" buttons for manual audio endpointing.
     * @return A future that will complete when input processing has been stopped. Result generation, if applicable for the input provided, may happen after this task completes and should be handled with the appropriate event.
     */
    public Future<Void> stopContinuousRecognitionAsync() {
        final IntentRecognizer thisReco = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { thisReco.stopContinuousRecognition(getImpl()); }};
                thisReco.doAsyncRecognitionAction(runnable);
                return null;
        }});
    }

    /**
     * Adds a simple phrase that may be spoken by the user, indicating a specific user intent.
     * @param simplePhrase The phrase corresponding to the intent.
     */
    public void addIntent(String simplePhrase) {
        Contracts.throwIfNullOrWhitespace(simplePhrase, "simplePhrase");

        IntentTrigger trigger = IntentTrigger.fromPhrase(simplePhrase);
        Contracts.throwIfFail(addIntent(super.getImpl(), simplePhrase, trigger.getImpl()));
    }

    /**
     * Adds a simple phrase that may be spoken by the user, indicating a specific user intent.
     * @param simplePhrase The phrase corresponding to the intent.
     * @param intentId A custom id String to be returned in the IntentRecognitionResult's getIntentId() method.
     */
    public void addIntent(String simplePhrase, String intentId) {
        Contracts.throwIfNullOrWhitespace(simplePhrase, "simplePhrase");
        Contracts.throwIfNullOrWhitespace(intentId, "intentId");

        IntentTrigger trigger = IntentTrigger.fromPhrase(simplePhrase);
        Contracts.throwIfFail(addIntent(super.getImpl(), intentId, trigger.getImpl()));
    }

    /**
     * Adds a single intent by name from the specified Language Understanding Model.
     * @param model The language understanding model containing the intent.
     * @param intentName The name of the single intent to be included from the language understanding model.
     */
    public void addIntent(LanguageUnderstandingModel model, String intentName) {
        Contracts.throwIfNull(model, "model");
        Contracts.throwIfNullOrWhitespace(intentName, "intentName");

        IntentTrigger trigger = IntentTrigger.fromModel(model.getImpl(), intentName);
        Contracts.throwIfFail(addIntent(super.getImpl(), intentName, trigger.getImpl()));
    }

    /**
     * Adds a single intent by name from the specified Language Understanding Model.
     * @param model The language understanding model containing the intent.
     * @param intentName The name of the single intent to be included from the language understanding model.
     * @param intentId A custom id String to be returned in the IntentRecognitionResult's getIntentId() method.
     */
    public void addIntent(LanguageUnderstandingModel model, String intentName, String intentId) {
        Contracts.throwIfNull(model, "model");
        Contracts.throwIfNullOrWhitespace(intentName, "intentName");
        Contracts.throwIfNullOrWhitespace(intentId, "intentId");

        IntentTrigger trigger = IntentTrigger.fromModel(model.getImpl(), intentName);
        Contracts.throwIfFail(addIntent(super.getImpl(), intentId, trigger.getImpl()));
    }

    /**
     * Adds all intents from the specified Language Understanding Model.
     * @param model The language understanding model containing the intents.
     * @param intentId A custom id String to be returned in the IntentRecognitionResult's getIntentId() method.
     */
    public void addAllIntents(LanguageUnderstandingModel model, String intentId) {
        Contracts.throwIfNull(model, "model");
        Contracts.throwIfNullOrWhitespace(intentId, "intentId");

        IntentTrigger trigger = IntentTrigger.fromModel(model.getImpl());
        Contracts.throwIfFail(addIntent(super.getImpl(), intentId, trigger.getImpl()));
    }

    /**
     * Adds all intents from the specified Language Understanding Model.
     * @param model The language understanding model containing the intents.
     */
    public void addAllIntents(LanguageUnderstandingModel model) {
        Contracts.throwIfNull(model, "model");
        IntentTrigger trigger = IntentTrigger.fromModel(model.getImpl());
        Contracts.throwIfFail(addIntent(super.getImpl(), null, trigger.getImpl()));
    }

    /**
     * Configures the recognizer with the given keyword model. After calling this method, the recognizer is listening 
     * for the keyword to start the recognition. Call stopKeywordRecognitionAsync() to end the keyword initiated recognition.
     * User must subscribe to events to receive recognition results.
     * @param model The keyword recognition model that specifies the keyword to be recognized.
     * @return A task representing the asynchronous operation that starts the recognition.
     */
    public Future<Void> startKeywordRecognitionAsync(KeywordRecognitionModel model) {
        Contracts.throwIfNull(model, "model");

        final IntentRecognizer thisReco = this;
        final KeywordRecognitionModel model2 = model;
        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { thisReco.startKeywordRecognition(getImpl(), model2.getImpl()); }};
                thisReco.doAsyncRecognitionAction(runnable);
                return null;
        }});
    }

    /**
     * Ends the keyword initiated recognition.
     * @return A task representing the asynchronous operation that stops the recognition.
     */
    public Future<Void> stopKeywordRecognitionAsync() {
        final IntentRecognizer thisReco = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { thisReco.stopKeywordRecognition(getImpl()); }};
                thisReco.doAsyncRecognitionAction(runnable);
                return null;
        }});
    }

    /**
     * Takes a collection of language understanding models, makes a copy of 
     * them, and applies them to the recognizer. This application takes effect
     * at different times depending on the LanguageUnderstandingModel type.
     * PatternMatchingModels will become active immediately whereas 
     * LanguageUnderstandingModels utilizing the LUIS service will become 
     * active immediately unless the recognizer is in the middle of intent
     * recognition in which case it will take effect after the next Recognized
     * event.
     * This replaces any previously applied models.
     * @throws NullPointerException If the collection passed is null.
     * @param collection A collection of shared pointers to LanguageUnderstandingModels.
     * @return True if the application of the models takes effect immediately. Otherwise false.
     */
    public boolean applyLanguageModels(Collection<LanguageUnderstandingModel> collection) throws NullPointerException
    {
        Contracts.throwIfNull(collection, "collection");
        boolean result = true;

        // Clear existing language models.
        Contracts.throwIfFail(clearLanguageModels(super.getImpl()));

        // Add the new ones.
        for(LanguageUnderstandingModel model : collection)
        {
                Contracts.throwIfNull(model, "model");
                // Check for subtype first as a PatternMatchingModel is a LanguageUnderstandingModel.
                if (model instanceof PatternMatchingModel)
                {
                    PatternMatchingModel simpleModel = (PatternMatchingModel)model;
                    String json = buildModelJson(simpleModel);
                    Contracts.throwIfFail(importPatternMatchingModel(super.getImpl(), json.toString()));
                }
                else if (model instanceof LanguageUnderstandingModel)
                {
                    IntentTrigger intentTrigger = IntentTrigger.fromModel(model.getImpl());
                    Contracts.throwIfFail(addIntent(super.getImpl(), null, intentTrigger.getImpl()));
                    result = false;
                }
            }
        return result;
    }

    /*
     * Disposes the associated resources.
     * 
     * @param disposing whether to dispose managed resources.
     */
    @Override
    protected void dispose(final boolean disposing) {
        if (disposed) {
            return;
        }

        if (disposing) {

            if (propertyHandle != null) {
                propertyHandle.close();
                propertyHandle = null;
            }
            intentRecognizerObjects.remove(this);
            super.dispose(disposing);
        }
    }

    /**
     * This is used to keep any instance of this class alive that is subscribed to downstream events.
     */
    static java.util.Set<IntentRecognizer> intentRecognizerObjects = java.util.Collections.synchronizedSet(new java.util.HashSet<IntentRecognizer>());

    private void initialize() {
        final IntentRecognizer _this = this;

        this.recognizing.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                intentRecognizerObjects.add(_this);
                Contracts.throwIfFail(recognizingSetCallback(_this.getImpl().getValue()));
            }
        });

        this.recognized.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                intentRecognizerObjects.add(_this);
                Contracts.throwIfFail(recognizedSetCallback(_this.getImpl().getValue()));                
            }
        });

        this.canceled.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                intentRecognizerObjects.add(_this);
                Contracts.throwIfFail(canceledSetCallback(_this.getImpl().getValue()));                
            }
        });

        this.sessionStarted.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                intentRecognizerObjects.add(_this);
                Contracts.throwIfFail(sessionStartedSetCallback(_this.getImpl().getValue()));
            }
        });

        this.sessionStopped.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                intentRecognizerObjects.add(_this);
                Contracts.throwIfFail(sessionStoppedSetCallback(_this.getImpl().getValue()));                
            }
        });

        this.speechStartDetected.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                intentRecognizerObjects.add(_this);
                Contracts.throwIfFail(speechStartDetectedSetCallback(_this.getImpl().getValue()));
            }
        });

        this.speechEndDetected.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                intentRecognizerObjects.add(_this);
                Contracts.throwIfFail(speechEndDetectedSetCallback(_this.getImpl().getValue()));                
            }
        });

        IntRef propHandle = new IntRef(0);
        Contracts.throwIfFail(getPropertyBagFromRecognizerHandle(_this.getImpl(), propHandle));
        propertyHandle = new PropertyCollection(propHandle);
    }

    private void recognizingEventCallback(long eventArgs)
    {
        try {
            Contracts.throwIfNull(this, "recognizer");
            if (this.disposed) {
                return;
            }
            IntentRecognitionEventArgs resultEventArg = new IntentRecognitionEventArgs(eventArgs, true);
            EventHandlerImpl<IntentRecognitionEventArgs> handler = this.recognizing;
            if (handler != null) {
                handler.fireEvent(this, resultEventArg);
            }    
        } catch (Exception e) {}
    }

    private void recognizedEventCallback(long eventArgs)
    {
        try {
            Contracts.throwIfNull(this, "recognizer");
            if (this.disposed) {
                return;
            }
            IntentRecognitionEventArgs resultEventArg = new IntentRecognitionEventArgs(eventArgs, true);
            EventHandlerImpl<IntentRecognitionEventArgs> handler = this.recognized;
            if (handler != null) {
                handler.fireEvent(this, resultEventArg);
            }    
        } catch (Exception e) {}
    }

    private void canceledEventCallback(long eventArgs)
    {
        try {
            Contracts.throwIfNull(this, "recognizer");
            if (this.disposed) {
                return;
            }
            IntentRecognitionCanceledEventArgs resultEventArg = new IntentRecognitionCanceledEventArgs(eventArgs, true);
            EventHandlerImpl<IntentRecognitionCanceledEventArgs> handler = this.canceled;
            if (handler != null) {
                handler.fireEvent(this, resultEventArg);
            }    
        } catch (Exception e) {}
    }

    private String buildModelJson(PatternMatchingModel simpleModel)
    {
        JsonBuilder jsonBuilder = new JsonBuilder();
        int modelItem = jsonBuilder.addItem(jsonBuilder.root, 0, "modelId");
        jsonBuilder.setString(modelItem, simpleModel.getId());

        int intentsArrayRootItem = jsonBuilder.addItem(jsonBuilder.root, 0, "intents");
        jsonBuilder.setJson(intentsArrayRootItem, "[]");
        int index = 0;
        for (PatternMatchingIntent intent : simpleModel.getIntents().values())
        {
            int intentItem = jsonBuilder.addItem(intentsArrayRootItem, index, null);
            int idItem = jsonBuilder.addItem(intentItem, 0, "id");
            jsonBuilder.setString(idItem, intent.getId());
            int phrasesArrayRootItem = jsonBuilder.addItem(intentItem, 0, "phrases");
            int phraseIndex = 0;
            jsonBuilder.setJson(phrasesArrayRootItem, "[]");
            for (String phrase : intent.Phrases)
            {
                int phraseItem = jsonBuilder.addItem(phrasesArrayRootItem, phraseIndex, null);
                jsonBuilder.setString(phraseItem, phrase);
                phraseIndex++;
            }
            index++;
        }

        index = 0;
        int entitiesArrayRootItem = jsonBuilder.addItem(jsonBuilder.root, 0, "entities");
        jsonBuilder.setJson(entitiesArrayRootItem, "[]");
        for (PatternMatchingEntity entity : simpleModel.getEntities().values())
        {
            int entityItem = jsonBuilder.addItem(entitiesArrayRootItem, index, null);
            int idItem = jsonBuilder.addItem(entityItem, 0, "id");
            jsonBuilder.setString(idItem, entity.getId());
            int typeItem = jsonBuilder.addItem(entityItem, 0, "type");
            jsonBuilder.setInteger(typeItem, (int)entity.getType().getValue());
            int modeItem = jsonBuilder.addItem(entityItem, 0, "mode");
            jsonBuilder.setInteger(modeItem, (int)entity.getMatchMode().getValue());
            int phrasesArrayRootItem = jsonBuilder.addItem(entityItem, 0, "phrases");
            int phraseIndex = 0;
            jsonBuilder.setJson(phrasesArrayRootItem, "[]");
            for (String phrase : entity.Phrases)
            {
                int phraseItem = jsonBuilder.addItem(phrasesArrayRootItem, phraseIndex, null);
                jsonBuilder.setString(phraseItem, phrase);
                phraseIndex++;
            }
            index++;
        }

        return jsonBuilder.toString();
    }

    private final native long addIntent(SafeHandle recoHandle, String intentId, SafeHandle triggerHandle);
    private final native long clearLanguageModels(SafeHandle recoHandle);
    private final native long createIntentRecognizerFromConfig(SafeHandle recoHandle, SafeHandle speechConfigHandle, SafeHandle audioConfigHandle);
    private final native long importPatternMatchingModel(SafeHandle recoHandle, String modelJson);
    private final native long recognizeTextOnce(SafeHandle recoHandle, String text,  IntRef resultRef);

    private PropertyCollection propertyHandle = null;
}
