//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.speaker;

import java.lang.AutoCloseable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.PropertyCollection;

/**
 * A VoiceProfile represents a speaker's uniqueness in his/her voice.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class VoiceProfile implements AutoCloseable {

    /**
     * Initializes a new instance of Voice Profile.
     * @param id An unique id.
     * @param type A type of the Voice Profile.
    */
    public VoiceProfile(String id, VoiceProfileType type) {
        IntRef result = new IntRef(0);
        Contracts.throwIfFail(createVoiceProfileFromIdAndType(id, type.getValue(), result));
        this.voiceProfileHandle = new SafeHandle(result.getValue(), SafeHandleType.VoiceProfile);
    }

    /*! \cond INTERNAL */

    /**
     * Initializes a new instance of Voice Profile.
     * @param handle handle to the VoiceProfile.
     */
    public VoiceProfile(long handle) {
        Contracts.throwIfNull(handle, "handle");
        this.voiceProfileHandle = new SafeHandle(handle, SafeHandleType.VoiceProfile);
    }

    /*! \endcond */

    /**
     * Get a voice profile id.
     * @return The voice profile id.
     */
    public String getId() {
        StringRef result = new StringRef("");
        Contracts.throwIfFail(getId(voiceProfileHandle, result));
        return result.getValue();
    }

    /**
     * Get a voice profile type.
     * @return The voice profile type.
     */
    public VoiceProfileType getType() {
        IntRef result = new IntRef(0);
        Contracts.throwIfFail(getType(voiceProfileHandle, result));
        return VoiceProfileType.values()[(int)result.getValue() - 1];
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    public void close() {
        if (disposed) {
            return;
        }
        if (this.voiceProfileHandle != null) {
            this.voiceProfileHandle.close();
            this.voiceProfileHandle = null;
        }
        disposed = true;
    }

    /*! \cond INTERNAL */

    /**
     * Returns the voice profile implementation.
     * @return The implementation handle.
     */
    public SafeHandle getImpl() {
        return this.voiceProfileHandle;
    }

    /*! \endcond */

    private SafeHandle voiceProfileHandle = null;
    private boolean disposed = false;

    private final native long createVoiceProfileFromIdAndType(String id, int type, IntRef result);
    private final native long getId(SafeHandle voiceProfileHandle, StringRef result);
    private final native long getType(SafeHandle voiceProfileHandle, IntRef result);
}
