//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.transcription;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.Future;

import com.microsoft.cognitiveservices.speech.audio.AudioConfig;
import com.microsoft.cognitiveservices.speech.SpeechConfig;
import com.microsoft.cognitiveservices.speech.SourceLanguageConfig;
import com.microsoft.cognitiveservices.speech.AutoDetectSourceLanguageConfig;
import com.microsoft.cognitiveservices.speech.transcription.ConversationTranscriptionCanceledEventArgs;
import com.microsoft.cognitiveservices.speech.util.EventHandlerImpl;
import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.AsyncThreadService;
import com.microsoft.cognitiveservices.speech.PropertyId;
import com.microsoft.cognitiveservices.speech.OutputFormat;
import com.microsoft.cognitiveservices.speech.PropertyCollection;

/**
 * Performs conversation transcribing for audio input streams, and gets transcribed text and speaker id as a result.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class ConversationTranscriber extends com.microsoft.cognitiveservices.speech.Recognizer
{
    /**
     * The event transcribing signals that an intermediate recognition result is received.
     */
    final public EventHandlerImpl<ConversationTranscriptionEventArgs> transcribing = new EventHandlerImpl<ConversationTranscriptionEventArgs>(eventCounter);

    /**
     * The event transcribed signals that a final recognition result is received.
     */
    final public EventHandlerImpl<ConversationTranscriptionEventArgs> transcribed = new EventHandlerImpl<ConversationTranscriptionEventArgs>(eventCounter);

    /**
     * The event canceled signals that the recognition was canceled.
     */
    final public EventHandlerImpl<ConversationTranscriptionCanceledEventArgs> canceled = new EventHandlerImpl<ConversationTranscriptionCanceledEventArgs>(eventCounter);

    /**
     * Initializes a new instance of Conversation Transcriber.
     * @param speechConfig speech configuration.
     */
    public ConversationTranscriber(SpeechConfig speechConfig) {
        super(null);
        Contracts.throwIfNull(speechConfig, "speechConfig");
        Contracts.throwIfNull(super.getImpl(), "recoHandle");
        Contracts.throwIfFail(createConversationTranscriberFromConfig(super.getImpl(), speechConfig.getImpl(), null));
        initialize();
    }

    /**
     * Initializes a new instance of Conversation Transcriber.
     * @param speechConfig speech configuration.
     * @param audioConfig audio configuration.
     */
    public ConversationTranscriber(SpeechConfig speechConfig, AudioConfig audioConfig) {
        super(audioConfig);

        Contracts.throwIfNull(speechConfig, "speechConfig");
        if (audioConfig == null) {
            Contracts.throwIfFail(createConversationTranscriberFromConfig(super.getImpl(), speechConfig.getImpl(), null));
        } else {
            Contracts.throwIfFail(createConversationTranscriberFromConfig(super.getImpl(), speechConfig.getImpl(), audioConfig.getImpl()));
        }
        initialize();
    }    

    /**
     * Initializes a new instance of Conversation Transcriber.
     * @param speechConfig speech configuration.
     * @param autoDetectSourceLangConfig the configuration for auto detecting source language
     */
    public ConversationTranscriber(SpeechConfig speechConfig, AutoDetectSourceLanguageConfig autoDetectSourceLangConfig) {
        super(null);

        Contracts.throwIfNull(speechConfig, "speechConfig");
        Contracts.throwIfNull(autoDetectSourceLangConfig, "autoDetectSourceLangConfig");
        Contracts.throwIfFail(createConversationTranscriberFromAutoDetectSourceLangConfig(super.getImpl(), speechConfig.getImpl(), autoDetectSourceLangConfig.getImpl(), null));
        initialize();
    }    

    /**
     * Initializes a new instance of Conversation Transcriber.
     * @param speechConfig speech configuration.
     * @param autoDetectSourceLangConfig the configuration for auto detecting source language
     * @param audioConfig audio configuration.
     */
    public ConversationTranscriber(SpeechConfig speechConfig, AutoDetectSourceLanguageConfig autoDetectSourceLangConfig, AudioConfig audioConfig) {
        super(audioConfig);

        Contracts.throwIfNull(speechConfig, "speechConfig");
        Contracts.throwIfNull(autoDetectSourceLangConfig, "autoDetectSourceLangConfig");
        if (audioConfig == null) {
            Contracts.throwIfFail(createConversationTranscriberFromAutoDetectSourceLangConfig(super.getImpl(), speechConfig.getImpl(), autoDetectSourceLangConfig.getImpl(), null));
        } else {
            Contracts.throwIfFail(createConversationTranscriberFromAutoDetectSourceLangConfig(super.getImpl(), speechConfig.getImpl(), autoDetectSourceLangConfig.getImpl(), audioConfig.getImpl()));
        }
        initialize();
    }    

    /**
     * Initializes a new instance of Conversation Transcriber.
     * @param speechConfig speech configuration.
     * @param sourceLanguageConfig the configuration for source language
     */
    public ConversationTranscriber(SpeechConfig speechConfig, SourceLanguageConfig sourceLanguageConfig) {
        super(null);

        Contracts.throwIfNull(speechConfig, "speechConfig");
        Contracts.throwIfNull(sourceLanguageConfig, "sourceLanguageConfig");
        Contracts.throwIfFail(createConversationTranscriberFromSourceLangConfig(super.getImpl(), speechConfig.getImpl(), sourceLanguageConfig.getImpl(), null));
        initialize();
    }

    /**
     * Initializes a new instance of Conversation Transcriber.
     * @param speechConfig speech configuration.
     * @param sourceLanguageConfig the configuration for source language
     * @param audioConfig audio configuration.
     */
    public ConversationTranscriber(SpeechConfig speechConfig, SourceLanguageConfig sourceLanguageConfig, AudioConfig audioConfig) {
        super(audioConfig);

        Contracts.throwIfNull(speechConfig, "speechConfig");
        Contracts.throwIfNull(sourceLanguageConfig, "sourceLanguageConfig");
        if (audioConfig == null) {
            Contracts.throwIfFail(createConversationTranscriberFromSourceLangConfig(super.getImpl(), speechConfig.getImpl(), sourceLanguageConfig.getImpl(), null));
        } else {
            Contracts.throwIfFail(createConversationTranscriberFromSourceLangConfig(super.getImpl(), speechConfig.getImpl(), sourceLanguageConfig.getImpl(), audioConfig.getImpl()));
        }
        initialize();
    }    

    /**
     * Initializes a new instance of Conversation Transcriber.
     * @param speechConfig speech configuration.
     * @param sourceLanguage the recognition source language
     */
    public ConversationTranscriber(SpeechConfig speechConfig, String sourceLanguage) {
        super(null);

        Contracts.throwIfNull(speechConfig, "speechConfig");
        Contracts.throwIfIllegalLanguage(sourceLanguage, "invalid language value");
        Contracts.throwIfFail(createConversationTranscriberFromSourceLangConfig(super.getImpl(), speechConfig.getImpl(), SourceLanguageConfig.fromLanguage(sourceLanguage).getImpl(), null));
        initialize();
    }    

    /**
     * Initializes a new instance of Conversation Transcriber.
     * @param speechConfig speech configuration.
     * @param sourceLanguage the recognition source language
     * @param audioConfig audio configuration.
     */
    public ConversationTranscriber(SpeechConfig speechConfig, String sourceLanguage, AudioConfig audioConfig) {
        super(audioConfig);

        Contracts.throwIfNull(speechConfig, "speechConfig");
        Contracts.throwIfIllegalLanguage(sourceLanguage, "invalid language value");
        if (audioConfig == null) {
            Contracts.throwIfFail(createConversationTranscriberFromSourceLangConfig(super.getImpl(), speechConfig.getImpl(), SourceLanguageConfig.fromLanguage(sourceLanguage).getImpl(), null));
        } else {
            Contracts.throwIfFail(createConversationTranscriberFromSourceLangConfig(super.getImpl(), speechConfig.getImpl(), SourceLanguageConfig.fromLanguage(sourceLanguage).getImpl(), audioConfig.getImpl()));
        }
        initialize();
    }

    /**
     * Sets the authorization token used to communicate with the service.
     * Note: The caller needs to ensure that the authorization token is valid. Before the authorization token expires,
     * the caller needs to refresh it by calling this setter with a new valid token.
     * Otherwise, the recognizer will encounter errors during recognition.
     * @param token Authorization token.
     */
    public void setAuthorizationToken(String token) {
        Contracts.throwIfNullOrWhitespace(token, "token");
        propertyHandle.setProperty(PropertyId.SpeechServiceAuthorization_Token, token);
    }

    /**
     * Gets the authorization token used to communicate with the service.
     * @return Authorization token.
     */
    public String getAuthorizationToken() {
        return propertyHandle.getProperty(PropertyId.SpeechServiceAuthorization_Token);
    }    

    /**
     * Gets the spoken language of recognition.
     * @return The spoken language of recognition.
     */
    public String getSpeechRecognitionLanguage() {
        return propertyHandle.getProperty(PropertyId.SpeechServiceConnection_RecoLanguage);
    }

    /**
     * Gets the output format of recognition.
     * @return The output format of recognition.
     */
    public OutputFormat getOutputFormat() {
        if (propertyHandle.getProperty(PropertyId.SpeechServiceResponse_RequestDetailedResultTrueFalse).equals("true")) {
            return OutputFormat.Detailed;
        } else {
            return OutputFormat.Simple;
        }
    }

    /**
     * The collection of properties and their values defined for this ConversationTranscriber.
     * @return The collection of properties and their values defined for this ConversationTranscriber.
     */
    public PropertyCollection getProperties() {
        return propertyHandle;
    }

    /**
     * Starts conversation transcribing on a continuous audio stream, until stopTranscribingAsync() is called.
     * User must subscribe to events to receive transcription results.
     * @return A task representing the asynchronous operation that starts the transcription.
     */
    public Future<Void> startTranscribingAsync() {
        final ConversationTranscriber thisReco = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { thisReco.startContinuousRecognition(getImpl()); }};
                thisReco.doAsyncRecognitionAction(runnable);
                return null;
        }});
    }

    /**
     *  Stops conversation transcribing.
     *
     *  @return A task representing the asynchronous operation that stops the transcription.
     */
    public Future<Void> stopTranscribingAsync() {
        final ConversationTranscriber thisReco = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { thisReco.stopContinuousRecognition(getImpl()); }};
                thisReco.doAsyncRecognitionAction(runnable);
                return null;
        }});
    }

    /*
     * Dispose of associated resources.
     * 
     * @param disposing whether to dispose managed resources.
     */
    @Override
    protected void dispose(final boolean disposing) {
        if (disposed) {
            return;
        }

        if (disposing) {
                        
            if (propertyHandle != null) {
                propertyHandle.close();
                propertyHandle = null;
            }
            conversationTranscriberObjects.remove(this);
            super.dispose(disposing);
        }
    }

    /**
     * This is used to keep any instance of this class alive that is subscribed to downstream events.
     */
    static java.util.Set<ConversationTranscriber> conversationTranscriberObjects = java.util.Collections.synchronizedSet(new java.util.HashSet<ConversationTranscriber>());

    private void initialize() {
        final ConversationTranscriber _this = this;

        this.transcribing.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                conversationTranscriberObjects.add(_this);
                Contracts.throwIfFail(recognizingSetCallback(_this.getImpl().getValue()));
            }
        });

        this.transcribed.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                conversationTranscriberObjects.add(_this);
                Contracts.throwIfFail(recognizedSetCallback(_this.getImpl().getValue()));                
            }
        });

        this.canceled.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                conversationTranscriberObjects.add(_this);
                Contracts.throwIfFail(canceledSetCallback(_this.getImpl().getValue()));                
            }
        });

        this.sessionStarted.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                conversationTranscriberObjects.add(_this);
                Contracts.throwIfFail(sessionStartedSetCallback(_this.getImpl().getValue()));
            }
        });

        this.sessionStopped.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                conversationTranscriberObjects.add(_this);
                Contracts.throwIfFail(sessionStoppedSetCallback(_this.getImpl().getValue()));                
            }
        });

        this.speechStartDetected.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                conversationTranscriberObjects.add(_this);
                Contracts.throwIfFail(speechStartDetectedSetCallback(_this.getImpl().getValue()));
            }
        });

        this.speechEndDetected.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                conversationTranscriberObjects.add(_this);
                Contracts.throwIfFail(speechEndDetectedSetCallback(_this.getImpl().getValue()));                
            }
        });

        IntRef propHandle = new IntRef(0);
        Contracts.throwIfFail(getPropertyBagFromRecognizerHandle(_this.getImpl(), propHandle));
        propertyHandle = new PropertyCollection(propHandle);
    }

    private void recognizingEventCallback(long eventArgs)
    {
        try {
            Contracts.throwIfNull(this, "recognizer");
            if (this.disposed) {
                return;
            }
            ConversationTranscriptionEventArgs resultEventArg = new ConversationTranscriptionEventArgs(eventArgs, true);
            EventHandlerImpl<ConversationTranscriptionEventArgs> handler = this.transcribing;
            if (handler != null) {
                handler.fireEvent(this, resultEventArg);
            }    
        } catch (Exception e) {}
    }

    private void recognizedEventCallback(long eventArgs)
    {
        try {
            Contracts.throwIfNull(this, "recognizer");
            if (this.disposed) {
                return;
            }
            ConversationTranscriptionEventArgs resultEventArg = new ConversationTranscriptionEventArgs(eventArgs, true);
            EventHandlerImpl<ConversationTranscriptionEventArgs> handler = this.transcribed;
            if (handler != null) {
                handler.fireEvent(this, resultEventArg);
            }    
        } catch (Exception e) {}
    }

    private void canceledEventCallback(long eventArgs)
    {
        try {
            Contracts.throwIfNull(this, "recognizer");
            if (this.disposed) {
                return;
            }
            ConversationTranscriptionCanceledEventArgs resultEventArg = new ConversationTranscriptionCanceledEventArgs(eventArgs, true);
            EventHandlerImpl<ConversationTranscriptionCanceledEventArgs> handler = this.canceled;
            if (handler != null) {
                handler.fireEvent(this, resultEventArg);
            }    
        } catch (Exception e) {}
    }

    private final native long createConversationTranscriberFromConfig(SafeHandle recoHandle, SafeHandle speechConfigHandle, SafeHandle audioConfigHandle);
    private final native long createConversationTranscriberFromAutoDetectSourceLangConfig(SafeHandle recoHandle, SafeHandle speechConfigHandle, SafeHandle autoDetectSourceLangConfigHandle, SafeHandle audioConfigHandle);      
    private final native long createConversationTranscriberFromSourceLangConfig(SafeHandle recoHandle, SafeHandle speechConfigHandle, SafeHandle SourceLangConfigHandle, SafeHandle audioConfigHandle);

    private PropertyCollection propertyHandle = null;
}
