//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.transcription;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.SpeechRecognitionResult;
import com.microsoft.cognitiveservices.speech.util.StringRef;

/**
 * Class that defines conversation transcription result.
 */
public class ConversationTranscriptionResult extends SpeechRecognitionResult {

    /**
     * Internal constructor for creating a ConversationTranscriptionResult from a native handle.
     * 
     * @param result The native handle.
     */
    protected ConversationTranscriptionResult(long result) {
        super(result);

        if (result != 0) {
            StringRef speakerIdStr = new StringRef("");
            Contracts.throwIfFail(getSpeakerId(super.getImpl(), speakerIdStr));
            this.speakerId = speakerIdStr.getValue();
        }
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    @Override
    public void close() {
        super.close();
    }

    /**
     * A String that represents the speaker id in the conversation.
     * @return the speaker ID string.
     */
    public String getSpeakerId() {
        return this.speakerId;
    }

    /**
     * Returns a String that represents the conversation transcription result.
     * @return A String that represents the conversation transcription result.
     */
    @Override
    public String toString() {
        return "ResultId:" + this.getResultId()+
                " Status:" + this.getReason() +
                " SpeakerId:" + this.speakerId +
                " Recognized text:<" + this.getText() + ">.";
    }

    private final native long getSpeakerId(SafeHandle resultHandle, StringRef speakerId);

    private String speakerId;
}
