//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//

package com.microsoft.cognitiveservices.speech.util;

import com.microsoft.cognitiveservices.speech.util.JsonValueJNI;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;

/*! \cond INTERNAL */
/**
 * The type Json value.
 */

public class JsonValue implements AutoCloseable {

    /**
     * Instantiates a new Json value.
     *
     * @param item   the item
     * @param handle the handle
     */
    public JsonValue(int item, SafeHandle handle) {
       this.item = item;
       parserHandle = handle;
    }

    /**
     * Parse json value.
     *
     * @param json the json string
     * @return the json value
     */
    public static JsonValue Parse(String json) {
        return JsonValueJNI.createParser(json);
    }

    /**
     * As string.
     *
     * @return the string value
     */
    public String asString() {
        return JsonValueJNI.asString(parserHandle, item);
    }

    /**
     * As int.
     *
     * @return the long value
     */
    public long asInt() {
        return JsonValueJNI.asInt(parserHandle, item);
    }

    /**
     * As float.
     *
     * @return the float value
     */
    public float asFloat() {
        return JsonValueJNI.asFloat(parserHandle, item);
    }

    /**
     * As boolean.
     *
     * @return the boolean value
     */
    public boolean asBoolean() {
        return JsonValueJNI.asBoolean(parserHandle, item);
    }

    /**
     * As json string.
     *
     * @return the json string value
     */
    public String asJson() {
        return JsonValueJNI.asJson(parserHandle, item);
    }

    /**
     * Get json value.
     *
     * @param key the key
     * @return the json value
     */
    public JsonValue get(String key) {
        return JsonValueJNI.getValue(parserHandle,item,0, key);
    }

    /**
     * Get json value.
     *
     * @param index the index
     * @return the json value
     */
    public JsonValue get(int index) {
        return JsonValueJNI.getValue(parserHandle,item,index, null);
    }

    /**
     * Get the name of the value.
     * @param index The index to get the name of.
     * @return String representing the name of the value if any.
     */
    public String getName(int index) {
        return JsonValueJNI.getName(parserHandle, item, index, null);
    }

    /**
     * Has value.
     *
     * @param index the index
     * @return the boolean value
     */
    public boolean hasValue(int index) {
        return JsonValueJNI.hasValue(parserHandle,item,index,null);
    }

    /**
     * Has value.
     *
     * @param key the key
     * @return the boolean value
     */
    public boolean hasValue(String key) {
        return JsonValueJNI.hasValue(parserHandle,item,0,key);
    }

    /**
     * Count int.
     *
     * @return the int
     */
    public int count() {
        return JsonValueJNI.count(parserHandle, item);
    }

    /***
     * Close
     * @throws Exception will throw if private handle.close fails.
     */
    @Override
    public void close() throws Exception {
        if(parserHandle != null){
            parserHandle.close();
            parserHandle = null;
        }
    }

    private int item;
    private SafeHandle parserHandle;
}

/*! \endcond */
