//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;

import com.microsoft.cognitiveservices.speech.SpeechConfig;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.TelemetryManager;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

/**
 * Class that defines hybrid (cloud and embedded) configurations for speech recognition and speech synthesis.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class HybridSpeechConfig implements AutoCloseable {

    static {
        // Trigger loading of the native libraries.
        try {
            Class.forName(SpeechConfig.class.getName());
        }
        catch (ClassNotFoundException ex) {
            throw new IllegalStateException(ex);
        }
    }

    /**
     * Creates an instance of hybrid speech config.
     */
    HybridSpeechConfig(long handleValue) {
        this.config = new SpeechConfig(handleValue);
        TelemetryManager.getSingleton();
    }

    /**
     * Creates an instance of the hybrid speech config with specified cloud and embedded speech configs.
     * @param cloudSpeechConfig Cloud speech configuration.
     * @param embeddedSpeechConfig Embedded speech configuration.
     * @return A hybrid speech config instance.
     */
    public final static HybridSpeechConfig fromConfigs(SpeechConfig cloudSpeechConfig, EmbeddedSpeechConfig embeddedSpeechConfig) {
        Contracts.throwIfNull(cloudSpeechConfig, "cloudSpeechConfig");
        Contracts.throwIfNull(embeddedSpeechConfig, "embeddedSpeechConfig");
        IntRef configRef = new IntRef(0);
        Contracts.throwIfFail(createHybridSpeechConfig(configRef, cloudSpeechConfig.getImpl(), embeddedSpeechConfig.config.getImpl()));
        HybridSpeechConfig hybridSpeechConfig = new HybridSpeechConfig(configRef.getValue());
        return hybridSpeechConfig;
    }

    /**
     * Sets the speech recognition output format.
     * @param value The recognition output format.
     */
    public final void setSpeechRecognitionOutputFormat(OutputFormat value) {
        config.setOutputFormat(value);
    }

    /**
     * Gets the speech recognition output format.
     * @return The recognition output format.
     */
    public final OutputFormat getSpeechRecognitionOutputFormat() {
        return config.getOutputFormat();
    }

    /**
     * Sets the speech synthesis output format.
     * @param value The synthesis output format ID (e.g. Riff16Khz16BitMonoPcm).
     */
    public final void setSpeechSynthesisOutputFormat(SpeechSynthesisOutputFormat value) {
        config.setSpeechSynthesisOutputFormat(value);
    }

    /**
     * Gets the speech synthesis output format.
     * @return Returns the synthesis output format.
     */
    public final String getSpeechSynthesisOutputFormat() {
        return config.getSpeechSynthesisOutputFormat();
    }

    /**
     * Sets a named property as value.
     * @param name the name of the property.
     * @param value the value.
     */
    public void setProperty(String name, String value) {
        config.setProperty(name, value);
    }

    /**
     * Sets the property by propertyId.
     * @param id PropertyId of the property.
     * @param value The value.
     */
    public void setProperty(PropertyId id, String value) {
        config.setProperty(id, value);
    }

    /**
     * Gets a named property as value.
     * @param name the name of the property.
     * @return The value.
     */
    public String getProperty(String name) {
        return config.getProperty(name);
    }

    /**
     * Gets the property by propertyId.
     * @param id PropertyId of the property.
     * @return The value.
     */
    public String getProperty(PropertyId id) {
        return config.getProperty(id);
    }

    /**
     * Dispose of associated resources.
     */
    @Override
    public final void close() {
        if (disposed) {
            return;
        }

        if (config != null)
        {
            config.close();
            config = null;
        }

        disposed = true;
    }

    /*! \cond INTERNAL */
    /**
     * Returns a internal handle to SpeechConfig implementation.
     * @return The implementation handle.
     */
    public SafeHandle getImpl() {
        Contracts.throwIfNull(config, "config");
        return config.getImpl();
    }
    /*! \endcond */

    SpeechConfig config = null;

    private final static native long createHybridSpeechConfig(IntRef configHandle, SafeHandle cloudSpeechConfigHandle, SafeHandle embeddedSpeechConfigHandle);

    private boolean disposed = false;
}
