//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.PropertyCollection;

/**
 * Contains detailed information about the retrieved synthesis voices list.
 * Note: close() must be called in order to release underlying resources held by the object.
 * Added in version 1.16.0
 */
public class SynthesisVoicesResult implements AutoCloseable {

    /*! \cond PROTECTED */
    /**
     * PROTECTED
     * @param result PROTECTED
     */
    protected SynthesisVoicesResult(IntRef result) {
        Contracts.throwIfNull(result, "result");

        this.resultHandle = new SafeHandle(result.getValue(), SafeHandleType.SynthesisResult);

        IntRef intRef = new IntRef(0);
        Contracts.throwIfFail(getVoiceNumber(resultHandle, intRef));
        int voiceNumber = (int)intRef.getValue();

        List<VoiceInfo> voiceList = new ArrayList<VoiceInfo>();
        for (int i = 0; i < voiceNumber; i++) {
            IntRef voiceInfoRef = new IntRef(0);
            Contracts.throwIfFail(getVoiceInfo(resultHandle, i, voiceInfoRef));
            voiceList.add(new VoiceInfo(voiceInfoRef));
        }
        this.voices = Collections.unmodifiableList(voiceList);

        StringRef stringRef = new StringRef("");
        Contracts.throwIfFail(getResultId(resultHandle, stringRef));
        this.resultId = stringRef.getValue();

        Contracts.throwIfFail(getResultReason(resultHandle, intRef));
        this.reason = ResultReason.values()[(int)intRef.getValue()];

        IntRef propertyRef = new IntRef(0);
        Contracts.throwIfFail(getPropertyBagFromResult(resultHandle, propertyRef));
        this.properties = new PropertyCollection(propertyRef);

        this.errorDetails = properties.getProperty(PropertyId.CancellationDetails_ReasonDetailedText);
    }

    /*! \endcond */

    /**
     * Gets the retrieved voices list
     * @return The voices list
     */
    public List<VoiceInfo> getVoices() {
        return this.voices;
    }

    /**
     * Gets the unique ID of the result.
     * @return The unique ID of the result.
     */
    public String getResultId() {
        return this.resultId;
    }

    /**
     * Gets the reason the result was created.
     * @return The reason of the result.
     */
    public ResultReason getReason() {
        return this.reason;
    }

    /**
     * Gets the error details of the result.
     * @return The error details.
     */
    public String getErrorDetails() {
        return this.errorDetails;
    }

    /**
     *  The set of properties exposed in the result.
     * @return The set of properties exposed in the result.
     */
    public PropertyCollection getProperties() {
        return this.properties;
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    public void close() {
        if (this.resultHandle != null) {
            this.resultHandle.close();
            this.resultHandle = null;
        }

        if (this.properties != null) {
            this.properties.close();
            this.properties = null;
        }

        if (this.voices != null) {
            for (VoiceInfo v: this.voices) {
                v.close();
            }
            this.voices = null;
        }
    }

    /*! \cond INTERNAL */

    /**
     * Returns the synthesis result implementation.
     * @return The implementation of the result.
     */
    public SafeHandle getImpl() {
        return this.resultHandle;
    }

    /*! \endcond */

    private final native long getResultId(SafeHandle resultHandle, StringRef propertyRef);
    private final native long getResultReason(SafeHandle resultHandle, IntRef propertyRef);
    private final native long getPropertyBagFromResult(SafeHandle resultHandle, IntRef propertyRef);
    private final native long getVoiceNumber(SafeHandle resultHandle, IntRef voiceNumber);
    private final native long getVoiceInfo(SafeHandle synthHandle, int index, IntRef resultRef);

    private String resultId;
    private ResultReason reason;
    private String errorDetails;
    private List<VoiceInfo> voices;
    private PropertyCollection properties;
    private SafeHandle resultHandle = null;
}
