//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.transcription;

import java.lang.AutoCloseable;
import java.util.concurrent.Future;
import com.microsoft.cognitiveservices.speech.SpeechConfig;
import com.microsoft.cognitiveservices.speech.PropertyId;
import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.AsyncThreadService;
import com.microsoft.cognitiveservices.speech.PropertyCollection;
import com.microsoft.cognitiveservices.speech.transcription.Participant;
import com.microsoft.cognitiveservices.speech.transcription.User;

/**
 * Performs meeting management including add and remove participants.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class Meeting implements AutoCloseable
{
    /**
     * Initializes a new instance of Meeting.
     * @param speechConfig speech configuration.
     * @return A task representing the asynchronous operation that creates a meeting.
     */
    public static Future<Meeting> createMeetingAsync(SpeechConfig speechConfig) {
        Contracts.throwIfNull(speechConfig, "speechConfig");
        final String finalMeetingId = "";
        final SpeechConfig finalSpeechConfig = speechConfig;

        AsyncThreadService.initialize();
        return AsyncThreadService.submit(new java.util.concurrent.Callable<Meeting>() {
            public Meeting call() {
                IntRef handleRef = new IntRef(0);
                Contracts.throwIfFail(createMeetingFromConfig(handleRef, finalSpeechConfig.getImpl(), finalMeetingId));
                return new Meeting(handleRef.getValue());
            }
        });
    }
    
    /**
     * Initializes a new instance of Meeting.
     * @param speechConfig speech configuration.
     * @param meetingId a unqiue identification of your meeting.
     * @return A task representing the asynchronous operation that creates a meeting.
     */
    public static Future<Meeting> createMeetingAsync(SpeechConfig speechConfig, String meetingId) {
        Contracts.throwIfNull(speechConfig, "speechConfig");
        Contracts.throwIfNull(meetingId, "meetingId");
        
        final String finalMeetingId = meetingId;
        final SpeechConfig finalSpeechConfig = speechConfig;

        AsyncThreadService.initialize();
        return AsyncThreadService.submit(new java.util.concurrent.Callable<Meeting>() {
            public Meeting call() {
                IntRef handleRef = new IntRef(0);
                Contracts.throwIfFail(createMeetingFromConfig(handleRef, finalSpeechConfig.getImpl(), finalMeetingId));
                return new Meeting(handleRef.getValue());
            }
        });
    }

    /**
     * Dispose of associated resources.
     * Note: close() must be called in order to release underlying resources held by the object.
     */
    @Override
    public void close() {
        synchronized (meetingLock) {
            if (activeAsyncMeetingCounter != 0) {
                throw new IllegalStateException("Cannot dispose a recognizer while async recognition is running. Await async recognitions to avoid unexpected disposals.");
            }
            dispose(true);
        }
    }

    /**
     * Gets the meeting Id.
     * @return the meeting Id.
     */
    public String getMeetingId() {
        StringRef meetingIdRef = new StringRef("");
        Contracts.throwIfFail(getMeetingId(meetingHandle, meetingIdRef));
        return meetingIdRef.getValue();
    }

    /**
     * Sets the authorization token used to communicate with the service.
     * Note: The caller needs to ensure that the authorization token is valid. Before the authorization token expires,
     * the caller needs to refresh it by calling this setter with a new valid token.
     * Otherwise, the recognizer will encounter errors during recognition.
     * @param token Authorization token.
     */
    public void setAuthorizationToken(String token) {
        Contracts.throwIfNullOrWhitespace(token, "token");
        propertyHandle.setProperty(PropertyId.SpeechServiceAuthorization_Token, token);
    }

    /**
     * Gets the authorization token used to communicate with the service.
     * @return Authorization token.
     */
    public String getAuthorizationToken() {
        return propertyHandle.getProperty(PropertyId.SpeechServiceAuthorization_Token);
    }

    /**
     * The collection of properties and their values defined for this meeting.
     * @return The collection of properties and their values defined for this meeting.
     */
    public PropertyCollection getProperties() {
        return this.propertyHandle;
    }

    /**
     *   Add a participant to a meeting using a participant object.
     *
     * @param participant A participant object.
     * @return A task representing the asynchronous operation of adding a participant.
     */
    public Future<Participant> addParticipantAsync(Participant participant) {
        final Participant finalParticipant = participant;
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Participant>() {
            public Participant call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(addParticipant(meetingHandle, finalParticipant.getImpl())); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return finalParticipant;
        }});
    }

    /**
     *   Add a participant to a meeting using the user's id.
     *
     * @param userId The user id.
     * @return A task representing the asynchronous operation of adding a participant.
     */
    public Future<Participant> addParticipantAsync(final String userId) {
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Participant>() {
            public Participant call() {
                final Participant[] result = new Participant[1];
                Runnable runnable = new Runnable() { public void run() {
                Participant participant = Participant.from(userId);
                Contracts.throwIfFail(addParticipant(meetingHandle, participant.getImpl()));
                result[0] = participant;
                }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return result[0];
        }});
    }

    /**
     *   Add a participant to a meeting using a user object.
     *
     * @param user A user object.
     * @return A task representing the asynchronous operation of adding a participant.
     */
    public Future<User> addParticipantAsync(final User user) {
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<User>() {
            public User call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(addParticipantByUser(meetingHandle, user.getImpl())); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return user;
        }});
    }

    /**
     *   Remove a participant from a meeting using a user object.
     *
     * @param user A user object.
     * @return Am empty task representing the asynchronous operation of removing a participant.
     */
    public Future<Void> removeParticipantAsync(User user) {
        final User finalUser = user;
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(removeParticipantByUser(meetingHandle, finalUser.getImpl())); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /**
     *   Remove a participant from a meeting using a participant object.
     *
     * @param participant A participant object.
     * @return Am empty task representing the asynchronous operation of removing a participant.
     */
    public Future<Void> removeParticipantAsync(Participant participant) {
        final Participant finalParticipant = participant;
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(removeParticipant(meetingHandle, finalParticipant.getImpl())); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /**
     *   Remove a participant from a meeting using a user id.
     *
     * @param userId A user id.
     * @return Am empty task representing the asynchronous operation of removing a participant.
     */
    public Future<Void> removeParticipantAsync(String userId) {
        final String finalUserId = userId;
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(removeParticipantByUserId(meetingHandle, finalUserId)); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /**
     *   End a meeting.
     *
     * @return An empty task representing the asynchronous operation that ending a meeting.
     */
    public Future<Void> endMeetingAsync() {
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(endMeeting(meetingHandle)); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /**
     * Start a meeting.
     *
     * @return An asynchronous operation representing starting a meeting.
     */
    public Future<Void> startMeetingAsync() {
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(startMeeting(meetingHandle)); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /**
     * Delete a meeting. After this no one will be able to join the meeting.
     *
     * @return An asynchronous operation representing deleting a meeting.
     */
    public Future<Void> deleteMeetingAsync() {
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(deleteMeeting(meetingHandle)); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /**
     * Lock a meeting. This will prevent new participants from joining.
     *
     * @return An asynchronous operation representing locking a meeting.
     */
    public Future<Void> lockMeetingAsync() {
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(lockMeeting(meetingHandle)); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /**
     * Unlocks a meeting.
     *
     * @return An asynchronous operation representing unlocking a meeting.
     */
    public Future<Void> unlockMeetingAsync() {
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(unlockMeeting(meetingHandle)); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /**
     * Mute all other participants in the meeting. After this no other participants will have their speech recognitions broadcast, nor be able to send text messages.
     *
     * @return An asynchronous operation representing muting all participants.
     */
    public Future<Void> muteAllParticipantsAsync() {
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(muteAll(meetingHandle)); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /**
     * Unmute all other participants in the meeting.
     *
     * @return An asynchronous operation representing un-muting all participants.
     */
    public Future<Void> unmuteAllParticipantsAsync() {
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(unmuteAll(meetingHandle)); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /**
     * Mute a participant.
     *
     * @param userId A user identifier.
     * @return An asynchronous operation representing muting a particular participant.
     */
    public Future<Void> muteParticipantAsync(String userId) {
        final String finalUserId = userId;
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(muteParticipant(meetingHandle, finalUserId)); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /**
     * Unmute a participant.
     *
     * @param userId A user identifier.
     * @return An asynchronous operation representing un-muting a particular participant.
     */
    public Future<Void> unmuteParticipantAsync(String userId) {
        final String finalUserId = userId;
        final Meeting thisMeeting = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { Contracts.throwIfFail(unmuteParticipant(meetingHandle, finalUserId)); }};
                thisMeeting.doAsyncMeetingAction(runnable);
                return null;
        }});
    }

    /*! \cond INTERNAL */

    /**
     * Returns the comversation implementation.
     * @return The implementation of the meeting.
     */
    public SafeHandle getImpl() {
        return meetingHandle;
    }
    
    /*! \endcond */

    /*! \cond PROTECTED */
    /**
     * PROTECTED
     * @param disposing PROTECTED
     */

    protected void dispose(boolean disposing)
    {
        if (disposed) {
            return;
        }

        if (disposing) {

            if (meetingHandle != null) {
                meetingHandle.close();
                meetingHandle = null;
            }

            if (propertyHandle != null) {
                propertyHandle.close();
                propertyHandle = null;
            }

            // Shutdown of async thread service
            AsyncThreadService.shutdown();
            disposed = true;
        }
    }

    /**
    * Protected constructor.
    * @param handleValue Internal meeting implementation
    */
    protected Meeting(long handleValue) {
        this.meetingHandle = new SafeHandle(handleValue, SafeHandleType.Meeting);
        IntRef propbagRef = new IntRef(0);
        Contracts.throwIfFail(getPropertyBag(meetingHandle, propbagRef));
        propertyHandle = new PropertyCollection(propbagRef);
    }

    /*! \endcond */

    private void doAsyncMeetingAction(Runnable meetingImplAction) {
        synchronized (meetingLock) {
            activeAsyncMeetingCounter++;
        }
        if (disposed) {
            throw new IllegalStateException(this.getClass().getName());
        }
        try {
            meetingImplAction.run();
        } finally {
            synchronized (meetingLock) {
                activeAsyncMeetingCounter--;
            }
        }
    }

    private SafeHandle meetingHandle;
    private boolean disposed = false;
    private final Object meetingLock = new Object();
    private int activeAsyncMeetingCounter = 0;
    private com.microsoft.cognitiveservices.speech.PropertyCollection propertyHandle;

    private final static native long createMeetingFromConfig(IntRef meetingHandle, SafeHandle speechConfigHandle, String id);
    private final native long getMeetingId(SafeHandle meetingHandle, StringRef meetingIdRef);
    private final native long addParticipant(SafeHandle meetingHandle, SafeHandle participantHandle);
    private final native long addParticipantByUser(SafeHandle meetingHandle, SafeHandle userHandle);
    private final native long removeParticipant(SafeHandle meetingHandle, SafeHandle participantHandle);
    private final native long removeParticipantByUser(SafeHandle meetingHandle, SafeHandle userHandle);
    private final native long removeParticipantByUserId(SafeHandle meetingHandle, String userId);
    private final native long startMeeting(SafeHandle meetingHandle);
    private final native long endMeeting(SafeHandle meetingHandle);
    private final native long deleteMeeting(SafeHandle meetingHandle);
    private final native long lockMeeting(SafeHandle meetingHandle);
    private final native long unlockMeeting(SafeHandle meetingHandle);
    private final native long muteParticipant(SafeHandle meetingHandle, String userId);
    private final native long muteAll(SafeHandle meetingHandle);
    private final native long unmuteParticipant(SafeHandle meetingHandle, String userId);
    private final native long unmuteAll(SafeHandle meetingHandle);
    private final native long getPropertyBag(SafeHandle meetingHandle, IntRef propbagHandle);
}
