//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.transcription;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.SpeechRecognitionResult;
import com.microsoft.cognitiveservices.speech.util.StringRef;

/**
 * Class that defines meeting transcription result.
 */
public class MeetingTranscriptionResult extends SpeechRecognitionResult {

    /**
     * Internal constructor for creating a MeetingTranscriptionResult from a native handle.
     * 
     * @param result The native handle.
     */
    protected MeetingTranscriptionResult(long result) {
        super(result);

        if (result != 0) {
            StringRef userIdStr = new StringRef("");
            Contracts.throwIfFail(getUserId(super.getImpl(), userIdStr));
            this.userId = userIdStr.getValue();

            StringRef utteranceIdStr = new StringRef("");
            Contracts.throwIfFail(getUtteranceId(super.getImpl(), utteranceIdStr));
            this.utteranceId = utteranceIdStr.getValue();
        }
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    @Override
    public void close() {
        super.close();
    }

    /**
     * A String that represents the user id in the meeting.
     * @return the user ID string.
     */
    public String getUserId() {
        return this.userId;
    }

    /**
     * A string that represents the utterance. This id is consistence for intermediates and final speech recognition result from one speaker.
     * @return the utterance ID string.
     */
    public String getUtteranceId() {
        return this.utteranceId;
    }

    /**
     * Returns a String that represents the meeting transcription result.
     * @return A String that represents the meeting transcription result.
     */
    @Override
    public String toString() {
        return "ResultId:" + this.getResultId()+
                " Status:" + this.getReason() +
                " UserId:" + this.userId +
                " UtteranceId:" + this.utteranceId +
                " Recognized text:<" + this.getText() + ">.";
    }

    private final native long getUserId(SafeHandle resultHandle, StringRef userId);
    private final native long getUtteranceId(SafeHandle resultHandle, StringRef utteranceId);

    private String userId;
    private String utteranceId;
}
