//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//

package com.microsoft.cognitiveservices.speech;

import com.microsoft.cognitiveservices.speech.util.Contracts;

/**
 * Native logging and other diagnostics
 */
public final class Diagnostics {
    /**
     * Starts console (for Android, logcat) logging
     * @param logToStderr Whether the log should be emitted to standard error or standard out; has no effect on Android.
     */
    public static void startConsoleLogging(boolean logToStderr) {
        Contracts.throwIfFail(logConsoleStartLogging(logToStderr));
    }

    /**
     * Starts console (for Android, logcat) logging
     */
    public static void startConsoleLogging() {
        startConsoleLogging(true);
    }

    /**
     * Stops console logging
     */
    public static void stopConsoleLogging() {
        Contracts.throwIfFail(logConsoleStopLogging());
    }

    /**
     * Enables the native memory logger
     */
    public static void startMemoryLogging() {
        Contracts.throwIfFail(logMemoryStartLogging());
    }

    /**
     * Stops the native memory logger
     */
    public static void stopMemoryLogging() {
        Contracts.throwIfFail(logMemoryStopLogging());
    }

    /**
     * Dumps the contents of the memory logger
     * @param filename The name of the file to write to. Set this to an empty string if not needed
     * @param linePrefix The prefix to apply to each line of logged memory content, e.g. "CRBN"
     * @param emitToStdOut Whether the log should be emitted to standard output (for Android, logcat) in addition to any other targets
     * @param emitToStdErr Whether the log should be emitted to standard error (for Android, logcat) in addition to any other targets
     */
    public static void dumpMemoryLog(String filename, String linePrefix, boolean emitToStdOut, boolean emitToStdErr) {
        Contracts.throwIfFail(logMemoryDump(filename, linePrefix, emitToStdOut, emitToStdErr));
    }

    /**
     * Dumps the contents of the memory logger on exit
     * @param filename The name of the file to write to. Set this to an empty string if not needed
     * @param linePrefix The prefix to apply to each line of logged memory content, e.g. "CRBN"
     * @param emitToStdOut Whether the log should be emitted to standard output (for Android, logcat) in addition to any other targets
     * @param emitToStdErr Whether the log should be emitted to standard error (for Android, logcat) in addition to any other targets
     */
    public static void dumpMemoryLogOnExit(String filename, String linePrefix, boolean emitToStdOut, boolean emitToStdErr) {
        Contracts.throwIfFail(logMemoryDumpOnExit(filename, linePrefix, emitToStdOut, emitToStdErr));
    }

    /**
     * Enables the native file logger
     * @param filename The name of resulting log file
     * @param append If true will append to file (if already exists). Otherwise will create a new file
     */
    public static void startFileLogging(String filename, boolean append) {
        Contracts.throwIfFail(logFileStartLogging(filename, append));
    }

    /**
     * Stops the native file logger
     */
    public static void stopFileLogging() {
        Contracts.throwIfFail(logFileStopLogging());
    }

    private final native static long logConsoleStartLogging(boolean logToStderr);
    private final native static long logConsoleStopLogging();
    private final native static long logMemoryStartLogging();
    private final native static long logMemoryStopLogging();
    private final native static long logMemoryDump(String filename, String linePrefix, boolean emitToStdOut, boolean emitToStdErr);
    private final native static long logMemoryDumpOnExit(String filename, String linePrefix, boolean emitToStdOut, boolean emitToStdErr);
    private final native static long logFileStartLogging(String filename, boolean append);
    private final native static long logFileStopLogging();
}
