//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.StringRef;

/**
 * Defines contents of speech synthesis viseme event.
 * Added in version 1.16.0
 */
public class SpeechSynthesisVisemeEventArgs {

    /*! \cond INTERNAL */

    SpeechSynthesisVisemeEventArgs(long eventArgs) {
        Contracts.throwIfNull(eventArgs, "eventArgs");

        SafeHandle eventHandle = new SafeHandle(eventArgs, SafeHandleType.SynthesisEvent);

        StringRef stringRef = new StringRef("");
        Contracts.throwIfFail(getResultId(eventHandle, stringRef));
        this.resultId = stringRef.getValue();

        IntRef audioOffsetRef = new IntRef(0);
        IntRef visemeIdRef = new IntRef(0);
        Contracts.throwIfFail(getVisemeEventValues(eventHandle, audioOffsetRef, visemeIdRef));
        this.audioOffset = audioOffsetRef.getValue();
        this.visemeId = visemeIdRef.getValue();
        Contracts.throwIfFail(getAnimationFromHandle(eventHandle, stringRef));
        this.animation = stringRef.getValue();
        eventHandle.close();
    }

    /*! \endcond */

    /**
     * Gets the unique ID of the result.
     * Added in version 1.25.0
     * @return Specifies unique ID of the result.
     */
    public String getResultId() {
        return this.resultId;
    }

    /**
     * Specifies current viseme's time offset in output audio, by ticks (100ns).
     * @return Current viseme's time offset in output audio, by ticks (100ns).
     */
    public long getAudioOffset() {
        return this.audioOffset;
    }

    /**
     * Specifies current viseme ID.
     * @return The current viseme ID.
     */
    public long getVisemeId() {
        return this.visemeId;
    }

    /**
     * Specifies the animation, could be in svg or other format.
     * @return The animation string.
     */
    public String getAnimation() {
        return this.animation;
    }

    private final native long getResultId(SafeHandle eventHandle, StringRef resultIdRef);
    private final native long getVisemeEventValues(SafeHandle eventHandle, IntRef audioOffset, IntRef visemeId);
    private final native long getAnimationFromHandle(SafeHandle eventHandle, StringRef animationText);

    private String resultId;
    private long audioOffset;
    private long visemeId;
    private String animation;

}
