//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.intent;

import java.util.ArrayList;
import java.util.Collection;
import com.microsoft.cognitiveservices.speech.util.KeyedItem;

/**
*  Represents a pattern matching entity used for intent recognition.
*/
public class PatternMatchingEntity implements KeyedItem {
    /**
    * An Id used to define this entity if it is matched. This id must appear in
    * an intent phrase or the entity will never be matched.
    */
    private String entityId;
    
    /**
    * The EntityMatchMode of this Entity.
    */
    private EntityMatchMode mode;

    /**
    * The EntityType of this Entity.
    */
    private EntityType type;

    /**
    * Returns the ID for this item. This will be used to retrieve the entity
    * if it is included in the IntentRecognitionResult.
    *  @return Returns the ID for this item.
    */
    @Override
    public String getId() {
        return entityId;
    }

    /**
    * Sets the ID for this item. This will be used to retrieve the entity
    * if it is included in the IntentRecognitionResult.
    *  @param id The ID for this item.
    */
    public void setId(String id) {
        entityId = id;
    }

    /**
    * Returns the {@code EntityMatchMode} for this item.
    *  @return Returns the {@code EntityMatchMode} for this item.
    */
    public EntityMatchMode getMatchMode() {
        return mode;
    }

    /**
    * Sets the {@code EntityMatchMode} for this item.
    *  @param mode The {@code EntityMatchMode} for this item.
    */
    public void setMatchMode(EntityMatchMode mode) {
        this.mode = mode;
    }

    /**
    * Returns the {@code EntityType} for this item.
    *  @return Returns the {@code EntityType} for this item.
    */
    public EntityType getType() {
        return type;
    }

    /**
    * Sets the {@code EntityType} for this item.
    *  @param type The {@code EntityType} for this item.
    */
    public void setType(EntityType type) {
        this.type = type;
    }

    /**
    * A collection of strings used to match the entity for List type entities.
    * Strict Mode means the entity must appear in the list.
    */
    public Collection<String> Phrases;

    /**
    * Internal constructor that sets the properties with the input parameters.
    *
    * @param entityId A string that represents a unique Id for this entity.
    * @param type The EntityType.
    * @param mode The EntityMatchMode.
    * @param phrases Collection of phrases.
    */
    PatternMatchingEntity(
        String entityId, 
        EntityType type, 
        EntityMatchMode mode, 
        Collection<String> phrases) {
            this.entityId = entityId;
            this.type = type;
            this.mode = mode;
            if(phrases == null){
                Phrases = new ArrayList<>();
            }
            else{
                Phrases = phrases;
            }
    }

    /**
    * Creates a pattern matching entity using the specified intent ID.
    * @param entityId A String that represents a unique Id for this entity.
    * @return The Pattern Matching Entity being created.
    */
    public static PatternMatchingEntity CreateAnyEntity(String entityId){
         return new PatternMatchingEntity(entityId, EntityType.Any, EntityMatchMode.Basic, null);
    }

    /**
    * Creates a pattern matching entity using the specified intent ID. The
    * PrebuiltInteger Entity will match words representing numbers in lexical,
    * digit, and ordinal formats.
    * @param entityId A String that represents a unique Id for this entity.
    * @return The Pattern Matching Entity being created.
    */
    public static PatternMatchingEntity CreateIntegerEntity(String entityId){
        return new PatternMatchingEntity(entityId, EntityType.PrebuiltInteger, EntityMatchMode.Basic, null);
    }

    /**
    * Creates a pattern matching entity using the specified intent ID,
    * EntityMatchMode, phrases. This entity type will match based on the 
    * phrases provided.
    * @param entityId A string that represents a unique Id for this entity.
    * @param mode The {@code EntityMatchMode} for the List entity. Strict means the
    * captured entity must appear in the phrases list.
    * @param phrases A list of phrases used to match the list entity.
    * @return The Pattern Matching Entity being created.
    */
    public static PatternMatchingEntity CreateListEntity(String entityId, EntityMatchMode mode, Collection<String> phrases) {
        return new PatternMatchingEntity(entityId, EntityType.List, mode, phrases);
    }

    /**
    * Creates a pattern matching entity using the specified intent ID,
    * EntityMatchMode, phrases. This entity type will match based on the 
    * phrases provided.
    * @param entityId A string that represents a unique Id for this entity.
    * @param mode The {@code EntityMatchMode} for the List entity. Strict means the
    * captured entity must appear in the phrases list.
    * @param phrases A list of phrases used to match the list entity.
    * @return The Pattern Matching Entity being created.
    */
    public static PatternMatchingEntity CreateListEntity(String entityId, EntityMatchMode mode, String... phrases) {
        Collection<String> list = new ArrayList<String>();
        for(String phrase : phrases)
        {
            list.add(phrase);
        }
        return new PatternMatchingEntity(entityId, EntityType.List, mode, list);
    }

    /**
    * Used to define the match mode of an entity used for intent recognition. 
    * Currently this only affects List entities as all other entities only have one mode.
    */
    public enum EntityMatchMode
    {
        /**
        * This is the basic or default mode of matching based on the EntityType.
        */
        Basic(0),
        /**
        * This will match only exact matches within the entities phrases.
        */
        Strict(1),
        /**
        * This will match text within the slot the entity is in, but not 
        * require anything from that text.
        */
        Fuzzy(2);

        private EntityMatchMode (int id) {
            this.id = id;
        }
    
        /**
         * Returns the internal value property id
         *
         * @return the speech property id
         */
        public int getValue() { return this.id; }
    
        private final int id;
    };

    /**
    * Used to define the type of entity used for intent recognition.
    */
    public enum EntityType
    {
        /**
        * This will match any text that fills the slot.
        */
        Any(0),
        /**
        * This will match text that is contained within the list or any text if
        *  the mode is set to EntityMatchMode.Fuzzy.
        */
        List(1),
        /**
        * This will match cardinal and ordinal integers.
        */
        PrebuiltInteger(2);

        private EntityType(int id) {
            this.id = id;
        }
    
        /**
         * Returns the internal value property id
         *
         * @return the speech property id
         */
        public int getValue() { return this.id; }
    
        private final int id;
    };

}
