//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.transcription;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.Future;

import com.microsoft.cognitiveservices.speech.audio.AudioConfig;
import com.microsoft.cognitiveservices.speech.SpeechConfig;
import com.microsoft.cognitiveservices.speech.transcription.MeetingTranscriptionCanceledEventArgs;
import com.microsoft.cognitiveservices.speech.util.EventHandlerImpl;
import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.AsyncThreadService;
import com.microsoft.cognitiveservices.speech.PropertyId;
import com.microsoft.cognitiveservices.speech.OutputFormat;
import com.microsoft.cognitiveservices.speech.PropertyCollection;

/**
 * Performs meeting transcribing for audio input streams, and gets transcribed text and user id as a result.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class MeetingTranscriber extends com.microsoft.cognitiveservices.speech.Recognizer
{
    /**
     * The event transcribing signals that an intermediate recognition result is received.
     */
    final public EventHandlerImpl<MeetingTranscriptionEventArgs> transcribing = new EventHandlerImpl<MeetingTranscriptionEventArgs>(eventCounter);

    /**
     * The event transcribed signals that a final recognition result is received.
     */
    final public EventHandlerImpl<MeetingTranscriptionEventArgs> transcribed = new EventHandlerImpl<MeetingTranscriptionEventArgs>(eventCounter);

    /**
     * The event canceled signals that the recognition was canceled.
     */
    final public EventHandlerImpl<MeetingTranscriptionCanceledEventArgs> canceled = new EventHandlerImpl<MeetingTranscriptionCanceledEventArgs>(eventCounter);

    /**
     * Initializes a new instance of Meeting Transcriber.
     */
    public MeetingTranscriber() {
        super(null);

        Contracts.throwIfNull(super.getImpl(), "recoHandle");
        Contracts.throwIfFail(createMeetingTranscriberFromConfig(super.getImpl(), null));
        initialize();
    }

    /**
     * Initializes a new instance of Meeting Transcriber.
     * @param audioConfig audio configuration.
     */
    public MeetingTranscriber(AudioConfig audioConfig) {
        super(audioConfig);

        if (audioConfig == null) {
            Contracts.throwIfFail(createMeetingTranscriberFromConfig(super.getImpl(), null));
        } else {
            Contracts.throwIfFail(createMeetingTranscriberFromConfig(super.getImpl(), audioConfig.getImpl()));
        }
        initialize();
    }

    /**
     * Gets the spoken language of recognition.
     * @return The spoken language of recognition.
     */
    public String getSpeechRecognitionLanguage() {
        return propertyHandle.getProperty(PropertyId.SpeechServiceConnection_RecoLanguage);
    }

    /**
     * Gets the output format of recognition.
     * @return The output format of recognition.
     */
    public OutputFormat getOutputFormat() {
        if (propertyHandle.getProperty(PropertyId.SpeechServiceResponse_RequestDetailedResultTrueFalse).equals("true")) {
            return OutputFormat.Detailed;
        } else {
            return OutputFormat.Simple;
        }
    }

    /**
     * The collection of properties and their values defined for this MeetingTranscriber.
     * @return The collection of properties and their values defined for this MeetingTranscriber.
     */
    public PropertyCollection getProperties() {
        return propertyHandle;
    }


     /**
     *  Join a meeting.
     *  A meeting transcriber must join a meeting before transcribing audio.
     * @param meeting the meeting to be joined.
     * @return A task representing the asynchronous operation that joins a meeting.
     */
    public Future<Void> joinMeetingAsync(Meeting meeting) {
        final MeetingTranscriber thisReco = this;
        final Meeting thisMeeting = meeting;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { thisReco.joinMeeting(getImpl(), thisMeeting.getImpl()); }};
                thisReco.doAsyncRecognitionAction(runnable);
                return null;
        }});
    }

     /**
     *  Leave a meeting.
     *  After leaving a meeting, no transcribing and transcribed events will be sent out.
     *
     * @return A task representing the asynchronous operation that leaves a meeting.
     */
    public Future<Void> leaveMeetingAsync() {
        final MeetingTranscriber thisReco = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { thisReco.leaveMeeting(getImpl()); }};
                thisReco.doAsyncRecognitionAction(runnable);
                return null;
        }});
    }

    /**
     * Starts meeting transcribing on a continuous audio stream, until stopTranscribingAsync() is called.
     * User must subscribe to events to receive transcription results.
     * @return A task representing the asynchronous operation that starts the transcription.
     */
    public Future<Void> startTranscribingAsync() {
        final MeetingTranscriber thisReco = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { thisReco.startContinuousRecognition(getImpl()); }};
                thisReco.doAsyncRecognitionAction(runnable);
                return null;
        }});
    }

    /**
     *  Stops meeting transcribing.
     *
     *  @return A task representing the asynchronous operation that stops the transcription.
     */
    public Future<Void> stopTranscribingAsync() {
        final MeetingTranscriber thisReco = this;

        return AsyncThreadService.submit(new java.util.concurrent.Callable<Void>() {
            public Void call() {
                Runnable runnable = new Runnable() { public void run() { thisReco.stopContinuousRecognition(getImpl()); }};
                thisReco.doAsyncRecognitionAction(runnable);
                return null;
        }});
    }

    @Override
    protected void dispose(final boolean disposing) {
        if (disposed) {
            return;
        }

        if (disposing) {
                        
            if (propertyHandle != null) {
                propertyHandle.close();
                propertyHandle = null;
            }
            meetingTranscriberObjects.remove(this);
            super.dispose(disposing);
        }
    }

    /**
     * This is used to keep any instance of this class alive that is subscribed to downstream events.
     */
    static java.util.Set<MeetingTranscriber> meetingTranscriberObjects = java.util.Collections.synchronizedSet(new java.util.HashSet<MeetingTranscriber>());

    private void initialize() {
        final MeetingTranscriber _this = this;

        this.transcribing.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                meetingTranscriberObjects.add(_this);
                Contracts.throwIfFail(recognizingSetCallback(_this.getImpl().getValue()));
            }
        });

        this.transcribed.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                meetingTranscriberObjects.add(_this);
                Contracts.throwIfFail(recognizedSetCallback(_this.getImpl().getValue()));                
            }
        });

        this.canceled.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                meetingTranscriberObjects.add(_this);
                Contracts.throwIfFail(canceledSetCallback(_this.getImpl().getValue()));                
            }
        });

        this.sessionStarted.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                meetingTranscriberObjects.add(_this);
                Contracts.throwIfFail(sessionStartedSetCallback(_this.getImpl().getValue()));
            }
        });

        this.sessionStopped.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                meetingTranscriberObjects.add(_this);
                Contracts.throwIfFail(sessionStoppedSetCallback(_this.getImpl().getValue()));                
            }
        });

        this.speechStartDetected.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                meetingTranscriberObjects.add(_this);
                Contracts.throwIfFail(speechStartDetectedSetCallback(_this.getImpl().getValue()));
            }
        });

        this.speechEndDetected.updateNotificationOnConnected(new Runnable(){
            @Override
            public void run() {
                meetingTranscriberObjects.add(_this);
                Contracts.throwIfFail(speechEndDetectedSetCallback(_this.getImpl().getValue()));                
            }
        });

        IntRef propHandle = new IntRef(0);
        Contracts.throwIfFail(getPropertyBagFromRecognizerHandle(_this.getImpl(), propHandle));
        propertyHandle = new PropertyCollection(propHandle);
    }

    private void recognizingEventCallback(long eventArgs)
    {
        try {
            Contracts.throwIfNull(this, "recognizer");
            if (this.disposed) {
                return;
            }
            MeetingTranscriptionEventArgs resultEventArg = new MeetingTranscriptionEventArgs(eventArgs, true);
            EventHandlerImpl<MeetingTranscriptionEventArgs> handler = this.transcribing;
            if (handler != null) {
                handler.fireEvent(this, resultEventArg);
            }    
        } catch (Exception e) {}
    }

    private void recognizedEventCallback(long eventArgs)
    {
        try {
            Contracts.throwIfNull(this, "recognizer");
            if (this.disposed) {
                return;
            }
            MeetingTranscriptionEventArgs resultEventArg = new MeetingTranscriptionEventArgs(eventArgs, true);
            EventHandlerImpl<MeetingTranscriptionEventArgs> handler = this.transcribed;
            if (handler != null) {
                handler.fireEvent(this, resultEventArg);
            }    
        } catch (Exception e) {}
    }

    private void canceledEventCallback(long eventArgs)
    {
        try {
            Contracts.throwIfNull(this, "recognizer");
            if (this.disposed) {
                return;
            }
            MeetingTranscriptionCanceledEventArgs resultEventArg = new MeetingTranscriptionCanceledEventArgs(eventArgs, true);
            EventHandlerImpl<MeetingTranscriptionCanceledEventArgs> handler = this.canceled;
            if (handler != null) {
                handler.fireEvent(this, resultEventArg);
            }    
        } catch (Exception e) {}
    }

    private final native long createMeetingTranscriberFromConfig(SafeHandle recoHandle, SafeHandle audioConfigHandle);
    private final native long joinMeeting(SafeHandle recoHandle, SafeHandle meetingHandle);
    private final native long leaveMeeting(SafeHandle recoHandle);
    
    private PropertyCollection propertyHandle = null;
}
