//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
//
#include "com_microsoft_cognitiveservices_speech_SpeechSynthesizer.h"
#include "jni_utils.h"
#include "speechapi_c_synthesizer.h"
#include "speechapi_c_factory.h"
#include <string.h>

void SynthesisStartedCallback(SPXSYNTHHANDLE , SPXEVENTHANDLE eventHandle, void* context)
{
    CallbackEventMethod(context, "synthesisStartedEventCallback", eventHandle);
}

void SpeechSynthesizingCallback(SPXSYNTHHANDLE , SPXEVENTHANDLE eventHandle, void* context)
{
    CallbackEventMethod(context, "synthesizingEventCallback", eventHandle);
}

void SynthesisCompletedCallback(SPXSYNTHHANDLE , SPXEVENTHANDLE eventHandle, void* context)
{
    CallbackEventMethod(context, "synthesisCompletedEventCallback", eventHandle);
}

void SynthesisCanceledCallback(SPXSYNTHHANDLE , SPXEVENTHANDLE eventHandle, void* context)
{
    CallbackEventMethod(context, "synthesisCanceledEventCallback", eventHandle);
}

void WordBoundaryCallback(SPXSYNTHHANDLE , SPXEVENTHANDLE eventHandle, void* context)
{
    CallbackEventMethod(context, "wordBoundaryEventCallback", eventHandle);
}

void VisemeReceivedCallback(SPXSYNTHHANDLE , SPXEVENTHANDLE eventHandle, void* context)
{
    CallbackEventMethod(context, "visemeReceivedEventCallback", eventHandle);
}

void BookmarkReachedCallback(SPXSYNTHHANDLE , SPXEVENTHANDLE eventHandle, void* context)
{
    CallbackEventMethod(context, "bookmarkReachedEventCallback", eventHandle);
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    createSpeechSynthesizerFromConfig
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_createSpeechSynthesizerFromConfig
  (JNIEnv *env, jobject , jobject synthesizerHandle, jobject speechConfigHandle, jobject audioConfigHandle)
{
    SPXSYNTHHANDLE synthHandle = SPXHANDLE_INVALID;
    jlong speechConfig = GetObjectHandle(env, speechConfigHandle);
    jlong audioConfig = GetObjectHandle(env, audioConfigHandle);
    SPXHR hr = synthesizer_create_speech_synthesizer_from_config(&synthHandle, (SPXSPEECHCONFIGHANDLE)speechConfig, (SPXAUDIOCONFIGHANDLE)audioConfig);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, synthesizerHandle, (jlong)synthHandle);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    createSpeechSynthesizerFromAutoDetectSourceLangConfig
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_createSpeechSynthesizerFromAutoDetectSourceLangConfig
  (JNIEnv *env, jobject , jobject synthesizerHandle, jobject speechConfigHandle, jobject autoDetectHandle, jobject audioConfigHandle)
{
    SPXSYNTHHANDLE synthHandle = SPXHANDLE_INVALID;
    jlong speechConfig = GetObjectHandle(env, speechConfigHandle);
    jlong autoDetectLangConfig = GetObjectHandle(env, autoDetectHandle);
    jlong audioConfig = GetObjectHandle(env, audioConfigHandle);
    SPXHR hr = synthesizer_create_speech_synthesizer_from_auto_detect_source_lang_config(&synthHandle, (SPXSPEECHCONFIGHANDLE)speechConfig, (SPXAUTODETECTSOURCELANGCONFIGHANDLE)autoDetectLangConfig, (SPXAUDIOCONFIGHANDLE)audioConfig);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, synthesizerHandle, (jlong)synthHandle);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    speakText
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Ljava/lang/String;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_speakText
  (JNIEnv *env, jobject , jobject synthesizerHandle, jstring textStr, jobject resultRef)
{
    jlong synthHandle = GetObjectHandle(env, synthesizerHandle);
    const std::string text = GetStdString(env, textStr);
    SPXRESULTHANDLE result = SPXHANDLE_INVALID;
    SPXHR hr = synthesizer_speak_text((SPXSYNTHHANDLE)synthHandle, text.c_str(), static_cast<uint32_t>(text.length()), &result);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, resultRef, (jlong)result);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    speakSsml
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Ljava/lang/String;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_speakSsml
  (JNIEnv *env, jobject , jobject synthesizerHandle, jstring ssmlStr, jobject resultRef)
{
    jlong synthHandle = GetObjectHandle(env, synthesizerHandle);
    std::string ssml = GetStdString(env, ssmlStr);
    SPXRESULTHANDLE result = SPXHANDLE_INVALID;
    SPXHR hr = synthesizer_speak_ssml((SPXSYNTHHANDLE)synthHandle, ssml.c_str(), static_cast<uint32_t>(ssml.length()), &result);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, resultRef, (jlong)result);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    startSpeakingText
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Ljava/lang/String;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_startSpeakingText
  (JNIEnv *env, jobject , jobject synthesizerHandle, jstring textStr, jobject resultRef)
{
    jlong synthHandle = GetObjectHandle(env, synthesizerHandle);
    std::string text = GetStdString(env, textStr);
    SPXRESULTHANDLE result = SPXHANDLE_INVALID;
    SPXHR hr = synthesizer_start_speaking_text((SPXSYNTHHANDLE)synthHandle, text.c_str(), static_cast<uint32_t>(text.length()), &result);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, resultRef, (jlong)result);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    startSpeakingSsml
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Ljava/lang/String;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_startSpeakingSsml
  (JNIEnv *env, jobject , jobject synthesizerHandle, jstring ssmlStr, jobject resultRef)
{
    jlong synthHandle = GetObjectHandle(env, synthesizerHandle);
    std::string ssml = GetStdString(env, ssmlStr);
    SPXRESULTHANDLE result = SPXHANDLE_INVALID;
    SPXHR hr = synthesizer_start_speaking_ssml((SPXSYNTHHANDLE)synthHandle, ssml.c_str(), static_cast<uint32_t>(ssml.length()), &result);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, resultRef, (jlong)result);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    stopSpeaking
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_stopSpeaking
  (JNIEnv *env, jobject , jobject synthesizerHandle)
{
    jlong synthHandle = GetObjectHandle(env, synthesizerHandle);
    SPXHR hr = synthesizer_stop_speaking((SPXSYNTHHANDLE)synthHandle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    getVoices
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Ljava/lang/String;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_getVoices
  (JNIEnv *env, jobject , jobject synthesizerHandle, jstring localeStr, jobject resultRef)
{
    jlong synthHandle = GetObjectHandle(env, synthesizerHandle);
    const char* locale = GetStringUTFChars(env, localeStr);
    SPXRESULTHANDLE result = SPXHANDLE_INVALID;
    SPXHR hr = synthesizer_get_voices_list((SPXSYNTHHANDLE)synthHandle, locale, &result);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, resultRef, (jlong)result);
    }
    ReleaseStringUTFChars(env, localeStr, locale);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    synthesisStartedSetCallback
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_synthesisStartedSetCallback
  (JNIEnv *env, jobject obj, jobject synthesizerHandle)
{
    jlong handle = GetObjectHandle(env, synthesizerHandle);
    AddGlobalReferenceForHandle(env, obj, (SPXHANDLE)handle);
    SPXHR hr = synthesizer_started_set_callback((SPXSYNTHHANDLE)handle, SynthesisStartedCallback, (SPXHANDLE)handle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    synthesizingSetCallback
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_synthesizingSetCallback
  (JNIEnv* env, jobject obj, jobject synthesizerHandle)
{
    jlong handle = GetObjectHandle(env, synthesizerHandle);
    AddGlobalReferenceForHandle(env, obj, (SPXHANDLE)handle);
    SPXHR hr = synthesizer_synthesizing_set_callback((SPXSYNTHHANDLE)handle, SpeechSynthesizingCallback, (SPXHANDLE)handle);
    return (jlong)hr;
}
/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    synthesisCompletedSetCallback
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_synthesisCompletedSetCallback
  (JNIEnv* env, jobject obj, jobject synthesizerHandle)
{
    jlong handle = GetObjectHandle(env, synthesizerHandle);
    AddGlobalReferenceForHandle(env, obj, (SPXHANDLE)handle);
    SPXHR hr = synthesizer_completed_set_callback((SPXSYNTHHANDLE)handle, SynthesisCompletedCallback, (SPXHANDLE)handle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    synthesisCanceledSetCallback
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_synthesisCanceledSetCallback
  (JNIEnv* env, jobject obj, jobject synthesizerHandle)
{
    jlong handle = GetObjectHandle(env, synthesizerHandle);
    AddGlobalReferenceForHandle(env, obj, (SPXHANDLE)handle);
    SPXHR hr = synthesizer_canceled_set_callback((SPXSYNTHHANDLE)handle, SynthesisCanceledCallback, (SPXHANDLE)handle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    wordBoundarySetCallback
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_wordBoundarySetCallback
  (JNIEnv* env, jobject obj, jobject synthesizerHandle)
{
    jlong handle = GetObjectHandle(env, synthesizerHandle);
    AddGlobalReferenceForHandle(env, obj, (SPXHANDLE)handle);
    SPXHR hr = synthesizer_word_boundary_set_callback((SPXSYNTHHANDLE)handle, WordBoundaryCallback, (SPXHANDLE)handle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    visemeReceivedSetCallback
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_visemeReceivedSetCallback
  (JNIEnv* env, jobject obj, jobject synthesizerHandle)
{
    jlong handle = GetObjectHandle(env, synthesizerHandle);
    AddGlobalReferenceForHandle(env, obj, (SPXHANDLE)handle);
    SPXHR hr = synthesizer_viseme_received_set_callback((SPXSYNTHHANDLE)handle, VisemeReceivedCallback, (SPXHANDLE)handle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    bookmarkReachedSetCallback
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_bookmarkReachedSetCallback
  (JNIEnv* env, jobject obj, jobject synthesizerHandle)
{
    jlong handle = GetObjectHandle(env, synthesizerHandle);
    AddGlobalReferenceForHandle(env, obj, (SPXHANDLE)handle);
    SPXHR hr = synthesizer_bookmark_reached_set_callback((SPXSYNTHHANDLE)handle, BookmarkReachedCallback, (SPXHANDLE)handle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_SpeechSynthesizer
 * Method:    getPropertyBagFromSynthesizerHandle
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_SpeechSynthesizer_getPropertyBagFromSynthesizerHandle
  (JNIEnv* env, jobject , jobject synthHandle, jobject propertyHandle)
{
    SPXPROPERTYBAGHANDLE propHandle = SPXHANDLE_INVALID;
    jlong handle = GetObjectHandle(env, synthHandle);
    SPXHR hr = synthesizer_get_property_bag((SPXSYNTHHANDLE)handle, &propHandle);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, propertyHandle, (jlong)propHandle);
    }
    return (jlong)hr;
}

