// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
package com.microsoft.gctoolkit.parser.jvm;

import com.microsoft.gctoolkit.parser.unified.UnifiedLoggingLevel;
import com.microsoft.gctoolkit.parser.unified.UnifiedLoggingTokens;
import com.microsoft.gctoolkit.time.DateTimeStamp;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.Arrays;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;

public class Decorators {

    /**
     * This class represents the decorators found in the log line as defined by JEP 158 (https://openjdk.java.net/jeps/158).
     * The list, taken from that document is as:
     * time -- Current time and date in ISO-8601 format
     * uptime -- Time since the start of the JVM in seconds and milliseconds (e.g., 6.567s)
     * timemillis -- The same value as generated by System.currentTimeMillis()
     * uptimemillis -- Milliseconds since the JVM started
     * timenanos -- The same value as generated by System.nanoTime()
     * uptimenanos -- Nanoseconds since the JVM started
     * pid -- The process identifier
     * tid -- The thread identifier
     * level -- The level associated with the log message
     * tags -- The tag-set associated with the log message
     *
     * This implementation takes advantage of the property that the ordering of tags is stable.
     * The default decorators (as of the time of authoring JDK 9-17) include [uptime][level][tags].
     * For example, [1.361s][info][gc,heap]. This reads uptime of 1.361 seconds. The tag for the log record
     * is gc.heap at the info level.
     *
     * At the moment, GCToolkit does not differentiate between timenanos and uptimenanos as there is no formatting hint
     * to allow for a differentiation. Less importantly, there is currently no way to differentiate between pid and tid.
     * Todo: GCToolkit captures time in the DateTimeStamp class. That class will capture uptime or time or both. If both
     * are missing, GCToolkit JVMEvents will have no sense of time. It is possible that the other timing fields could fill
     * in cases where both the time and uptime decorators were missing.
     * @param line
     */

    private static final Logger LOGGER = Logger.getLogger(Decorators.class.getName());

    // This is to help differentiate between JVM running time and wall clock time.
    private static final long TWENTY_YEARS_IN_MILLIS = 731L * 24L * 60L * 60L * 1000L;
    private static final long TWENTY_YEARS_IN_NANO = 731L * 24L * 60L * 60L * 1000L;

    int numberOfDecorators;

    private Matcher decoratorMatcher = null;
    private String tags;

    public Decorators(String line) {
        extractValues(line);
    }

    private void extractValues(String line) {

        if (!line.startsWith("["))
            return;

        decoratorMatcher = UnifiedLoggingTokens.DECORATORS.matcher(line);
        if (!decoratorMatcher.find()) {
            return;
        }

        for ( int i = 1; i <= decoratorMatcher.groupCount(); i++) {
            if ( decoratorMatcher.group(i) != null)
                numberOfDecorators++;
        }

        Matcher tagMatcher = UnifiedLoggingTokens.TAGS.matcher(line);
        if (tagMatcher.find()) {
            numberOfDecorators++;
            tags = String.join(",", Arrays.asList(tagMatcher.group(1).trim().split(",")));
        }
    }

    // For some reason, ISO_DATE_TIME doesn't like that time-zone is -0100. It wants -01:00.
    private static final DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZ");

    public ZonedDateTime getDateStamp() {
        try {
            String value = decoratorMatcher.group(1);
            if (value != null) {
                TemporalAccessor temporalAccessor = formatter.parse(value.substring(1, value.length()-1));
                return ZonedDateTime.from(temporalAccessor);            }
        } catch (NullPointerException npe) {
            LOGGER.log(Level.SEVERE, npe.getMessage(), npe);
        }
        return null;
    }

    public double getUpTime() {
        String value = decoratorMatcher.group(2);
        if (value != null) {
            value = value.replace(",", ".");
            return Double.parseDouble(unboxValue(value, 1));
        }
        return -1.0d;
    }

    private long extractClock(int groupIndex, long threshold) {
        long clockReading = -1L;
        String stringValue = decoratorMatcher.group(groupIndex);
        if (stringValue != null) {
            clockReading = Long.parseLong(unboxValue(stringValue, 2));
            if (decoratorMatcher.group(groupIndex + 1) == null)
                if (clockReading < threshold)
                    clockReading = -1L;
        }
        return clockReading;
    }

    public long getTimeMillis() {
        return extractClock(3, TWENTY_YEARS_IN_MILLIS);
    }

    public long getUptimeMillis() {
        String value = decoratorMatcher.group(4);
        if (value == null) {
            value = decoratorMatcher.group(3);
        }
        if (value != null) {
            long longValue = Long.parseLong(unboxValue(value, 2));
            if (longValue < TWENTY_YEARS_IN_MILLIS)
                return longValue;
        }
        return -1L;
    }

    public long getTimeNano() {
        return extractClock(5, TWENTY_YEARS_IN_NANO);
    }

    public long getUptimeNano() {
        String value = decoratorMatcher.group(6);
        if (value == null) {
            value = decoratorMatcher.group(5);
        }
        if (value != null) {
            long longValue = Long.parseLong(unboxValue(value, 2));
            if (longValue < TWENTY_YEARS_IN_NANO)
                return longValue;
        }
        return -1L;
    }

    public int getPid() {
        String value = decoratorMatcher.group(7);
        if (value != null) {
            return Integer.parseInt(unboxValue(value));
        }
        return -1;
    }

    public int getTid() {
        String value = decoratorMatcher.group(8);
        if (value != null) {
            return Integer.parseInt(unboxValue(value));
        }
        return -1;
    }

    public Optional<UnifiedLoggingLevel> getLogLevel() {
        String level = decoratorMatcher.group(9);
        if (level != null)
            try {
                return Optional.of(UnifiedLoggingLevel.valueOf(unboxValue(level)));
            } catch (IllegalArgumentException e) {
                LOGGER.fine("No such debug level: " + level);
                LOGGER.fine(e.getMessage());
                return Optional.empty();
            }
        return Optional.empty();
    }

    public DateTimeStamp getDateTimeStamp() {
        return new DateTimeStamp(getDateStamp(), getUpTime());
    }

    public int getNumberOfDecorators() {
        return numberOfDecorators;
    }

    private String unboxValue(String boxedValue, int postFix) {
        return boxedValue.substring(1, boxedValue.length() - (1 + postFix));
    }

    private String unboxValue(String boxedValue) {
        return unboxValue(boxedValue, 0).trim();
    }

    public boolean tagsContain(String tagList) {
        return tags.contains(tagList);
    }

    public String getTags() {
        return tags;
    }
}