"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.source = void 0;
const source = "// packages/playwright-core/src/server/injected/selectorUtils.ts\nfunction shouldSkipForTextMatching(element) {\n  return element.nodeName === \"SCRIPT\" || element.nodeName === \"STYLE\" || document.head && document.head.contains(element);\n}\nfunction elementText(cache, root) {\n  let value = cache.get(root);\n  if (value === void 0) {\n    value = { full: \"\", immediate: [] };\n    if (!shouldSkipForTextMatching(root)) {\n      let currentImmediate = \"\";\n      if (root instanceof HTMLInputElement && (root.type === \"submit\" || root.type === \"button\")) {\n        value = { full: root.value, immediate: [root.value] };\n      } else {\n        for (let child = root.firstChild; child; child = child.nextSibling) {\n          if (child.nodeType === Node.TEXT_NODE) {\n            value.full += child.nodeValue || \"\";\n            currentImmediate += child.nodeValue || \"\";\n          } else {\n            if (currentImmediate)\n              value.immediate.push(currentImmediate);\n            currentImmediate = \"\";\n            if (child.nodeType === Node.ELEMENT_NODE)\n              value.full += elementText(cache, child).full;\n          }\n        }\n        if (currentImmediate)\n          value.immediate.push(currentImmediate);\n        if (root.shadowRoot)\n          value.full += elementText(cache, root.shadowRoot).full;\n      }\n    }\n    cache.set(root, value);\n  }\n  return value;\n}\n\n// packages/playwright-core/src/server/injected/selectorGenerator.ts\nvar cacheAllowText = /* @__PURE__ */ new Map();\nvar cacheDisallowText = /* @__PURE__ */ new Map();\nvar kNthScore = 1e3;\nfunction querySelector(injectedScript, selector, ownerDocument) {\n  try {\n    const parsedSelector = injectedScript.parseSelector(selector);\n    return {\n      selector,\n      elements: injectedScript.querySelectorAll(parsedSelector, ownerDocument)\n    };\n  } catch (e) {\n    return {\n      selector,\n      elements: []\n    };\n  }\n}\nfunction generateSelector(injectedScript, targetElement, strict) {\n  injectedScript._evaluator.begin();\n  try {\n    targetElement = targetElement.closest(\"button,select,input,[role=button],[role=checkbox],[role=radio]\") || targetElement;\n    const targetTokens = generateSelectorFor(injectedScript, targetElement, strict);\n    const bestTokens = targetTokens || cssFallback(injectedScript, targetElement, strict);\n    const selector = joinTokens(bestTokens);\n    const parsedSelector = injectedScript.parseSelector(selector);\n    return {\n      selector,\n      elements: injectedScript.querySelectorAll(parsedSelector, targetElement.ownerDocument)\n    };\n  } finally {\n    cacheAllowText.clear();\n    cacheDisallowText.clear();\n    injectedScript._evaluator.end();\n  }\n}\nfunction filterRegexTokens(textCandidates) {\n  return textCandidates.filter((c) => c[0].selector[0] !== \"/\");\n}\nfunction generateSelectorFor(injectedScript, targetElement, strict) {\n  if (targetElement.ownerDocument.documentElement === targetElement)\n    return [{ engine: \"css\", selector: \"html\", score: 1 }];\n  const calculate = (element, allowText) => {\n    const allowNthMatch = element === targetElement;\n    let textCandidates = allowText ? buildTextCandidates(injectedScript, element, element === targetElement).map((token) => [token]) : [];\n    if (element !== targetElement) {\n      textCandidates = filterRegexTokens(textCandidates);\n    }\n    const noTextCandidates = buildCandidates(injectedScript, element).map((token) => [token]);\n    let result = chooseFirstSelector(injectedScript, targetElement.ownerDocument, element, [...textCandidates, ...noTextCandidates], allowNthMatch, strict);\n    textCandidates = filterRegexTokens(textCandidates);\n    const checkWithText = (textCandidatesToUse) => {\n      const allowParentText = allowText && !textCandidatesToUse.length;\n      const candidates = [...textCandidatesToUse, ...noTextCandidates].filter((c) => {\n        if (!result)\n          return true;\n        return combineScores(c) < combineScores(result);\n      });\n      let bestPossibleInParent = candidates[0];\n      if (!bestPossibleInParent)\n        return;\n      for (let parent = parentElementOrShadowHost(element); parent; parent = parentElementOrShadowHost(parent)) {\n        const parentTokens = calculateCached(parent, allowParentText);\n        if (!parentTokens)\n          continue;\n        if (result && combineScores([...parentTokens, ...bestPossibleInParent]) >= combineScores(result))\n          continue;\n        bestPossibleInParent = chooseFirstSelector(injectedScript, parent, element, candidates, allowNthMatch, strict);\n        if (!bestPossibleInParent)\n          return;\n        const combined = [...parentTokens, ...bestPossibleInParent];\n        if (!result || combineScores(combined) < combineScores(result))\n          result = combined;\n      }\n    };\n    checkWithText(textCandidates);\n    if (element === targetElement && textCandidates.length)\n      checkWithText([]);\n    return result;\n  };\n  const calculateCached = (element, allowText) => {\n    const cache = allowText ? cacheAllowText : cacheDisallowText;\n    let value = cache.get(element);\n    if (value === void 0) {\n      value = calculate(element, allowText);\n      cache.set(element, value);\n    }\n    return value;\n  };\n  return calculateCached(targetElement, true);\n}\nfunction buildCandidates(injectedScript, element) {\n  const candidates = [];\n  for (const attribute of [\"data-testid\", \"data-test-id\", \"data-test\"]) {\n    if (element.getAttribute(attribute))\n      candidates.push({ engine: \"css\", selector: `[${attribute}=${quoteAttributeValue(element.getAttribute(attribute))}]`, score: 1 });\n  }\n  if (element.nodeName === \"INPUT\") {\n    const input = element;\n    if (input.placeholder)\n      candidates.push({ engine: \"css\", selector: `[placeholder=${quoteAttributeValue(input.placeholder)}]`, score: 10 });\n  }\n  if (element.getAttribute(\"aria-label\"))\n    candidates.push({ engine: \"css\", selector: `[aria-label=${quoteAttributeValue(element.getAttribute(\"aria-label\"))}]`, score: 10 });\n  if (element.getAttribute(\"alt\") && [\"APPLET\", \"AREA\", \"IMG\", \"INPUT\"].includes(element.nodeName))\n    candidates.push({ engine: \"css\", selector: `${cssEscape(element.nodeName.toLowerCase())}[alt=${quoteAttributeValue(element.getAttribute(\"alt\"))}]`, score: 10 });\n  if (element.getAttribute(\"role\"))\n    candidates.push({ engine: \"css\", selector: `${cssEscape(element.nodeName.toLowerCase())}[role=${quoteAttributeValue(element.getAttribute(\"role\"))}]`, score: 50 });\n  if (element.getAttribute(\"name\") && [\"BUTTON\", \"FORM\", \"FIELDSET\", \"IFRAME\", \"INPUT\", \"KEYGEN\", \"OBJECT\", \"OUTPUT\", \"SELECT\", \"TEXTAREA\", \"MAP\", \"META\", \"PARAM\"].includes(element.nodeName))\n    candidates.push({ engine: \"css\", selector: `${cssEscape(element.nodeName.toLowerCase())}[name=${quoteAttributeValue(element.getAttribute(\"name\"))}]`, score: 50 });\n  if ([\"INPUT\", \"TEXTAREA\"].includes(element.nodeName) && element.getAttribute(\"type\") !== \"hidden\") {\n    if (element.getAttribute(\"type\"))\n      candidates.push({ engine: \"css\", selector: `${cssEscape(element.nodeName.toLowerCase())}[type=${quoteAttributeValue(element.getAttribute(\"type\"))}]`, score: 50 });\n  }\n  if ([\"INPUT\", \"TEXTAREA\", \"SELECT\"].includes(element.nodeName))\n    candidates.push({ engine: \"css\", selector: cssEscape(element.nodeName.toLowerCase()), score: 50 });\n  const idAttr = element.getAttribute(\"id\");\n  if (idAttr && !isGuidLike(idAttr))\n    candidates.push({ engine: \"css\", selector: makeSelectorForId(idAttr), score: 100 });\n  candidates.push({ engine: \"css\", selector: cssEscape(element.nodeName.toLowerCase()), score: 200 });\n  return candidates;\n}\nfunction buildTextCandidates(injectedScript, element, allowHasText) {\n  if (element.nodeName === \"SELECT\")\n    return [];\n  const text = elementText(injectedScript._evaluator._cacheText, element).full.trim().replace(/\\s+/g, \" \").substring(0, 80);\n  if (!text)\n    return [];\n  const candidates = [];\n  let escaped = text;\n  if (text.includes('\"') || text.includes(\">>\") || text[0] === \"/\")\n    escaped = `/.*${escapeForRegex(text)}.*/`;\n  candidates.push({ engine: \"text\", selector: escaped, score: 10 });\n  if (allowHasText && escaped === text) {\n    let prefix = element.nodeName.toLowerCase();\n    if (element.hasAttribute(\"role\"))\n      prefix += `[role=${quoteAttributeValue(element.getAttribute(\"role\"))}]`;\n    candidates.push({ engine: \"css\", selector: `${prefix}:has-text(\"${text}\")`, score: 30 });\n  }\n  return candidates;\n}\nfunction parentElementOrShadowHost(element) {\n  if (element.parentElement)\n    return element.parentElement;\n  if (!element.parentNode)\n    return null;\n  if (element.parentNode.nodeType === Node.DOCUMENT_FRAGMENT_NODE && element.parentNode.host)\n    return element.parentNode.host;\n  return null;\n}\nfunction makeSelectorForId(id) {\n  return /^[a-zA-Z][a-zA-Z0-9\\-\\_]+$/.test(id) ? \"#\" + id : `[id=\"${cssEscape(id)}\"]`;\n}\nfunction cssFallback(injectedScript, targetElement, strict) {\n  const kFallbackScore = 1e7;\n  const root = targetElement.ownerDocument;\n  const tokens = [];\n  function uniqueCSSSelector(prefix) {\n    const path = tokens.slice();\n    if (prefix)\n      path.unshift(prefix);\n    const selector = path.join(\" > \");\n    const parsedSelector = injectedScript.parseSelector(selector);\n    const node = injectedScript.querySelector(parsedSelector, targetElement.ownerDocument, false);\n    return node === targetElement ? selector : void 0;\n  }\n  function makeStrict(selector) {\n    const token = { engine: \"css\", selector, score: kFallbackScore };\n    if (!strict)\n      return [token];\n    const parsedSelector = injectedScript.parseSelector(selector);\n    const elements = injectedScript.querySelectorAll(parsedSelector, targetElement.ownerDocument);\n    if (elements.length === 1)\n      return [token];\n    const nth = { engine: \"nth\", selector: String(elements.indexOf(targetElement)), score: kNthScore };\n    return [token, nth];\n  }\n  for (let element = targetElement; element && element !== root; element = parentElementOrShadowHost(element)) {\n    const nodeName = element.nodeName.toLowerCase();\n    let bestTokenForLevel = \"\";\n    if (element.id) {\n      const token = makeSelectorForId(element.id);\n      const selector = uniqueCSSSelector(token);\n      if (selector)\n        return makeStrict(selector);\n      bestTokenForLevel = token;\n    }\n    const parent = element.parentNode;\n    const classes = [...element.classList];\n    for (let i = 0; i < classes.length; ++i) {\n      const token = \".\" + classes.slice(0, i + 1).join(\".\");\n      const selector = uniqueCSSSelector(token);\n      if (selector)\n        return makeStrict(selector);\n      if (!bestTokenForLevel && parent) {\n        const sameClassSiblings = parent.querySelectorAll(token);\n        if (sameClassSiblings.length === 1)\n          bestTokenForLevel = token;\n      }\n    }\n    if (parent) {\n      const siblings = [...parent.children];\n      const sameTagSiblings = siblings.filter((sibling) => sibling.nodeName.toLowerCase() === nodeName);\n      const token = sameTagSiblings.indexOf(element) === 0 ? cssEscape(nodeName) : `${cssEscape(nodeName)}:nth-child(${1 + siblings.indexOf(element)})`;\n      const selector = uniqueCSSSelector(token);\n      if (selector)\n        return makeStrict(selector);\n      if (!bestTokenForLevel)\n        bestTokenForLevel = token;\n    } else if (!bestTokenForLevel) {\n      bestTokenForLevel = nodeName;\n    }\n    tokens.unshift(bestTokenForLevel);\n  }\n  return makeStrict(uniqueCSSSelector());\n}\nfunction escapeForRegex(text) {\n  return text.replace(/[.*+?^>${}()|[\\]\\\\]/g, \"\\\\$&\");\n}\nfunction quoteAttributeValue(text) {\n  return `\"${cssEscape(text).replace(/\\\\ /g, \" \")}\"`;\n}\nfunction joinTokens(tokens) {\n  const parts = [];\n  let lastEngine = \"\";\n  for (const { engine, selector } of tokens) {\n    if (parts.length && (lastEngine !== \"css\" || engine !== \"css\" || selector.startsWith(\":nth-match(\")))\n      parts.push(\">>\");\n    lastEngine = engine;\n    if (engine === \"css\")\n      parts.push(selector);\n    else\n      parts.push(`${engine}=${selector}`);\n  }\n  return parts.join(\" \");\n}\nfunction combineScores(tokens) {\n  let score = 0;\n  for (let i = 0; i < tokens.length; i++)\n    score += tokens[i].score * (tokens.length - i);\n  return score;\n}\nfunction chooseFirstSelector(injectedScript, scope, targetElement, selectors, allowNthMatch, strict) {\n  const joined = selectors.map((tokens) => ({ tokens, score: combineScores(tokens) }));\n  joined.sort((a, b) => a.score - b.score);\n  let bestWithIndex = null;\n  for (const { tokens } of joined) {\n    const parsedSelector = injectedScript.parseSelector(joinTokens(tokens));\n    const result = injectedScript.querySelectorAll(parsedSelector, scope);\n    const isStrictEnough = !strict || result.length === 1;\n    const index = result.indexOf(targetElement);\n    if (index === 0 && isStrictEnough) {\n      return tokens;\n    }\n    if (!allowNthMatch || bestWithIndex || index === -1 || result.length > 5)\n      continue;\n    const nth = { engine: \"nth\", selector: String(index), score: kNthScore };\n    bestWithIndex = [...tokens, nth];\n  }\n  return bestWithIndex;\n}\nfunction isGuidLike(id) {\n  let lastCharacterType;\n  let transitionCount = 0;\n  for (let i = 0; i < id.length; ++i) {\n    const c = id[i];\n    let characterType;\n    if (c === \"-\" || c === \"_\")\n      continue;\n    if (c >= \"a\" && c <= \"z\")\n      characterType = \"lower\";\n    else if (c >= \"A\" && c <= \"Z\")\n      characterType = \"upper\";\n    else if (c >= \"0\" && c <= \"9\")\n      characterType = \"digit\";\n    else\n      characterType = \"other\";\n    if (characterType === \"lower\" && lastCharacterType === \"upper\") {\n      lastCharacterType = characterType;\n      continue;\n    }\n    if (lastCharacterType && lastCharacterType !== characterType)\n      ++transitionCount;\n    lastCharacterType = characterType;\n  }\n  return transitionCount >= id.length / 4;\n}\nfunction cssEscape(s) {\n  let result = \"\";\n  for (let i = 0; i < s.length; i++)\n    result += cssEscapeOne(s, i);\n  return result;\n}\nfunction cssEscapeOne(s, i) {\n  const c = s.charCodeAt(i);\n  if (c === 0)\n    return \"\\uFFFD\";\n  if (c >= 1 && c <= 31 || c >= 48 && c <= 57 && (i === 0 || i === 1 && s.charCodeAt(0) === 45))\n    return \"\\\\\" + c.toString(16) + \" \";\n  if (i === 0 && c === 45 && s.length === 1)\n    return \"\\\\\" + s.charAt(i);\n  if (c >= 128 || c === 45 || c === 95 || c >= 48 && c <= 57 || c >= 65 && c <= 90 || c >= 97 && c <= 122)\n    return s.charAt(i);\n  return \"\\\\\" + s.charAt(i);\n}\n\n// packages/playwright-core/src/server/injected/highlight.ts\nvar Highlight = class {\n  constructor(isUnderTest) {\n    this._highlightElements = [];\n    this._isUnderTest = isUnderTest;\n    this._outerGlassPaneElement = document.createElement(\"x-pw-glass\");\n    this._outerGlassPaneElement.style.position = \"fixed\";\n    this._outerGlassPaneElement.style.top = \"0\";\n    this._outerGlassPaneElement.style.right = \"0\";\n    this._outerGlassPaneElement.style.bottom = \"0\";\n    this._outerGlassPaneElement.style.left = \"0\";\n    this._outerGlassPaneElement.style.zIndex = \"2147483647\";\n    this._outerGlassPaneElement.style.pointerEvents = \"none\";\n    this._outerGlassPaneElement.style.display = \"flex\";\n    this._tooltipElement = document.createElement(\"x-pw-tooltip\");\n    this._actionPointElement = document.createElement(\"x-pw-action-point\");\n    this._actionPointElement.setAttribute(\"hidden\", \"true\");\n    this._innerGlassPaneElement = document.createElement(\"x-pw-glass-inner\");\n    this._innerGlassPaneElement.style.flex = \"auto\";\n    this._innerGlassPaneElement.appendChild(this._tooltipElement);\n    this._glassPaneShadow = this._outerGlassPaneElement.attachShadow({ mode: isUnderTest ? \"open\" : \"closed\" });\n    this._glassPaneShadow.appendChild(this._innerGlassPaneElement);\n    this._glassPaneShadow.appendChild(this._actionPointElement);\n    const styleElement = document.createElement(\"style\");\n    styleElement.textContent = `\n        x-pw-tooltip {\n          align-items: center;\n          backdrop-filter: blur(5px);\n          background-color: rgba(0, 0, 0, 0.7);\n          border-radius: 2px;\n          box-shadow: rgba(0, 0, 0, 0.1) 0px 3.6px 3.7px,\n                      rgba(0, 0, 0, 0.15) 0px 12.1px 12.3px,\n                      rgba(0, 0, 0, 0.1) 0px -2px 4px,\n                      rgba(0, 0, 0, 0.15) 0px -12.1px 24px,\n                      rgba(0, 0, 0, 0.25) 0px 54px 55px;\n          color: rgb(204, 204, 204);\n          display: none;\n          font-family: 'Dank Mono', 'Operator Mono', Inconsolata, 'Fira Mono',\n                      'SF Mono', Monaco, 'Droid Sans Mono', 'Source Code Pro', monospace;\n          font-size: 12.8px;\n          font-weight: normal;\n          left: 0;\n          line-height: 1.5;\n          max-width: 600px;\n          padding: 3.2px 5.12px 3.2px;\n          position: absolute;\n          top: 0;\n        }\n        x-pw-action-point {\n          position: absolute;\n          width: 20px;\n          height: 20px;\n          background: red;\n          border-radius: 10px;\n          pointer-events: none;\n          margin: -10px 0 0 -10px;\n          z-index: 2;\n        }\n        *[hidden] {\n          display: none !important;\n        }\n    `;\n    this._glassPaneShadow.appendChild(styleElement);\n  }\n  install() {\n    document.documentElement.appendChild(this._outerGlassPaneElement);\n  }\n  uninstall() {\n    this._outerGlassPaneElement.remove();\n  }\n  isInstalled() {\n    return this._outerGlassPaneElement.parentElement === document.documentElement && !this._outerGlassPaneElement.nextElementSibling;\n  }\n  showActionPoint(x, y) {\n    this._actionPointElement.style.top = y + \"px\";\n    this._actionPointElement.style.left = x + \"px\";\n    this._actionPointElement.hidden = false;\n    if (this._isUnderTest)\n      console.error(\"Action point for test: \" + JSON.stringify({ x, y }));\n  }\n  hideActionPoint() {\n    this._actionPointElement.hidden = true;\n  }\n  updateHighlight(elements, selector, isRecording) {\n    this._tooltipElement.textContent = selector;\n    this._tooltipElement.style.top = \"0\";\n    this._tooltipElement.style.left = \"0\";\n    this._tooltipElement.style.display = \"flex\";\n    const boxes = elements.map((e) => e.getBoundingClientRect());\n    const tooltipWidth = this._tooltipElement.offsetWidth;\n    const tooltipHeight = this._tooltipElement.offsetHeight;\n    const totalWidth = this._innerGlassPaneElement.offsetWidth;\n    const totalHeight = this._innerGlassPaneElement.offsetHeight;\n    if (boxes.length) {\n      const primaryBox = boxes[0];\n      let anchorLeft = primaryBox.left;\n      if (anchorLeft + tooltipWidth > totalWidth - 5)\n        anchorLeft = totalWidth - tooltipWidth - 5;\n      let anchorTop = primaryBox.bottom + 5;\n      if (anchorTop + tooltipHeight > totalHeight - 5) {\n        if (primaryBox.top > tooltipHeight + 5) {\n          anchorTop = primaryBox.top - tooltipHeight - 5;\n        } else {\n          anchorTop = totalHeight - 5 - tooltipHeight;\n        }\n      }\n      this._tooltipElement.style.top = anchorTop + \"px\";\n      this._tooltipElement.style.left = anchorLeft + \"px\";\n    } else {\n      this._tooltipElement.style.display = \"none\";\n    }\n    const pool = this._highlightElements;\n    this._highlightElements = [];\n    for (const box of boxes) {\n      const highlightElement = pool.length ? pool.shift() : this._createHighlightElement();\n      const color = isRecording ? \"#dc6f6f7f\" : \"#6fa8dc7f\";\n      highlightElement.style.backgroundColor = this._highlightElements.length ? \"#f6b26b7f\" : color;\n      highlightElement.style.left = box.x + \"px\";\n      highlightElement.style.top = box.y + \"px\";\n      highlightElement.style.width = box.width + \"px\";\n      highlightElement.style.height = box.height + \"px\";\n      highlightElement.style.display = \"block\";\n      this._highlightElements.push(highlightElement);\n      if (this._isUnderTest)\n        console.error(\"Highlight box for test: \" + JSON.stringify({ x: box.x, y: box.y, width: box.width, height: box.height }));\n    }\n    for (const highlightElement of pool) {\n      highlightElement.style.display = \"none\";\n      this._highlightElements.push(highlightElement);\n    }\n  }\n  maskElements(elements) {\n    const boxes = elements.map((e) => e.getBoundingClientRect());\n    const pool = this._highlightElements;\n    this._highlightElements = [];\n    for (const box of boxes) {\n      const highlightElement = pool.length ? pool.shift() : this._createHighlightElement();\n      highlightElement.style.backgroundColor = \"#F0F\";\n      highlightElement.style.left = box.x + \"px\";\n      highlightElement.style.top = box.y + \"px\";\n      highlightElement.style.width = box.width + \"px\";\n      highlightElement.style.height = box.height + \"px\";\n      highlightElement.style.display = \"block\";\n      this._highlightElements.push(highlightElement);\n    }\n    for (const highlightElement of pool) {\n      highlightElement.style.display = \"none\";\n      this._highlightElements.push(highlightElement);\n    }\n  }\n  _createHighlightElement() {\n    const highlightElement = document.createElement(\"x-pw-highlight\");\n    highlightElement.style.position = \"absolute\";\n    highlightElement.style.top = \"0\";\n    highlightElement.style.left = \"0\";\n    highlightElement.style.width = \"0\";\n    highlightElement.style.height = \"0\";\n    highlightElement.style.boxSizing = \"border-box\";\n    this._glassPaneShadow.appendChild(highlightElement);\n    return highlightElement;\n  }\n};\n\n// packages/playwright-core/src/server/injected/recorder.ts\nvar Recorder = class {\n  constructor(injectedScript) {\n    this._performingAction = false;\n    this._listeners = [];\n    this._hoveredModel = null;\n    this._hoveredElement = null;\n    this._activeModel = null;\n    this._expectProgrammaticKeyUp = false;\n    this._mode = \"none\";\n    this._injectedScript = injectedScript;\n    this._highlight = new Highlight(injectedScript.isUnderTest);\n    this._refreshListenersIfNeeded();\n    injectedScript.onGlobalListenersRemoved.add(() => this._refreshListenersIfNeeded());\n    globalThis.__pw_refreshOverlay = () => {\n      this._pollRecorderMode().catch((e) => console.log(e));\n    };\n    globalThis.__pw_refreshOverlay();\n    if (injectedScript.isUnderTest)\n      console.error(\"Recorder script ready for test\");\n  }\n  _refreshListenersIfNeeded() {\n    if (this._highlight.isInstalled())\n      return;\n    removeEventListeners(this._listeners);\n    this._listeners = [\n      addEventListener(document, \"click\", (event) => this._onClick(event), true),\n      addEventListener(document, \"auxclick\", (event) => this._onClick(event), true),\n      addEventListener(document, \"input\", (event) => this._onInput(event), true),\n      addEventListener(document, \"keydown\", (event) => this._onKeyDown(event), true),\n      addEventListener(document, \"keyup\", (event) => this._onKeyUp(event), true),\n      addEventListener(document, \"mousedown\", (event) => this._onMouseDown(event), true),\n      addEventListener(document, \"mouseup\", (event) => this._onMouseUp(event), true),\n      addEventListener(document, \"mousemove\", (event) => this._onMouseMove(event), true),\n      addEventListener(document, \"mouseleave\", (event) => this._onMouseLeave(event), true),\n      addEventListener(document, \"focus\", () => this._onFocus(), true),\n      addEventListener(document, \"scroll\", () => {\n        this._hoveredModel = null;\n        this._highlight.hideActionPoint();\n        this._updateHighlight();\n      }, true)\n    ];\n    this._highlight.install();\n  }\n  async _pollRecorderMode() {\n    var _a;\n    const pollPeriod = 1e3;\n    if (this._pollRecorderModeTimer)\n      clearTimeout(this._pollRecorderModeTimer);\n    const state = await globalThis.__pw_recorderState().catch((e) => null);\n    if (!state) {\n      this._pollRecorderModeTimer = setTimeout(() => this._pollRecorderMode(), pollPeriod);\n      return;\n    }\n    const { mode, actionPoint, actionSelector } = state;\n    if (mode !== this._mode) {\n      this._mode = mode;\n      this._clearHighlight();\n    }\n    if (actionPoint && this._actionPoint && actionPoint.x === this._actionPoint.x && actionPoint.y === this._actionPoint.y) {\n    } else if (!actionPoint && !this._actionPoint) {\n    } else {\n      if (actionPoint)\n        this._highlight.showActionPoint(actionPoint.x, actionPoint.y);\n      else\n        this._highlight.hideActionPoint();\n      this._actionPoint = actionPoint;\n    }\n    if (this._actionSelector && !((_a = this._hoveredModel) == null ? void 0 : _a.elements.length))\n      this._actionSelector = void 0;\n    if (actionSelector !== this._actionSelector) {\n      this._hoveredModel = actionSelector ? querySelector(this._injectedScript, actionSelector, document) : null;\n      this._updateHighlight();\n      this._actionSelector = actionSelector;\n    }\n    this._pollRecorderModeTimer = setTimeout(() => this._pollRecorderMode(), pollPeriod);\n  }\n  _clearHighlight() {\n    this._hoveredModel = null;\n    this._activeModel = null;\n    this._updateHighlight();\n  }\n  _actionInProgress(event) {\n    if (this._performingAction)\n      return true;\n    consumeEvent(event);\n    return false;\n  }\n  _consumedDueToNoModel(event, model) {\n    if (model)\n      return false;\n    consumeEvent(event);\n    return true;\n  }\n  _consumedDueWrongTarget(event) {\n    if (this._activeModel && this._activeModel.elements[0] === this._deepEventTarget(event))\n      return false;\n    consumeEvent(event);\n    return true;\n  }\n  _onClick(event) {\n    if (this._mode === \"inspecting\")\n      globalThis.__pw_recorderSetSelector(this._hoveredModel ? this._hoveredModel.selector : \"\");\n    if (this._shouldIgnoreMouseEvent(event))\n      return;\n    if (this._actionInProgress(event))\n      return;\n    if (this._consumedDueToNoModel(event, this._hoveredModel))\n      return;\n    const checkbox = asCheckbox(this._deepEventTarget(event));\n    if (checkbox) {\n      this._performAction({\n        name: checkbox.checked ? \"check\" : \"uncheck\",\n        selector: this._hoveredModel.selector,\n        signals: []\n      });\n      return;\n    }\n    this._performAction({\n      name: \"click\",\n      selector: this._hoveredModel.selector,\n      position: positionForEvent(event),\n      signals: [],\n      button: buttonForEvent(event),\n      modifiers: modifiersForEvent(event),\n      clickCount: event.detail\n    });\n  }\n  _shouldIgnoreMouseEvent(event) {\n    const target = this._deepEventTarget(event);\n    if (this._mode === \"none\")\n      return true;\n    if (this._mode === \"inspecting\") {\n      consumeEvent(event);\n      return true;\n    }\n    const nodeName = target.nodeName;\n    if (nodeName === \"SELECT\")\n      return true;\n    if (nodeName === \"INPUT\" && [\"date\"].includes(target.type))\n      return true;\n    return false;\n  }\n  _onMouseDown(event) {\n    if (this._shouldIgnoreMouseEvent(event))\n      return;\n    if (!this._performingAction)\n      consumeEvent(event);\n    this._activeModel = this._hoveredModel;\n  }\n  _onMouseUp(event) {\n    if (this._shouldIgnoreMouseEvent(event))\n      return;\n    if (!this._performingAction)\n      consumeEvent(event);\n  }\n  _onMouseMove(event) {\n    if (this._mode === \"none\")\n      return;\n    const target = this._deepEventTarget(event);\n    if (this._hoveredElement === target)\n      return;\n    this._hoveredElement = target;\n    this._updateModelForHoveredElement();\n  }\n  _onMouseLeave(event) {\n    if (this._deepEventTarget(event).nodeType === Node.DOCUMENT_NODE) {\n      this._hoveredElement = null;\n      this._updateModelForHoveredElement();\n    }\n  }\n  _onFocus() {\n    const activeElement = this._deepActiveElement(document);\n    const result = activeElement ? generateSelector(this._injectedScript, activeElement, true) : null;\n    this._activeModel = result && result.selector ? result : null;\n    if (this._injectedScript.isUnderTest)\n      console.error(\"Highlight updated for test: \" + (result ? result.selector : null));\n  }\n  _updateModelForHoveredElement() {\n    if (!this._hoveredElement) {\n      this._hoveredModel = null;\n      this._updateHighlight();\n      return;\n    }\n    const hoveredElement = this._hoveredElement;\n    const { selector, elements } = generateSelector(this._injectedScript, hoveredElement, true);\n    if (this._hoveredModel && this._hoveredModel.selector === selector || this._hoveredElement !== hoveredElement)\n      return;\n    this._hoveredModel = selector ? { selector, elements } : null;\n    this._updateHighlight();\n    if (this._injectedScript.isUnderTest)\n      console.error(\"Highlight updated for test: \" + selector);\n  }\n  _updateHighlight() {\n    const elements = this._hoveredModel ? this._hoveredModel.elements : [];\n    const selector = this._hoveredModel ? this._hoveredModel.selector : \"\";\n    this._highlight.updateHighlight(elements, selector, this._mode === \"recording\");\n  }\n  _onInput(event) {\n    if (this._mode !== \"recording\")\n      return true;\n    const target = this._deepEventTarget(event);\n    if ([\"INPUT\", \"TEXTAREA\"].includes(target.nodeName)) {\n      const inputElement = target;\n      const elementType = (inputElement.type || \"\").toLowerCase();\n      if ([\"checkbox\", \"radio\"].includes(elementType)) {\n        return;\n      }\n      if (elementType === \"file\") {\n        globalThis.__pw_recorderRecordAction({\n          name: \"setInputFiles\",\n          selector: this._activeModel.selector,\n          signals: [],\n          files: [...inputElement.files || []].map((file) => file.name)\n        });\n        return;\n      }\n      if (this._consumedDueWrongTarget(event))\n        return;\n      globalThis.__pw_recorderRecordAction({\n        name: \"fill\",\n        selector: this._activeModel.selector,\n        signals: [],\n        text: inputElement.value\n      });\n    }\n    if (target.nodeName === \"SELECT\") {\n      const selectElement = target;\n      if (this._actionInProgress(event))\n        return;\n      this._performAction({\n        name: \"select\",\n        selector: this._hoveredModel.selector,\n        options: [...selectElement.selectedOptions].map((option) => option.value),\n        signals: []\n      });\n    }\n  }\n  _shouldGenerateKeyPressFor(event) {\n    if ([\"Backspace\", \"Delete\", \"AltGraph\"].includes(event.key))\n      return false;\n    if (event.key === \"@\" && event.code === \"KeyL\")\n      return false;\n    if (navigator.platform.includes(\"Mac\")) {\n      if (event.key === \"v\" && event.metaKey)\n        return false;\n    } else {\n      if (event.key === \"v\" && event.ctrlKey)\n        return false;\n      if (event.key === \"Insert\" && event.shiftKey)\n        return false;\n    }\n    if ([\"Shift\", \"Control\", \"Meta\", \"Alt\"].includes(event.key))\n      return false;\n    const hasModifier = event.ctrlKey || event.altKey || event.metaKey;\n    if (event.key.length === 1 && !hasModifier)\n      return !!asCheckbox(this._deepEventTarget(event));\n    return true;\n  }\n  _onKeyDown(event) {\n    if (this._mode === \"inspecting\") {\n      consumeEvent(event);\n      return;\n    }\n    if (this._mode !== \"recording\")\n      return;\n    if (!this._shouldGenerateKeyPressFor(event))\n      return;\n    if (this._actionInProgress(event)) {\n      this._expectProgrammaticKeyUp = true;\n      return;\n    }\n    if (this._consumedDueWrongTarget(event))\n      return;\n    if (event.key === \" \") {\n      const checkbox = asCheckbox(this._deepEventTarget(event));\n      if (checkbox) {\n        this._performAction({\n          name: checkbox.checked ? \"uncheck\" : \"check\",\n          selector: this._activeModel.selector,\n          signals: []\n        });\n        return;\n      }\n    }\n    this._performAction({\n      name: \"press\",\n      selector: this._activeModel.selector,\n      signals: [],\n      key: event.key,\n      modifiers: modifiersForEvent(event)\n    });\n  }\n  _onKeyUp(event) {\n    if (this._mode === \"none\")\n      return;\n    if (!this._shouldGenerateKeyPressFor(event))\n      return;\n    if (!this._expectProgrammaticKeyUp) {\n      consumeEvent(event);\n      return;\n    }\n    this._expectProgrammaticKeyUp = false;\n  }\n  async _performAction(action) {\n    this._clearHighlight();\n    this._performingAction = true;\n    await globalThis.__pw_recorderPerformAction(action).catch(() => {\n    });\n    this._performingAction = false;\n    this._updateModelForHoveredElement();\n    this._onFocus();\n    if (this._injectedScript.isUnderTest) {\n      console.error(\"Action performed for test: \" + JSON.stringify({\n        hovered: this._hoveredModel ? this._hoveredModel.selector : null,\n        active: this._activeModel ? this._activeModel.selector : null\n      }));\n    }\n  }\n  _deepEventTarget(event) {\n    return event.composedPath()[0];\n  }\n  _deepActiveElement(document2) {\n    let activeElement = document2.activeElement;\n    while (activeElement && activeElement.shadowRoot && activeElement.shadowRoot.activeElement)\n      activeElement = activeElement.shadowRoot.activeElement;\n    return activeElement;\n  }\n};\nfunction modifiersForEvent(event) {\n  return (event.altKey ? 1 : 0) | (event.ctrlKey ? 2 : 0) | (event.metaKey ? 4 : 0) | (event.shiftKey ? 8 : 0);\n}\nfunction buttonForEvent(event) {\n  switch (event.which) {\n    case 1:\n      return \"left\";\n    case 2:\n      return \"middle\";\n    case 3:\n      return \"right\";\n  }\n  return \"left\";\n}\nfunction positionForEvent(event) {\n  const targetElement = event.target;\n  if (targetElement.nodeName !== \"CANVAS\")\n    return;\n  return {\n    x: event.offsetX,\n    y: event.offsetY\n  };\n}\nfunction consumeEvent(e) {\n  e.preventDefault();\n  e.stopPropagation();\n  e.stopImmediatePropagation();\n}\nfunction asCheckbox(node) {\n  if (!node || node.nodeName !== \"INPUT\")\n    return null;\n  const inputElement = node;\n  return [\"checkbox\", \"radio\"].includes(inputElement.type) ? inputElement : null;\n}\nfunction addEventListener(target, eventName, listener, useCapture) {\n  target.addEventListener(eventName, listener, useCapture);\n  const remove = () => {\n    target.removeEventListener(eventName, listener, useCapture);\n  };\n  return remove;\n}\nfunction removeEventListeners(listeners) {\n  for (const listener of listeners)\n    listener();\n  listeners.splice(0, listeners.length);\n}\nmodule.exports = Recorder;\n";
exports.source = source;