/*
 * Copyright (C) 2013-2015 RoboVM AB
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robovm.apple.coreimage;

/*<imports>*/
import java.io.*;
import java.nio.*;
import java.util.*;
import org.robovm.objc.*;
import org.robovm.objc.annotation.*;
import org.robovm.objc.block.*;
import org.robovm.rt.*;
import org.robovm.rt.annotation.*;
import org.robovm.rt.bro.*;
import org.robovm.rt.bro.annotation.*;
import org.robovm.rt.bro.ptr.*;
import org.robovm.apple.foundation.*;
import org.robovm.apple.corefoundation.*;
import org.robovm.apple.coregraphics.*;
import org.robovm.apple.opengles.*;
import org.robovm.apple.corevideo.*;
import org.robovm.apple.imageio.*;
import org.robovm.apple.uikit.*;
import org.robovm.apple.metal.*;
import org.robovm.apple.iosurface.*;
import org.robovm.apple.avfoundation.*;
/*</imports>*/

/*<javadoc>*/
/*</javadoc>*/
/*<annotations>*/@Library("CoreImage")/*</annotations>*/
@Marshaler(/*<name>*/CIDetectorOptions/*</name>*/.Marshaler.class)
/*<visibility>*/public/*</visibility>*/ class /*<name>*/CIDetectorOptions/*</name>*/ 
    extends /*<extends>*/NSDictionaryWrapper/*</extends>*/
    /*<implements>*//*</implements>*/ {

    /*<marshalers>*/
    public static class Marshaler {
        @MarshalsPointer
        public static CIDetectorOptions toObject(Class<CIDetectorOptions> cls, long handle, long flags) {
            NSDictionary o = (NSDictionary) NSObject.Marshaler.toObject(NSDictionary.class, handle, flags);
            if (o == null) {
                return null;
            }
            return new CIDetectorOptions(o);
        }
        @MarshalsPointer
        public static long toNative(CIDetectorOptions o, long flags) {
            if (o == null) {
                return 0L;
            }
            return NSObject.Marshaler.toNative(o.data, flags);
        }
    }
    public static class AsListMarshaler {
        @MarshalsPointer
        public static List<CIDetectorOptions> toObject(Class<? extends NSObject> cls, long handle, long flags) {
            NSArray<NSDictionary> o = (NSArray<NSDictionary>) NSObject.Marshaler.toObject(NSArray.class, handle, flags);
            if (o == null) {
                return null;
            }
            List<CIDetectorOptions> list = new ArrayList<>();
            for (int i = 0; i < o.size(); i++) {
                list.add(new CIDetectorOptions(o.get(i)));
            }
            return list;
        }
        @MarshalsPointer
        public static long toNative(List<CIDetectorOptions> l, long flags) {
            if (l == null) {
                return 0L;
            }
            NSArray<NSDictionary> array = new NSMutableArray<>();
            for (CIDetectorOptions i : l) {
                array.add(i.getDictionary());
            }
            return NSObject.Marshaler.toNative(array, flags);
        }
    }
    /*</marshalers>*/

    /*<constructors>*/
    CIDetectorOptions(NSDictionary data) {
        super(data);
    }
    public CIDetectorOptions() {}
    /*</constructors>*/

    /*<methods>*/
    public boolean has(NSString key) {
        return data.containsKey(key);
    }
    public NSObject get(NSString key) {
        if (has(key)) {
            return data.get(key);
        }
        return null;
    }
    public CIDetectorOptions set(NSString key, NSObject value) {
        data.put(key, value);
        return this;
    }
    

    public CIDetectorAccuracy getAccuracy() {
        if (has(Keys.Accuracy())) {
            NSString val = (NSString) get(Keys.Accuracy());
            return CIDetectorAccuracy.valueOf(val);
        }
        return null;
    }
    public CIDetectorOptions setAccuracy(CIDetectorAccuracy accuracy) {
        set(Keys.Accuracy(), accuracy.value());
        return this;
    }
    public boolean isTracking() {
        if (has(Keys.Tracking())) {
            NSNumber val = (NSNumber) get(Keys.Tracking());
            return val.booleanValue();
        }
        return false;
    }
    public CIDetectorOptions setTracking(boolean tracking) {
        set(Keys.Tracking(), NSNumber.valueOf(tracking));
        return this;
    }
    public double getMinFeatureSize() {
        if (has(Keys.MinFeatureSize())) {
            NSNumber val = (NSNumber) get(Keys.MinFeatureSize());
            return val.doubleValue();
        }
        return 0;
    }
    public CIDetectorOptions setMinFeatureSize(double minFeatureSize) {
        set(Keys.MinFeatureSize(), NSNumber.valueOf(minFeatureSize));
        return this;
    }
    /*</methods>*/
    
    /*<keys>*/
    @Library("CoreImage")
    public static class Keys {
        static { Bro.bind(Keys.class); }
        @GlobalValue(symbol="CIDetectorAccuracy", optional=true)
        public static native NSString Accuracy();
        @GlobalValue(symbol="CIDetectorTracking", optional=true)
        public static native NSString Tracking();
        @GlobalValue(symbol="CIDetectorMinFeatureSize", optional=true)
        public static native NSString MinFeatureSize();
        /**
         * @since Available in iOS 10.0 and later.
         */
        @GlobalValue(symbol="CIDetectorMaxFeatureCount", optional=true)
        public static native NSString MaxFeatureCount();
        /**
         * @since Available in iOS 9.0 and later.
         */
        @GlobalValue(symbol="CIDetectorNumberOfAngles", optional=true)
        public static native NSString NumberOfAngles();
        /**
         * @since Available in iOS 9.0 and later.
         */
        @GlobalValue(symbol="CIDetectorReturnSubFeatures", optional=true)
        public static native NSString ReturnSubFeatures();
    }
    /*</keys>*/
}
