/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp

import android.content.Context
import androidx.annotation.NonNull
import com.moengage.core.LogLevel
import com.moengage.core.internal.SdkInstanceManager
import com.moengage.core.internal.SdkInstanceManager.getSdkInstance
import com.moengage.core.internal.logger.Logger
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.postOnMainThread
import com.moengage.inapp.internal.ConfigurationChangeHandler
import com.moengage.inapp.internal.InAppInstanceProvider
import com.moengage.inapp.internal.InAppModuleManager
import com.moengage.inapp.internal.MODULE_TAG
import com.moengage.inapp.internal.isModuleEnabled
import com.moengage.inapp.internal.trackInAppClicked
import com.moengage.inapp.internal.trackInAppDismissed
import com.moengage.inapp.listeners.InAppLifeCycleListener
import com.moengage.inapp.listeners.OnClickActionListener
import com.moengage.inapp.listeners.SelfHandledAvailableListener
import com.moengage.inapp.model.SelfHandledCampaignData
import com.moengage.inapp.model.enums.InAppPosition

/**
 * Helper class for InApp Related features.
 * @author Umang Chamaria
 */
public class MoEInAppHelper private constructor() {

    private val tag = "${MODULE_TAG}MoEInAppHelper"

    /**
     * Try to show an In-App Message for the account configured as default.
     *
     * @param context instance of [Context]
     * @since 6.0.0
     */
    public fun showInApp(context: Context) {
        val instance = SdkInstanceManager.defaultInstance ?: run {
            Logger.print { "$tag showInApp() : Instance not initialised, cannot process further" }
            return
        }
        showInApp(instance, context)
    }

    /**
     * Try to show an In-App Message for the given account.
     *
     * @param context instance of [Context]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @since 6.0.0
     */
    public fun showInApp(@NonNull context: Context, @NonNull appId: String) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: run {
            Logger.print { "$tag showInApp() : Instance not initialised, cannot process further" }
            return
        }
        showInApp(instance, context)
    }

    private fun showInApp(sdkInstance: SdkInstance, context: Context) {
        InAppInstanceProvider.getControllerForInstance(sdkInstance).showInAppIfPossible(context)
    }

    /**
     * Fetch Self Handled campaign for the account configured as default.
     * @param context Instance of [Context]
     * @param listener Instance of [SelfHandledAvailableListener]
     * @since 6.0.0
     */
    public fun getSelfHandledInApp(
        @NonNull context: Context,
        @NonNull listener: SelfHandledAvailableListener
    ) {
        when (val instance = SdkInstanceManager.defaultInstance) {
            is SdkInstance -> getSelfHandledInApp(instance, context, listener)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag getSelfHandledInApp() : Instance not found" }
                postOnMainThread {
                    listener.onSelfHandledAvailable(null)
                }
            }
        }
    }

    /**
     * Fetch Self Handled campaign for the given account.
     * @param context Instance of [Context]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @param listener Instance of [SelfHandledAvailableListener]
     * @since 6.0.0
     */
    public fun getSelfHandledInApp(
        @NonNull context: Context,
        @NonNull appId: String,
        @NonNull listener: SelfHandledAvailableListener
    ) {
        when (val instance = SdkInstanceManager.getInstanceForAppId(appId)) {
            is SdkInstance -> getSelfHandledInApp(instance, context, listener)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag getSelfHandledInApp() : Instance not found" }
                postOnMainThread {
                    listener.onSelfHandledAvailable(null)
                }
            }
        }
    }

    private fun getSelfHandledInApp(
        sdkInstance: SdkInstance,
        context: Context,
        listener: SelfHandledAvailableListener
    ) {
        sdkInstance.logger.log { "$tag getSelfHandledInApp() : " }
        InAppInstanceProvider.getControllerForInstance(sdkInstance)
            .getSelfHandledInApp(context, listener)
    }

    /**
     * Mark self-handled campaign as shown for the account configured as default.
     *
     * @param context Instance of [Context]
     * @param data Instance of [SelfHandledCampaignData]
     * @since 6.0.0
     */
    public fun selfHandledShown(@NonNull context: Context, @NonNull data: SelfHandledCampaignData) {
        val instance = SdkInstanceManager.defaultInstance ?: return
        selfHandledShown(context, instance, data)
    }

    /**
     * Mark self-handled campaign as shown  for the given account.
     *
     * @param context Instance of [Context]
     * @param data Instance of [SelfHandledCampaignData]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @since 6.0.0
     */
    public fun selfHandledShown(
        @NonNull context: Context,
        @NonNull data: SelfHandledCampaignData,
        @NonNull appId: String
    ) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return
        selfHandledShown(context, instance, data)
    }

    private fun selfHandledShown(
        context: Context,
        sdkInstance: SdkInstance,
        data: SelfHandledCampaignData
    ) {
        if (!isModuleEnabled(context, sdkInstance)) return
        InAppInstanceProvider.getControllerForInstance(sdkInstance).selfHandledShown(context, data)
    }

    /**
     * Mark self-handled campaign as clicked for the account configured as default.
     *
     * @param context Instance of [Context]
     * @param data Instance of [SelfHandledCampaignData]
     * @since 6.0.0
     */
    public fun selfHandledClicked(
        @NonNull context: Context,
        @NonNull data: SelfHandledCampaignData
    ) {
        selfHandledClicked(context, data, -1)
    }

    /**
     * Mark self-handled campaign as clicked for the given instance.
     *
     * @param context Instance of [Context]
     * @param data Instance of [SelfHandledCampaignData]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @since 6.0.0
     */
    public fun selfHandledClicked(
        @NonNull context: Context,
        @NonNull data: SelfHandledCampaignData,
        @NonNull appId: String
    ) {
        selfHandledClicked(context, data, -1, appId)
    }

    /**
     * Mark self-handled campaign as clicked with a given id for the account configured as default.
     *
     * @param context Instance of [Context]
     * @param data Instance of [SelfHandledCampaignData]
     * @param widgetId Id for the widget that is clicked.
     * @since 6.0.0
     */
    public fun selfHandledClicked(
        @NonNull context: Context,
        @NonNull data: SelfHandledCampaignData,
        @NonNull widgetId: Int
    ) {
        val instance = SdkInstanceManager.defaultInstance ?: return
        selfHandledClicked(context, instance, data, widgetId)
    }

    /**
     * Mark self-handled campaign as clicked with a given id for the given instance.
     *
     * @param context Instance of [Context]
     * @param data Instance of [SelfHandledCampaignData]
     * @param widgetId Id for the widget that is clicked.
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @since 6.0.0
     */
    public fun selfHandledClicked(
        @NonNull context: Context,
        @NonNull data: SelfHandledCampaignData,
        widgetId: Int,
        @NonNull appId: String
    ) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return
        selfHandledClicked(context, instance, data, widgetId)
    }

    private fun selfHandledClicked(
        context: Context,
        sdkInstance: SdkInstance,
        data: SelfHandledCampaignData,
        widgetId: Int
    ) {
        if (!isModuleEnabled(context, sdkInstance)) return
        trackInAppClicked(context, sdkInstance, data.campaignData, widgetId)
    }

    /**
     * Mark self handled campaign as dismissed for the account configured as default.
     * @param context instance of [Context]
     * @param data instance of [SelfHandledCampaignData]
     * @since 6.0.0
     */
    public fun selfHandledDismissed(
        @NonNull context: Context,
        @NonNull data: SelfHandledCampaignData
    ) {
        val instance = SdkInstanceManager.defaultInstance ?: return
        selfHandledDismissed(context, instance, data)
    }

    /**
     * Mark self handled campaign as dismissed for the given account.
     * @param context instance of [Context]
     * @param data instance of [SelfHandledCampaignData]
     * @since 6.0.0
     */
    public fun selfHandledDismissed(
        @NonNull context: Context,
        @NonNull data: SelfHandledCampaignData,
        @NonNull appId: String
    ) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return
        selfHandledDismissed(context, instance, data)
    }

    private fun selfHandledDismissed(
        context: Context,
        sdkInstance: SdkInstance,
        data: SelfHandledCampaignData
    ) {
        try {
            if (!isModuleEnabled(context, sdkInstance)) return
            trackInAppDismissed(context, sdkInstance, data.campaignData)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag selfHandledDismissed() : " }
        }
    }

    /**
     * Register a callback to be notified whenever in-app is shown, closed. This API registers
     * callback for the account configured as default.
     *
     * @param listener instance of [InAppLifeCycleListener]
     * @since 5.2.0
     */
    public fun addInAppLifeCycleListener(@NonNull listener: InAppLifeCycleListener) {
        val instance = SdkInstanceManager.defaultInstance ?: return
        addInAppLifeCycleListener(instance, listener)
    }

    /**
     * Register a callback to be notified whenever in-app is shown, closed. This API registers
     * callback for the given account.
     *
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @param listener instance of [InAppLifeCycleListener]
     * @since 5.2.0
     */
    public fun addInAppLifeCycleListener(appId: String, @NonNull listener: InAppLifeCycleListener) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return
        addInAppLifeCycleListener(instance, listener)
    }

    private fun addInAppLifeCycleListener(
        sdkInstance: SdkInstance,
        listener: InAppLifeCycleListener
    ) {
        InAppInstanceProvider.getCacheForInstance(sdkInstance).lifeCycleListeners.add(listener)
    }

    /***
     * Notify SDK when screen orientation changes, for SDK to handle in-app display.
     * NOTE: Use this API only when your Activity handles the screen orientation change by
     * itself.
     * @see <a href="https://developer.android
     * .com/guide/topics/resources/runtime-changes#HandlingTheChange">Android doc link for
     * handling the configuration change </a>
     * @since 5.2.0
     */
    public fun onConfigurationChanged() {
        ConfigurationChangeHandler.getInstance().onConfigurationChanged(true)
    }

    public companion object {

        private var instance: MoEInAppHelper? = null

        @JvmStatic
        public fun getInstance(): MoEInAppHelper {
            return instance ?: synchronized(MoEInAppHelper::class.java) {
                val inst = instance ?: MoEInAppHelper()
                instance = inst
                inst
            }
        }
    }

    /**
     * Set the user context in which In-App should be shown for the account configured as default.
     * @param contexts [Set] of context
     * @since 6.0.0
     */
    public fun setInAppContext(contexts: Set<String>) {
        val instance = SdkInstanceManager.defaultInstance ?: return
        setInAppContext(instance, contexts)
    }

    /**
     * Set the user context in which In-App should be shown for the given account.
     * @param contexts [Set] of context
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @since 6.0.0
     */
    public fun setInAppContext(contexts: Set<String>, appId: String) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return
        setInAppContext(instance, contexts)
    }

    private fun setInAppContext(sdkInstance: SdkInstance, contexts: Set<String>) {
        InAppInstanceProvider.getCacheForInstance(sdkInstance).inAppContext = contexts
    }

    /**
     * Resets the user context set for inapp. This API resets the context for the account
     * configured as default.
     * @since 6.0.0
     */
    public fun resetInAppContext() {
        val instance = SdkInstanceManager.defaultInstance ?: return
        resetInAppContext(instance)
    }

    /**
     * Resets the user context set for inapp. This API resets the context for the given account.
     * @since 6.0.0
     */
    public fun resetInAppContext(appId: String) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return
        resetInAppContext(instance)
    }

    private fun resetInAppContext(sdkInstance: SdkInstance) {
        InAppInstanceProvider.getCacheForInstance(sdkInstance).inAppContext = emptySet()
    }

    /**
     * Set a listener for in-app click,listener is only called if the action is of type Navigation.
     * This API is sets a listener for the account configured as default.
     * @param listener Instance of [OnClickActionListener]
     * @since 6.0.0
     */
    public fun setClickActionListener(listener: OnClickActionListener?) {
        val instance = SdkInstanceManager.defaultInstance ?: return
        setClickActionListener(instance, listener)
    }

    /**
     * Set a listener for in-app click,listener is only called if the action is of type Navigation.
     * This API is sets a listener for the given account.
     * @param listener Instance of [OnClickActionListener]
     * @since 6.0.0
     */
    public fun setClickActionListener(appId: String, listener: OnClickActionListener?) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return
        setClickActionListener(instance, listener)
    }

    private fun setClickActionListener(sdkInstance: SdkInstance, listener: OnClickActionListener?) {
        InAppInstanceProvider.getCacheForInstance(sdkInstance).clickActionListener = listener
    }

    /**
     * Set a listener for receiving callback for event triggered self-handled campaign.
     * This API is sets a listener for the account configured as default.
     *
     * **Note:** Make sure the listener is set for the event is tracked.
     *
     * @param listener Instance of [SelfHandledAvailableListener]
     * @since 6.0.0
     */
    public fun setSelfHandledListener(listener: SelfHandledAvailableListener?) {
        val instance = SdkInstanceManager.defaultInstance ?: return
        setSelfHandledListener(instance, listener)
    }

    /**
     * Set a listener for receiving callback for event triggered self-handled campaign.
     * This API is sets a listener for the given account.
     *
     * **Note:** Make sure the listener is set for the event is tracked.
     *
     * @param listener Instance of [SelfHandledAvailableListener]
     * @since 6.0.0
     */
    public fun setSelfHandledListener(appId: String, listener: SelfHandledAvailableListener?) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return
        setSelfHandledListener(instance, listener)
    }

    private fun setSelfHandledListener(
        sdkInstance: SdkInstance,
        listener: SelfHandledAvailableListener?
    ) {
        sdkInstance.logger.log { "$tag setSelfHandledListener() : Setting self handled listener: $listener" }
        InAppInstanceProvider.getCacheForInstance(sdkInstance).selfHandledListener = listener
    }

    /**
     * Remove a registered listener for the account configured as default.
     *
     * @param listener instance of [InAppLifeCycleListener]
     * @since 6.0.0
     */
    public fun removeInAppLifeCycleListener(@NonNull listener: InAppLifeCycleListener) {
        val instance = SdkInstanceManager.defaultInstance ?: return
        removeInAppListener(instance, listener)
    }

    /**
     * Remove a registered listener for the given account.
     *
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @param listener instance of [InAppLifeCycleListener]
     * @since 6.0.0
     */
    public fun removeInAppLifeCycleListener(
        @NonNull appId: String,
        @NonNull listener: InAppLifeCycleListener
    ) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return
        removeInAppListener(instance, listener)
    }

    private fun removeInAppListener(sdkInstance: SdkInstance, listener: InAppLifeCycleListener) {
        sdkInstance.logger.log { "$tag removeInAppListener() : Removing in-app listener: $listener" }
        InAppInstanceProvider.getCacheForInstance(sdkInstance).lifeCycleListeners.remove(listener)
    }

    /**
     * API to enable Activity registration in [android.app.Activity.onResume]
     *
     * Use this API only when you have transparent or translucent activities.
     * If you are calling this API make sure that the [showInApp] is called in
     * [android.app.Activity.onResume] instead of [android.app.Activity.onStart]
     *
     * @since 6.2.0
     */
    public fun enableActivityRegistrationOnResume() {
        InAppModuleManager.enableActivityRegistrationOnResume()
    }

    /**
     * API to disable Activity registration in [android.app.Activity.onResume]
     *
     * Call this API only when [enableActivityRegistrationOnResume] is called. By default
     * `ActivityRegistrationOnResume`is disabled.
     *
     * @since 6.2.0
     */
    public fun disableActivityRegistrationOnResume() {
        InAppModuleManager.disableActivityRegistrationOnResume()
    }

    /**
     * Try to show a non-intrusive In-App nudge for the given account.
     *
     * @param context instance of [Context]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @since 7.0.0
     */
    @JvmOverloads
    public fun showNudge(@NonNull context: Context, appId: String? = null) {
        val instance = getSdkInstance(appId) ?: run {
            Logger.print { "$tag showNudge() : Instance not initialised, cannot process further" }
            return
        }
        showNudge(instance, context)
    }

    /**
     * Try to show a non-intrusive In-App nudge for the given account.
     *
     * @param context instance of [Context]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @param inAppPosition instance of [InAppPosition]
     * @since 7.0.0
     */
    @JvmOverloads
    public fun showNudge(
        @NonNull context: Context,
        inAppPosition: InAppPosition,
        appId: String? = null
    ) {
        val instance = getSdkInstance(appId) ?: run {
            Logger.print { "$tag showNudge() : Instance not initialised, cannot process further" }
            return
        }
        showNudge(instance, context, inAppPosition)
    }

    /**
     * @since 7.0.0
     */
    private fun showNudge(
        sdkInstance: SdkInstance,
        context: Context,
        inAppPosition: InAppPosition = InAppPosition.ANY
    ) {
        InAppInstanceProvider.getControllerForInstance(sdkInstance)
            .showNudgeIfPossible(context, inAppPosition)
    }
}