/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal

import android.app.Activity
import android.content.Intent
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.view.View
import android.widget.RatingBar
import android.widget.TextView
import com.moe.pushlibrary.activities.MoEActivity
import com.moengage.core.LogLevel
import com.moengage.core.PUSH_NOTIFICATION_NAVIGATION_DEEPLINK_LEGACY
import com.moengage.core.Properties
import com.moengage.core.analytics.MoEAnalyticsHelper
import com.moengage.core.internal.CoreInternalHelper
import com.moengage.core.internal.EVENT_ATTRIBUTE_FLOW
import com.moengage.core.internal.EXTRA_IS_EMBEDDED_WEB_VIEW
import com.moengage.core.internal.MAX_REQUEST_NOTIFICATION_PERMISSION_ATTEMPT
import com.moengage.core.internal.MOE_CAMPAIGN_NAME
import com.moengage.core.internal.actions.ActionManagerBase
import com.moengage.core.internal.global.GlobalResources
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.accountMetaForInstance
import com.moengage.core.internal.utils.buildEncodedDeepLinkUriFromString
import com.moengage.core.internal.utils.buildUriFromString
import com.moengage.core.internal.utils.canUseWebView
import com.moengage.core.internal.utils.copyTextToClipboardAndShowToast
import com.moengage.evaluator.ConditionEvaluator
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.actions.CallAction
import com.moengage.inapp.internal.model.actions.Condition
import com.moengage.inapp.internal.model.actions.ConditionAction
import com.moengage.inapp.internal.model.actions.CopyAction
import com.moengage.inapp.internal.model.actions.NavigateToSettingsAction
import com.moengage.inapp.internal.model.actions.RatingChangeAction
import com.moengage.inapp.internal.model.actions.SetTextAction
import com.moengage.inapp.internal.model.actions.ShareAction
import com.moengage.inapp.internal.model.actions.SmsAction
import com.moengage.inapp.internal.model.actions.TrackAction
import com.moengage.inapp.internal.model.actions.UserInputAction
import com.moengage.inapp.internal.model.enums.DataTrackType
import com.moengage.inapp.internal.model.enums.UserInputType
import com.moengage.inapp.model.CampaignData
import com.moengage.inapp.model.ClickData
import com.moengage.inapp.model.InAppBaseData
import com.moengage.inapp.model.actions.Action
import com.moengage.inapp.model.actions.CustomAction
import com.moengage.inapp.model.actions.NavigationAction
import com.moengage.inapp.model.actions.RequestNotificationAction
import com.moengage.inapp.model.enums.ActionType
import com.moengage.inapp.model.enums.NavigationType
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject

/**
 * @author Umang Chamaria
 */
internal class ActionHandler(private val context: Activity, private val sdkInstance: SdkInstance) :
    ActionManagerBase() {

    private val tag = "${MODULE_TAG}ActionHandler"

    fun onActionPerformed(inAppView: View, action: Action, payload: CampaignPayload) {
        try {
            when (action.actionType) {
                ActionType.DISMISS -> dismissAction(action, inAppView, payload)
                ActionType.TRACK_DATA -> trackAction(action, payload.campaignId)
                ActionType.NAVIGATE -> navigateAction(action, payload)
                ActionType.SHARE -> shareAction(action, payload.campaignId)
                ActionType.COPY_TEXT -> copyAction(action, payload.campaignId)
                ActionType.CALL -> callAction(action, payload.campaignId)
                ActionType.SMS -> smsAction(action, payload.campaignId)
                ActionType.CUSTOM_ACTION -> customAction(action, payload)
                ActionType.CONDITION_ACTION -> conditionAction(inAppView, action, payload)
                ActionType.USER_INPUT -> userInputAction(inAppView, action, payload)
                ActionType.REQUEST_NOTIFICATION_PERMISSION -> requestNotificationPermissionAction(
                    action,
                    payload
                )

                ActionType.NAVIGATE_SETTINGS_NOTIFICATIONS -> navigateToNotificationSettingsAction(
                    action,
                    payload
                )

                ActionType.RATING_CHANGE -> ratingChangeAction(inAppView, action, payload)
                ActionType.SET_TEXT -> setTextAction(inAppView, action, payload)
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag onActionPerformed() : " }
        }
    }

    private fun smsAction(action: Action, campaignId: String) {
        sdkInstance.logger.log { "$tag smsAction() : will try to trigger sms intent" }
        if (action !is SmsAction) {
            sdkInstance.logger.log { "$tag smsAction() : Not a valid sms action. $campaignId" }
            return
        }
        sdkInstance.logger.log { "$tag smsAction() : Sms Action: $action" }
        if (action.phoneNumber.isBlank() || action.message.isBlank()) {
            sdkInstance.logger.log(LogLevel.ERROR) { "$tag smsAction() : Number or message is null, $campaignId" }
            return
        }
        val uri = Uri.parse("smsto:" + action.phoneNumber)
        val intent = Intent(Intent.ACTION_SENDTO, uri)
        intent.putExtra("sms_body", action.message)
        context.startActivity(intent)
    }

    private fun callAction(action: Action, campaignId: String) {
        sdkInstance.logger.log { "$tag callAction() : Will try to trigger call intent" }
        if (action !is CallAction) {
            sdkInstance.logger.log { "$tag callAction() : Not a valid call action. $campaignId" }
            return
        }
        sdkInstance.logger.log { "$tag callAction() : $action" }
        if (action.phoneNumber.isBlank() || !isPhoneNumberValid(action.phoneNumber)) {
            sdkInstance.logger.log { "$tag callAction() : Empty/Invalid number. $campaignId" }
            return
        }
        triggerCallIntent(context, action.phoneNumber)
    }

    private fun shareAction(action: Action, campaignId: String) {
        sdkInstance.logger.log { "$tag shareAction() : Will try to share text" }
        if (action !is ShareAction) {
            sdkInstance.logger.log { "$tag shareAction() : Not a valid share action. $campaignId" }
            return
        }
        sdkInstance.logger.log { "$tag shareAction() : $action" }
        if (action.shareText.isBlank()) {
            sdkInstance.logger.log(LogLevel.ERROR) { "$tag shareAction() : Text empty, aborting. $campaignId" }
            return
        }
        triggerShareIntent(context, action.shareText)
    }

    private fun customAction(action: Action, payload: CampaignPayload) {
        if (action !is CustomAction) {
            sdkInstance.logger.log(LogLevel.ERROR) { "$tag customAction() : Not a custom Action, ${payload.campaignId}" }
            return
        }
        val listener =
            InAppInstanceProvider.getCacheForInstance(sdkInstance).clickActionListener ?: return
        val data = ClickData(
            InAppBaseData(
                CampaignData(payload.campaignId, payload.campaignName, payload.campaignContext),
                accountMetaForInstance(sdkInstance)
            ),
            action
        )
        GlobalResources.mainThread.post {
            try {
                listener.onClick(data)
            } catch (t: Throwable) {
                sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag customAction() : " }
            }
        }
    }

    private fun trackAction(action: Action, campaignId: String) {
        sdkInstance.logger.log { "$tag trackAction() : " }
        if (action !is TrackAction) {
            sdkInstance.logger.log { "$tag trackAction() : Not a valid track action. $campaignId" }
            return
        }
        when (action.trackType) {
            DataTrackType.EVENT -> trackEvent(action, campaignId)
            DataTrackType.USER_ATTRIBUTE -> trackUserAttribute(action, campaignId)
        }
    }

    private fun trackEvent(action: TrackAction, campaignId: String) {
        sdkInstance.logger.log { "$tag trackEvent() : " }
        if (action.name.isBlank()) {
            sdkInstance.logger.log { "$tag trackEvent() : Event name is blank, cannot track. $campaignId" }
            return
        }
        val properties = Properties()
        if (action.attributes != null) {
            for ((key, value) in action.attributes) {
                properties.addAttribute(key, value)
            }
        }
        MoEAnalyticsHelper.trackEvent(
            context,
            action.name.trim(),
            properties,
            sdkInstance.instanceMeta.instanceId
        )
    }

    private fun trackUserAttribute(action: TrackAction, campaignId: String) {
        sdkInstance.logger.log { "$tag trackUserAttribute() : " }
        if (action.name.isBlank()) {
            sdkInstance.logger.log { "$tag trackUserAttribute() : Attribute name is blank, cannot track, $campaignId" }
            return
        }
        MoEAnalyticsHelper.setUserAttribute(
            context,
            action.name.trim(),
            action.value,
            sdkInstance.instanceMeta.instanceId
        )
    }

    private fun copyAction(action: Action, campaignId: String) {
        sdkInstance.logger.log { "$tag copyAction() : " }
        if (action !is CopyAction) {
            sdkInstance.logger.log(LogLevel.ERROR) { "$tag copyAction() : Not a valid copy action, $campaignId" }
            return
        }
        sdkInstance.logger.log { "$tag copyAction() : $action" }
        if (action.textToCopy.isBlank()) {
            sdkInstance.logger.log(LogLevel.ERROR) { "$tag copyAction() : Text to copy is blank, aborting $campaignId" }
            return
        }
        copyTextToClipboardAndShowToast(context, action.textToCopy, action.message ?: "")
    }

    private fun dismissAction(action: Action, inAppView: View, campaignPayload: CampaignPayload) {
        sdkInstance.logger.log { "$tag dismissAction() : " }
        InAppInstanceProvider.getControllerForInstance(sdkInstance).viewHandler.apply {
            removeViewFromHierarchy(context.applicationContext, inAppView, campaignPayload)
            handleDismiss(campaignPayload)
        }
    }

    private fun navigateAction(action: Action, payload: CampaignPayload) {
        sdkInstance.logger.log { "$tag navigateAction() : " }
        if (action !is NavigationAction) {
            sdkInstance.logger.log(LogLevel.ERROR) { "$tag navigateAction() : Not a navigation action, ${payload.campaignId}" }
            return
        }
        sdkInstance.logger.log { "$tag navigateAction() : $action" }
        val clickListener =
            InAppInstanceProvider.getCacheForInstance(sdkInstance).clickActionListener
        val data = ClickData(
            InAppBaseData(
                CampaignData(payload.campaignId, payload.campaignName, payload.campaignContext),
                accountMetaForInstance(sdkInstance)
            ),
            action
        )
        if (clickListener != null && action.navigationType != NavigationType.RICH_LANDING &&
            clickListener.onClick(
                    data
                )
        ) {
            sdkInstance.logger.log { "$tag navigateAction() : Navigation handled by client." }
            return
        }
        val intent = when (action.navigationType) {
            NavigationType.SCREEN -> {
                val redirectIntent = Intent(context, Class.forName(action.navigationUrl))
                val extras = Bundle()
                if (action.keyValuePairs != null) {
                    for ((key, value) in action.keyValuePairs.entries) {
                        extras.putString(key, value.toString())
                    }
                    if (!extras.isEmpty) {
                        redirectIntent.putExtras(extras)
                    }
                }
                redirectIntent
            }

            NavigationType.DEEP_LINKING -> {
                Intent(
                    Intent.ACTION_VIEW,
                    buildEncodedDeepLinkUriFromString(
                        action.navigationUrl,
                        action.keyValuePairs ?: emptyMap()
                    )
                )
            }

            NavigationType.RICH_LANDING -> {
                if (canUseWebView(context)) {
                    val redirectIntent = Intent(context, MoEActivity::class.java)
                    redirectIntent.putExtra(
                        PUSH_NOTIFICATION_NAVIGATION_DEEPLINK_LEGACY,
                        buildUriFromString(
                            action.navigationUrl,
                            action.keyValuePairs ?: emptyMap()
                        ).toString()
                    )
                    redirectIntent.putExtra(EXTRA_IS_EMBEDDED_WEB_VIEW, true)
                    redirectIntent
                } else {
                    sdkInstance.logger.log { "$tag navigateAction() : Web View Disabled." }
                    null
                }
            }
        }
        intent?.let {
            context.startActivity(it)
        }
    }

    private fun userInputAction(inAppView: View, action: Action, payload: CampaignPayload) {
        sdkInstance.logger.log { "$tag userInputAction() : " }
        if (action !is UserInputAction) {
            sdkInstance.logger.log(LogLevel.ERROR) { "$tag userInputAction() : Not a valid user input action, ${payload.campaignId}" }
            return
        }
        sdkInstance.logger.log { "$tag userInputAction() : User input action: $action" }
        when (action.userInputType) {
            UserInputType.CUSTOM_RATING,
            UserInputType.RATING -> {
                val view =
                    inAppView.findViewById<View>(InAppConstants.WIDGET_BASE_ID + action.widgetId)
                if (view == null) {
                    sdkInstance.logger.log(LogLevel.ERROR) { "$tag userInputAction() : Did not find widget for id" }
                    return
                }
                if (view !is RatingBar) {
                    sdkInstance.logger.log(LogLevel.ERROR) { "$tag userInputAction() : given view is not rating, aborting, ${payload.campaignId}" }
                    return
                }
                val ratingInput = view.rating
                for (actionItem in action.actions) {
                    if (actionItem.actionType === ActionType.TRACK_DATA) {
                        val trackAction = actionItem as TrackAction
                        when (trackAction.trackType) {
                            DataTrackType.EVENT -> {
                                trackAction.attributes[InAppConstants.IN_APP_RATING_ATTRIBUTE] =
                                    ratingInput
                                trackEvent(trackAction, payload.campaignId)
                            }

                            DataTrackType.USER_ATTRIBUTE ->
                                MoEAnalyticsHelper
                                    .setUserAttribute(
                                        context,
                                        trackAction.name.trim(),
                                        ratingInput,
                                        sdkInstance.instanceMeta.instanceId
                                    )
                        }
                    } else {
                        onActionPerformed(inAppView, actionItem, payload)
                    }
                }
            }
        }
    }

    private fun conditionAction(inAppView: View, action: Action, payload: CampaignPayload) {
        try {
            sdkInstance.logger.log { "$tag conditionAction() : " }
            if (action !is ConditionAction) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag conditionAction() : Not a valid condition action, ${payload.campaignId}" }
                return
            }
            sdkInstance.logger.log { "$tag conditionAction() : Condition Action: $action" }
            val view = inAppView.findViewById<View>(InAppConstants.WIDGET_BASE_ID + action.widgetId)
            if (view == null) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag conditionAction() : Did not find view with id, ${payload.campaignId}" }
                return
            }
            if (view !is RatingBar) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag conditionAction() : Given view is not a rating widget, ${payload.campaignId}" }
                return
            }
            val ratingInput = view.rating
            val userInput = JSONObject()
            userInput.put(InAppConstants.IN_APP_RATING_ATTRIBUTE, ratingInput.toDouble())
            for (condition: Condition in action.conditions) {
                val evaluator = ConditionEvaluator(
                    transformConditionAttributeForPackage(condition.conditionAttribute),
                    userInput
                )
                if (evaluator.evaluate()) {
                    for (actionItem: Action in condition.actions) {
                        onActionPerformed(inAppView, actionItem, payload)
                    }
                }
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag conditionAction() : " }
        }
    }

    @Throws(JSONException::class)
    private fun transformConditionAttributeForPackage(conditionAttribute: JSONObject): JSONObject {
        val attributes = JSONObject()
        attributes.put("filter_operator", "and")
        val filters = JSONArray()
        filters.put(conditionAttribute)
        attributes.put("filters", filters)
        return attributes
    }

    private fun requestNotificationPermissionAction(action: Action, payload: CampaignPayload) {
        try {
            sdkInstance.logger.log { "$tag requestNotificationPermissionAction() : " }
            if (action !is RequestNotificationAction) {
                sdkInstance.logger.log(LogLevel.ERROR) {
                    "$tag requestNotificationPermissionAction() : Not a " +
                        "RequestNotificationAction, ${payload.campaignId}"
                }
                return
            }
            val requestCount = InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance)
                .getPushPermissionRequestCount()

            val clickListener =
                InAppInstanceProvider.getCacheForInstance(sdkInstance).clickActionListener

            if (clickListener != null) {
                val data = ClickData(
                    InAppBaseData(
                        CampaignData(
                            payload.campaignId,
                            payload.campaignName,
                            payload.campaignContext
                        ),
                        accountMetaForInstance(sdkInstance)
                    ),
                    RequestNotificationAction(action.actionType, requestCount)
                )
                if (clickListener.onClick(data)) {
                    sdkInstance.logger.log {
                        "$tag requestNotificationPermissionAction() : Request " +
                            "Notification handled by client."
                    }
                    return
                }
            }

            if (Build.VERSION.SDK_INT < 33) {
                sdkInstance.logger.log {
                    "$tag requestNotificationPermissionAction() : SDK version is < 33, " +
                        "redirecting to Notification Settings page."
                }
                CoreInternalHelper.navigateToNotificationSettings(context)
                return
            }

            if (requestCount >= MAX_REQUEST_NOTIFICATION_PERMISSION_ATTEMPT) {
                sdkInstance.logger.log {
                    "$tag requestNotificationPermissionAction() : requestCount: " +
                        " $requestCount >= $MAX_REQUEST_NOTIFICATION_PERMISSION_ATTEMPT, redirecting user to Notification Settings page."
                }
                CoreInternalHelper.navigateToNotificationSettings(context)
            } else {
                val dataPayload = mapOf(
                    MOE_CAMPAIGN_NAME to payload.campaignName,
                    EVENT_ATTRIBUTE_FLOW to EVENT_ATTRIBUTE_FLOW_TWO_STEP_OPT_IN
                )
                CoreInternalHelper.requestNotificationPermission(context, dataPayload)
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(
                LogLevel.ERROR,
                t
            ) { "$tag requestNotificationPermissionAction() : " }
        }
    }

    private fun navigateToNotificationSettingsAction(action: Action, payload: CampaignPayload) {
        try {
            sdkInstance.logger.log { "$tag navigateToNotificationSettingsAction() : " }
            if (action !is NavigateToSettingsAction) {
                sdkInstance.logger.log(LogLevel.ERROR) {
                    "$tag navigateToNotificationSettingsAction() : Not a " +
                        "NavigateToSettingsAction, ${payload.campaignId}"
                }
                return
            }
            CoreInternalHelper.navigateToNotificationSettings(context)
        } catch (t: Throwable) {
            sdkInstance.logger.log(
                LogLevel.ERROR,
                t
            ) { "$tag navigateToNotificationSettingsAction() : " }
        }
    }

    /**
     * Handles Set text action.
     * Sets text to a text view
     */
    private fun setTextAction(inAppView: View, action: Action, payload: CampaignPayload) {
        try {
            sdkInstance.logger.log { "$tag setTextAction() : " }
            if (action !is SetTextAction) {
                sdkInstance.logger.log(LogLevel.ERROR) {
                    "$tag setTextAction() : Not a SetTextAction, ${payload.campaignId}"
                }
                return
            }
            val view =
                inAppView.findViewById<View>(InAppConstants.WIDGET_BASE_ID + action.widgetId) as TextView
            view.text = action.content
            view.visibility = View.VISIBLE
        } catch (t: Throwable) {
            sdkInstance.logger.log(
                LogLevel.ERROR,
                t
            ) { "$tag setTextAction() : " }
        }
    }

    /**
     * Handles Rating Change Action
     */
    private fun ratingChangeAction(inAppView: View, action: Action, payload: CampaignPayload) {
        try {
            sdkInstance.logger.log { "$tag ratingChangeAction() : " }
            if (action !is RatingChangeAction) {
                sdkInstance.logger.log(LogLevel.ERROR) {
                    "$tag ratingChangeAction() : Not a RatingChangeAction, ${payload.campaignId}"
                }
                return
            }
            for (subAction in action.actions) {
                onActionPerformed(inAppView, subAction, payload)
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(
                LogLevel.ERROR,
                t
            ) { "$tag ratingChangeAction() : " }
        }
    }
}