/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal

import android.app.Activity
import com.moengage.core.LogLevel
import com.moengage.core.internal.SdkInstanceManager
import com.moengage.core.internal.logger.Logger
import com.moengage.core.internal.model.SdkInstance
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.NativeCampaignPayload
import com.moengage.inapp.internal.model.configmeta.ConfigChangeMeta
import com.moengage.inapp.internal.model.configmeta.HtmlInAppConfigMeta
import com.moengage.inapp.internal.model.configmeta.InAppConfigMeta
import com.moengage.inapp.internal.model.configmeta.NudgeConfigMeta
import com.moengage.inapp.internal.model.enums.InAppType

/**
 * Class to handle in-apps on device configuration change
 *
 * @author Arshiya Khanum
 */
internal class ConfigurationChangeHandler private constructor() {

    private val tag = "${InAppConstants.MODULE_TAG}ConfigurationChangeHandler"

    private val configChangeMeta = ConfigChangeMeta()

    internal companion object {

        private var instance: ConfigurationChangeHandler? = null

        @JvmStatic
        fun getInstance(): ConfigurationChangeHandler {
            return instance ?: synchronized(ConfigurationChangeHandler::class.java) {
                val inst = instance ?: ConfigurationChangeHandler()
                instance = inst
                inst
            }
        }
    }

    /**
     * Handle Dismissal/ReRendering of InApp on Configuration Change
     * @param shouldDismissInApp true if InApp Should be dismissed , otherwise false
     */
    internal fun onConfigurationChanged(shouldDismissInApp: Boolean) {
        Logger.print { "$tag onConfigurationChanged() : $shouldDismissInApp " }
        val currentActivity = InAppModuleManager.getActivity() ?: return
        handleInAppsOnOrientationChange(currentActivity, shouldDismissInApp)
        updateActivityData(currentActivity)
    }

    /**
     * Checks and returns true if the last shown in-app screen is same but the orientation is
     * different, else false
     */
    private fun hasOrientationChanged(activity: Activity): Boolean =
        activity.javaClass.name == configChangeMeta.activityName &&
            configChangeMeta.activityOrientation != activity.resources.configuration.orientation

    /**
     * Updates current [Activity]'s name and screen orientation.
     */
    private fun updateActivityData(activity: Activity) {
        try {
            val currentActivityName = activity.javaClass.name
            if (currentActivityName != configChangeMeta.activityName) {
                configChangeMeta.activityName = currentActivityName
            }
            configChangeMeta.activityOrientation = activity.resources.configuration.orientation

            Logger.print {
                "$tag updateActivityData() : configChangeMeta: " +
                    "configChangeMeta:[${configChangeMeta.activityName}, ${configChangeMeta.activityOrientation}]"
            }
        } catch (t: Throwable) {
            Logger.print(
                LogLevel.ERROR,
                t
            ) { "$tag updateActivityData() : exception encountered, resetting data" }
            clearLastSavedCampaignData()
        }
    }

    /**
     * Saves last shown in-app's data.
     * @param campaignPayload instance of [CampaignPayload]
     * @param sdkInstance instance of [SdkInstance]
     */
    internal fun saveLastInAppShownData(
        campaignPayload: CampaignPayload,
        sdkInstance: SdkInstance
    ) {
        try {
            sdkInstance.logger.log {
                "$tag saveLastInAppShownData() : Saving last in-app shown " +
                    "data: ${campaignPayload.campaignId} ${campaignPayload.inAppType.name}"
            }
            val inAppConfigMeta =
                getInAppConfigMetaFromPayload(campaignPayload, sdkInstance) ?: return

            if (inAppConfigMeta is NudgeConfigMeta) {
                // NOTE: We need to save last shown InApp Data for Nudges in [List] because
                // multiple nudges can be shown at same time
                configChangeMeta.lastShownNudges.add(inAppConfigMeta)
            } else {
                // NOTE: We need to save last shown General InApp Data for Handling Orientation Change
                configChangeMeta.lastShownGeneralCampaign = inAppConfigMeta
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(
                LogLevel.ERROR,
                t
            ) { "$tag saveLastInAppShownData() : resetting" }
            clearLastSavedCampaignData()
        }
    }

    /**
     * Clears saved data.
     */
    private fun clearLastSavedCampaignData() {
        configChangeMeta.lastShownGeneralCampaign = null
        configChangeMeta.lastShownNudges.clear()
    }

    /**
     * Re-renders last shown in-app
     * @param activity instance of [Activity]
     * @param sdkInstance instance of [SdkInstance]
     */
    internal fun showInAppOnConfigurationChange(activity: Activity, sdkInstance: SdkInstance) {
        sdkInstance.logger.log {
            "$tag showInAppOnConfigurationChange() : Will try to show in-app," +
                " ${configChangeMeta.lastShownGeneralCampaign}"
        }
        try {
            val inAppMeta =
                configChangeMeta.lastShownGeneralCampaign ?: return
            InAppInstanceProvider.getControllerForInstance(sdkInstance).viewHandler
                .removeAutoDismissRunnable(inAppMeta.campaignId)
            if (!canShowInAppInCurrentOrientation(
                    configChangeMeta.activityOrientation,
                    inAppMeta.supportedOrientations
                )
            ) {
                sdkInstance.logger.log {
                    "$tag showInAppOnConfigurationChange() : ${inAppMeta.campaignId} is not " +
                        "supported in current orientation."
                }
                InAppModuleManager.updateInAppVisibility(false)
                clearGeneralInAppFromConfigCache()
                return
            }
            if (inAppMeta !is HtmlInAppConfigMeta) return
            val inAppView =
                InAppInstanceProvider.getControllerForInstance(sdkInstance).viewHandler.buildInApp(
                    inAppMeta.campaignPayload,
                    getViewCreationMeta(activity.applicationContext)
                )

            if (inAppView != null && activity.javaClass.name == InAppModuleManager.getCurrentActivityName()) {
                InAppInstanceProvider.getControllerForInstance(sdkInstance).viewHandler.addInAppToViewHierarchy(
                    activity,
                    inAppView,
                    inAppMeta.campaignPayload,
                    true
                )
            } else {
                InAppModuleManager.updateInAppVisibility(false)
                clearGeneralInAppFromConfigCache()
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag showInAppOnConfigurationChange() : " }
        }
    }

    /**
     * Clear InApp Config Cache Data
     */
    internal fun clearData() {
        configChangeMeta.apply {
            activityName = null
            activityOrientation = -1
            lastShownGeneralCampaign = null
            lastShownNudges.clear()
        }
    }

    /**
     * Clear Intrusive General InApp From Config Cache when InApp is dismissed
     */
    internal fun clearGeneralInAppFromConfigCache() {
        Logger.print {
            "$tag clearGeneralInAppFromConfigCache(): Removing General InApp From Config Cache"
        }
        configChangeMeta.lastShownGeneralCampaign = null
    }

    /**
     * Clear Nudge InApp From Config Cache when InApp is dismissed
     * @param campaignPayload
     */
    internal fun removeNudgeInAppFromConfigCache(campaignPayload: CampaignPayload) {
        try {
            Logger.print {
                "$tag clearNudgeInAppConfigCache(): Removing InApp, ${campaignPayload.campaignId}"
            }
            val nudgeMeta = configChangeMeta.lastShownNudges.firstOrNull {
                it.campaignId == campaignPayload.campaignId
            } ?: return
            configChangeMeta.lastShownNudges.remove(nudgeMeta)
        } catch (t: Throwable) {
            Logger.print(LogLevel.ERROR, t) { "$tag clearNudgeInAppConfigCache():" }
        }
    }

    /**
     * Dismiss All Non Intrusive Nudge InApps from screen when screen orientation is changed.
     */
    private fun dismissNudgeCampaignsIfRequired(shouldDismissInApp: Boolean) {
        Logger.print {
            "$tag dismissNudgeCampaigns() : Dismissing Nudge InApp campaigns & Clearing Cache," +
                "shouldDismissInApp: $shouldDismissInApp "
        }
        if (shouldDismissInApp) {
            configChangeMeta.lastShownNudges.forEach { meta ->
                val sdkInstance = SdkInstanceManager.getInstanceForAppId(meta.instanceId) ?: return
                sdkInstance.logger.log { "$tag dismissNudgeCampaigns() : $meta" }
                InAppInstanceProvider.getControllerForInstance(sdkInstance).viewHandler
                    .dismissOnConfigurationChange(meta)
            }
            configChangeMeta.lastShownNudges.clear()
        }
    }

    /**
     * Handle Dismissal and Re-Rendering of InApps on Orientation Change if Required
     * @param currentActivity instance of [Activity]
     * @param shouldDismissInApp true if InApp Should be dismissed , otherwise false
     */
    private fun handleInAppsOnOrientationChange(
        currentActivity: Activity,
        shouldDismissInApp: Boolean
    ) {
        Logger.print { "$tag handleInAppsOnOrientationChange() : Dismiss & Re-render InApp if required" }
        if (hasOrientationChanged(currentActivity)) {
            Logger.print { "$tag handleInAppsOnOrientationChange() : Orientation of Activity is changed" }
            dismissNudgeCampaignsIfRequired(shouldDismissInApp)
            // Dismiss General InApp & Re-Render InApp if its HTML InApp Campaign & current
            // orientation is supported.
            val inAppConfigMeta = configChangeMeta.lastShownGeneralCampaign ?: return
            val sdkInstance = SdkInstanceManager.getInstanceForAppId(inAppConfigMeta.instanceId)
                ?: return
            if (shouldDismissInApp) {
                InAppInstanceProvider.getControllerForInstance(sdkInstance).viewHandler
                    .dismissOnConfigurationChange(inAppConfigMeta)
            }
            reRenderInApp(currentActivity, sdkInstance)
        }
    }

    /**
     * Get InApp Config Meta Data from Campaign Payload
     * Instead of caching whole Campaign Payload, we can cache the only data we required to optimize
     * memory usage
     * @param campaignPayload instance of [CampaignPayload]
     * @param sdkInstance instance of [SdkInstance]
     */
    private fun getInAppConfigMetaFromPayload(
        campaignPayload: CampaignPayload,
        sdkInstance: SdkInstance
    ): InAppConfigMeta? {
        sdkInstance.logger.log { "$tag getInAppConfigMetaFromPayload(): $campaignPayload" }
        return try {
            when {
                campaignPayload.templateType ==
                    InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE ->
                    NudgeConfigMeta(
                        sdkInstance.instanceMeta.instanceId,
                        campaignId = campaignPayload.campaignId,
                        containerId = getContainerIdFromCampaignPayload(campaignPayload),
                        position = (campaignPayload as NativeCampaignPayload).position,
                        supportedOrientations = campaignPayload.supportedOrientations
                    )

                campaignPayload.inAppType == InAppType.HTML ->
                    HtmlInAppConfigMeta(
                        sdkInstance.instanceMeta.instanceId,
                        campaignPayload
                    )

                else -> {
                    InAppConfigMeta(
                        sdkInstance.instanceMeta.instanceId,
                        campaignPayload.campaignId,
                        getContainerIdFromCampaignPayload(campaignPayload),
                        campaignPayload.supportedOrientations
                    )
                }
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getInAppConfigMetaFromPayload() : " }
            null
        }
    }
}