/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.inapp.internal

import android.content.Context
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.currentISOTime
import com.moengage.core.internal.utils.currentSeconds
import com.moengage.core.internal.utils.getRequestId
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.CampaignStats
import com.moengage.inapp.internal.model.StatModel
import com.moengage.inapp.internal.model.enums.EvaluationStatusCode
import com.moengage.inapp.internal.model.meta.InAppCampaign
import com.moengage.inapp.model.CampaignContext
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.util.*

/**
 * @author Umang Chamaria
 * Date: 2019-10-11
 */

// Campaign Delivery States
private const val CAMPAIGN_ATTEMPTED = "ATM"

// Campaign priority stage
public const val PRIORITY_STAGE_HIGHER_PRIORITY_CAMPAIGN_AVAILABLE: String =
    "PRT_HIGH_PRT_CMP_AVL"
private const val PRIORITY_STAGE_MAX_TIMES_SHOWN: String = "PRT_MAX_TIM_SWN"
private const val PRIORITY_STAGE_CAMPAIGN_DELAY: String = "PRT_MIN_DEL"
private const val PRIORITY_STAGE_GLOBAL_DELAY: String = "PRT_GBL_DEL"
private const val PRIORITY_STAGE_INVALID_SCREEN: String = "PRT_SCR_MISMATCH"
private const val PRIORITY_STAGE_INVALID_CONTEXT: String = "PRT_CTX_MISMATCH"
private const val PRIORITY_STAGE_PERSISTENCE: String = "PRT_PERST"
private const val PRIORITY_STAGE_EXPIRY: String = "PRT_EXP"
private const val PRIORITY_STAGE_ORIENTATION_UNSUPPORTED: String = "PRT_ORT_UNSPP"
private const val PRIORITY_STAGE_INAPP_BLOCKED: String = "PRT_INAPP_BLK"
private const val PRIORITY_STAGE_MAX_NUDGES_DISPLAY_ON_SCREEN_LIMIT_REACHED: String =
    "PRT_NUDGE_SCR_MAX_SHW_LMT"
private const val PRIORITY_STAGE_NUDGE_POSITION_UNAVAILABLE: String = "PRT_NUDGE_PSTN_UNAVL"

// Campaign delivery Stage
public const val DELIVERY_STAGE_API_FAILURE: String = "DLV_API_FLR"
public const val DELIVERY_STAGE_MANDATORY_PARAM_MISSING: String = "DLV_MAND_PARM_MIS"

// Campaign impression
public const val IMPRESSION_STAGE_IMAGE_DOWNLOAD_FAILURE: String = "IMP_IMG_FTH_FLR"
public const val IMPRESSION_STAGE_GIF_LIBRARY_NOT_PRESENT: String = "IMP_GIF_LIB_MIS"
public const val IMPRESSION_STAGE_HEIGHT_EXCEEDS_DEVICE: String = "IMP_HGT_EXD_DEVC"
public const val IMPRESSION_STAGE_ORIENTATION_UNSUPPORTED: String = "IMP_ORT_UNSPP"
public const val IMPRESSION_STAGE_ANOTHER_CAMPAIGN_VISIBLE: String = "IMP_ANTR_CMP_VISB"
private const val IMPRESSION_STAGE_MAX_TIMES_SHOWN: String = "IMP_MAX_TIM_SHW"
private const val IMPRESSION_STAGE_CAMPAIGN_DELAY: String = "IMP_MIN_DEL"
private const val IMPRESSION_STAGE_GLOBAL_DELAY: String = "IMP_GBL_DEL"
private const val IMPRESSION_STAGE_SCREEN_CHANGE: String = "IMP_SCR_CHG"
private const val IMPRESSION_STAGE_CONTEXT_CHANGE: String = "IMP_CTX_CHG"
private const val IMPRESSION_STAGE_PERSISTENCE: String = "IMP_PERST"
private const val IMPRESSION_STAGE_EXPIRY: String = "IMP_EXP"
private const val IMPRESSION_STAGE_INAPP_BLOCKED: String = "IMP_INAPP_BLK"
internal const val IMPRESSION_STAGE_FILE_DOWNLOAD_FAILURE: String = "IMP_FILE_DWNLD_FLR"
private const val IMPRESSION_STAGE_CANCELLED_BEFORE_DELAY: String = "IMP_CNCL_BFR_DEL"
private const val IMPRESSION_STAGE_MAX_NUDGES_DISPLAY_ON_SCREEN_LIMIT_REACHED: String =
    "IMP_NUDGE_SCR_MAX_SHW_LMT"
private const val IMPRESSION_STAGE_NUDGE_POSITION_UNAVAILABLE: String = "IMP_NUDGE_PSTN_UNAVL"
public const val IMPRESSION_STAGE_VIDEO_DOWNLOAD_FAILURE: String = "IMP_VDO_FTH_FLR"

private val priorityStageFailureMap: MutableMap<EvaluationStatusCode, String> = hashMapOf(
    EvaluationStatusCode.GLOBAL_DELAY to PRIORITY_STAGE_GLOBAL_DELAY,
    EvaluationStatusCode.EXPIRY to PRIORITY_STAGE_EXPIRY,
    EvaluationStatusCode.INVALID_SCREEN to PRIORITY_STAGE_INVALID_SCREEN,
    EvaluationStatusCode.INVALID_CONTEXT to PRIORITY_STAGE_INVALID_CONTEXT,
    EvaluationStatusCode.PERSISTENT to PRIORITY_STAGE_PERSISTENCE,
    EvaluationStatusCode.MAX_COUNT to PRIORITY_STAGE_MAX_TIMES_SHOWN,
    EvaluationStatusCode.CAMPAIGN_DELAY to PRIORITY_STAGE_CAMPAIGN_DELAY,
    EvaluationStatusCode.BLOCKED_ON_SCREEN to PRIORITY_STAGE_INAPP_BLOCKED,
    EvaluationStatusCode.ORIENTATION_NOT_SUPPORTED to PRIORITY_STAGE_ORIENTATION_UNSUPPORTED,
    EvaluationStatusCode.MAX_NUDGES_DISPLAY_ON_SCREEN_LIMIT_REACHED to PRIORITY_STAGE_MAX_NUDGES_DISPLAY_ON_SCREEN_LIMIT_REACHED,
    EvaluationStatusCode.NUDGE_POSITION_UNAVAILABLE to PRIORITY_STAGE_NUDGE_POSITION_UNAVAILABLE
)

private val impressionStageFailureMap: MutableMap<EvaluationStatusCode, String> = hashMapOf(
    EvaluationStatusCode.GLOBAL_DELAY to IMPRESSION_STAGE_GLOBAL_DELAY,
    EvaluationStatusCode.EXPIRY to IMPRESSION_STAGE_EXPIRY,
    EvaluationStatusCode.INVALID_SCREEN to IMPRESSION_STAGE_SCREEN_CHANGE,
    EvaluationStatusCode.INVALID_CONTEXT to IMPRESSION_STAGE_CONTEXT_CHANGE,
    EvaluationStatusCode.PERSISTENT to IMPRESSION_STAGE_PERSISTENCE,
    EvaluationStatusCode.MAX_COUNT to IMPRESSION_STAGE_MAX_TIMES_SHOWN,
    EvaluationStatusCode.CAMPAIGN_DELAY to IMPRESSION_STAGE_CAMPAIGN_DELAY,
    EvaluationStatusCode.BLOCKED_ON_SCREEN to IMPRESSION_STAGE_INAPP_BLOCKED,
    EvaluationStatusCode.ORIENTATION_NOT_SUPPORTED to IMPRESSION_STAGE_ORIENTATION_UNSUPPORTED,
    EvaluationStatusCode.CANCELLED_BEFORE_DELAY to IMPRESSION_STAGE_CANCELLED_BEFORE_DELAY,
    EvaluationStatusCode.MAX_NUDGES_DISPLAY_ON_SCREEN_LIMIT_REACHED to IMPRESSION_STAGE_MAX_NUDGES_DISPLAY_ON_SCREEN_LIMIT_REACHED,
    EvaluationStatusCode.NUDGE_POSITION_UNAVAILABLE to IMPRESSION_STAGE_NUDGE_POSITION_UNAVAILABLE
)

internal class DeliveryLogger(private val sdkInstance: SdkInstance) {

    private val tag = "${InAppConstants.MODULE_TAG}StatsLogger"

    private var stats: MutableMap<String, CampaignStats> = HashMap()

    private val lock = Any()

    internal fun logCampaignAttempted(campaignMetaList: List<InAppCampaign>) {
        bulkLogging(campaignMetaList, CAMPAIGN_ATTEMPTED)
    }

    internal fun logPriorityStageFailure(
        campaign: InAppCampaign,
        statusCode: EvaluationStatusCode
    ) {
        sdkInstance.logger.log { "$tag logPriorityStageFailure() : Campaign-id: ${campaign.campaignMeta.campaignId}, status code: $statusCode" }
        val reason = priorityStageFailureMap[statusCode] ?: return
        campaign.campaignMeta.campaignContext ?: return
        updateStatForCampaign(
            campaign.campaignMeta.campaignContext,
            currentISOTime(),
            reason
        )
    }

    internal fun logImpressionStageFailure(
        campaign: CampaignPayload,
        statusCode: EvaluationStatusCode
    ) {
        sdkInstance.logger.log { "$tag logImpressionStageFailure() : Campaign-id: ${campaign.campaignId}, status code: $statusCode" }
        val reason = impressionStageFailureMap[statusCode] ?: return
        campaign.campaignContext
        updateStatForCampaign(
            campaign.campaignContext,
            currentISOTime(),
            reason
        )
    }

    private fun bulkLogging(campaignList: List<InAppCampaign>, reason: String) {
        if (!isStatsLoggingEnabled()) return

        val timestamp = currentISOTime()
        for (campaignMeta in campaignList) {
            campaignMeta.campaignMeta.campaignContext ?: continue
            updateStatForCampaign(campaignMeta.campaignMeta.campaignContext, timestamp, reason)
        }
    }

    internal fun updateStatForCampaign(campaign: InAppCampaign, timestamp: String, reason: String) {
        sdkInstance.logger.log { "$tag updateStatForCampaign() : Campaign-id: ${campaign.campaignMeta.campaignId}, reason: $reason" }
        campaign.campaignMeta.campaignContext ?: return
        updateStatForCampaign(campaign.campaignMeta.campaignContext, timestamp, reason)
    }

    internal fun updateStatForCampaign(
        campaignPayload: CampaignPayload,
        timestamp: String,
        reason: String
    ) {
        updateStatForCampaign(campaignPayload.campaignContext, timestamp, reason)
    }

    internal fun updateStatForCampaign(
        campaignContext: CampaignContext,
        timestamp: String,
        reason: String
    ) {
        synchronized(lock) {
            if (!isStatsLoggingEnabled()) return

            var campaignStats = stats[campaignContext.formattedCampaignId]
            if (campaignStats == null) {
                campaignStats = CampaignStats()
                campaignStats.reasons[reason] = mutableListOf<String>(timestamp)
                stats[campaignContext.formattedCampaignId] = campaignStats
                return
            }
            val occurrencesList = campaignStats.reasons[reason]
            if (occurrencesList == null) {
                val timestampList: MutableList<String> = ArrayList()
                timestampList.add(timestamp)
                campaignStats.reasons[reason] = timestampList
            } else {
                occurrencesList.add(timestamp)
            }
        }
    }

    internal fun writeStatsToStorage(context: Context) {
        try {
            if (!isStatsLoggingEnabled()) {
                sdkInstance.logger.log { "$tag writeStatsToStorage() : Stats upload is disabled, will not store stats." }
                stats.clear()
                return
            }
            if (stats.isEmpty()) {
                sdkInstance.logger.log { "$tag writeStatsToStorage() : Not stats to store" }
                return
            }
            val statsJson = JSONObject()
            for ((key, value) in stats) {
                statsJson.put(key, campaignStatToJson(value))
            }
            sdkInstance.logger.log { "$tag writeStatsToStorage() : Recorded Stats: $statsJson" }

            if (statsJson.length() == 0) return
            stats.clear()

            InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance).writeStats(
                StatModel(currentSeconds(), getRequestId(), statsJson)
            )
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag writeStatsToStorage() : " }
        }
    }

    @Throws(JSONException::class)
    internal fun campaignStatToJson(stats: CampaignStats): JSONObject {
        val campaignStatJson = JSONObject()
        for ((key, value) in stats.reasons) {
            campaignStatJson.put(key, listToJsonArray(value))
        }
        return campaignStatJson
    }

    private fun listToJsonArray(timestampList: List<String>): JSONArray {
        val jsonArray = JSONArray()
        for (timeStamp in timestampList) {
            jsonArray.put(timeStamp)
        }
        return jsonArray
    }

    private fun isStatsLoggingEnabled(): Boolean {
        return sdkInstance.remoteConfig.inAppConfig.isStatsEnabled
    }

    internal fun uploadStats(context: Context) {
        try {
            val repository = InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance)
            if (!isModuleEnabled(context, sdkInstance)) return
            InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance)
                .writeStatsToStorage(context)
            repository.uploadStats()
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag uploadStats() : " }
        }
    }
}