/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal

import android.content.Context
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.Event
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.currentISOTime
import com.moengage.core.internal.utils.isNotificationEnabled
import com.moengage.core.internal.utils.isNullOrEmpty
import com.moengage.evaluator.ConditionEvaluator
import com.moengage.inapp.internal.InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE
import com.moengage.inapp.internal.model.InAppGlobalState
import com.moengage.inapp.internal.model.enums.CampaignSubType
import com.moengage.inapp.internal.model.enums.EvaluationStatusCode
import com.moengage.inapp.internal.model.meta.InAppCampaign
import com.moengage.inapp.internal.model.meta.Trigger
import com.moengage.inapp.internal.model.meta.TriggerCondition
import org.json.JSONObject
import java.util.Collections

/**
 * @author Umang Chamaria
 */
internal class Evaluator(private val sdkInstance: SdkInstance) {

    private val tag = InAppConstants.MODULE_TAG + "Evaluator"

    /**
     * Evaluate the [TriggerCondition] object against the performed event and the attributes. It
     * use the condition evulator library internally
     *
     * @param condition instance of [TriggerCondition]
     * @param eventAttributes event attributes with formatted wrt condition evaluator library
     * @return [Boolean] - true if condition matched else false
     * @since 7.1.0
     */
    fun evaluateTriggerCondition(condition: TriggerCondition, eventAttributes: JSONObject): Boolean {
        return try {
            sdkInstance.logger.log { "$tag evaluateTriggerCondition() : Attribute for evaluation: $eventAttributes" }
            if (isNullOrEmpty(condition.attributes)) return true
            ConditionEvaluator(
                condition.attributes,
                eventAttributes
            ).evaluate()
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag evaluateCondition() : " }
            false
        }
    }

    fun isServerSyncRequired(
        lastSyncTime: Long,
        currentTime: Long,
        syncInterval: Long,
        isInAppSynced: Boolean
    ): Boolean {
        return !isInAppSynced || lastSyncTime + syncInterval < currentTime
    }

    fun canShowInAppOnActivity(activityName: String, blockedActivityList: Set<String?>): Boolean {
        if (blockedActivityList.contains(activityName)) {
            sdkInstance.logger.log(LogLevel.INFO) { "$tag canShowInAppOnActivity() : InApp blocked on screen: $activityName" }
            return false
        }
        return true
    }

    fun isCampaignEligibleForDisplay(
        inAppCampaign: InAppCampaign,
        contexts: Set<String>?,
        currentActivityName: String,
        globalState: InAppGlobalState,
        currentOrientation: Int,
        hasPushPermission: Boolean
    ): EvaluationStatusCode {
        val meta = inAppCampaign.campaignMeta
        val campaignState = inAppCampaign.campaignState

        sdkInstance.logger.log {
            "$tag isCampaignEligibleForDisplay() : Evaluating: ${meta.campaignId}\n Campaign meta: $meta \n State: $campaignState"
        }
        if (meta.templateType == IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE) {
            if (InAppModuleManager.hasMaxNudgeDisplayLimitReached()) {
                sdkInstance.logger.log {
                    "$tag isCampaignEligibleForDisplay() : Cannot show campaign: ${meta.campaignId} " +
                        "reason: Max nudges display limit has reached."
                }
                return EvaluationStatusCode.MAX_NUDGES_DISPLAY_ON_SCREEN_LIMIT_REACHED
            }
            sdkInstance.logger.log {
                "$tag isCampaignEligibleForDisplay() : Max nudges display on screen check passed"
            }
            if (InAppModuleManager.isNudgePositionVisible(meta.position)) {
                sdkInstance.logger.log {
                    "$tag isCampaignEligibleForDisplay() : Cannot show campaign: ${meta.campaignId} " +
                        "reason: Another nudge is already shown in position: ${meta.position}."
                }
                return EvaluationStatusCode.NUDGE_POSITION_UNAVAILABLE
            }
            sdkInstance.logger.log {
                "$tag isCampaignEligibleForDisplay() : Position availability check passed for " +
                    "position: ${meta.position}"
            }
        }
        if (meta.campaignSubType == CampaignSubType.PUSH_OPT_IN && hasPushPermission) {
            sdkInstance.logger.log {
                "$tag isCampaignEligibleForDisplay() : Cannot show campaign: ${meta.campaignId} " +
                    "reason: The App already has Notification permission."
            }
            return EvaluationStatusCode.CAMPAIGN_PURPOSE_SERVED
        }
        if (!canShowInAppInCurrentOrientation(currentOrientation, meta.supportedOrientations)) {
            sdkInstance.logger.log(LogLevel.INFO) {
                "$tag isCampaignEligibleForDisplay() : Cannot show campaign: ${meta.campaignId}" +
                    " current screen orientation: " + currentOrientation +
                    " supported orientations : " + meta.supportedOrientations.toString() +
                    " reason: in-app is not supported on current orientation."
            }
            return EvaluationStatusCode.ORIENTATION_NOT_SUPPORTED
        }
        if (!canShowInAppOnActivity(
                currentActivityName,
                sdkInstance.initConfig.inApp.getOptedOutScreenName()
            )
        ) {
            sdkInstance.logger.log(LogLevel.INFO) { "$tag isCampaignEligibleForDisplay() : Cannot show campaign: ${meta.campaignId} reason: in-app blocked on screen." }
            return EvaluationStatusCode.BLOCKED_ON_SCREEN
        }
        sdkInstance.logger.log { "$tag isCampaignEligibleForDisplay() : InApp Blocked on screen check passed." }
        if ((globalState.lastShowTime + globalState.globalDelay > globalState.currentDeviceTime) &&
            !meta.deliveryControl.frequencyCapping.ignoreGlobalDelay
        ) {
            sdkInstance.logger.log(LogLevel.INFO) {
                "$tag isCampaignEligibleForDisplay() : Cannot show campaign: ${meta.campaignId} reason: global delay failure"
            }
            return EvaluationStatusCode.GLOBAL_DELAY
        }
        sdkInstance.logger.log { "$tag isCampaignEligibleForDisplay() : Global minimum delay check passed." }
        if (meta.expiryTime < globalState.currentDeviceTime) {
            sdkInstance.logger.log(LogLevel.INFO) {
                "$tag isCampaignEligibleForDisplay() : Cannot show campaign: meta.campaignId reason: campaign expired"
            }
            return EvaluationStatusCode.EXPIRY
        }
        sdkInstance.logger.log { "$tag isCampaignEligibleForDisplay() : Campaign expiry check passed." }
        if ((
            meta.displayControl.rules.screenName != null &&
                meta.displayControl.rules.screenName != currentActivityName
            )
        ) {
            sdkInstance.logger.log(LogLevel.INFO) {
                "$tag isCampaignEligibleForDisplay() : Cannot show campaign: ${meta.campaignId} reason: cannot show in-app on this screen"
            }
            return EvaluationStatusCode.INVALID_SCREEN
        }
        sdkInstance.logger.log { "$tag isCampaignEligibleForDisplay() : Show only in screen check has passed" }
        if (!meta.displayControl.rules.context.isNullOrEmpty()) {
            if (contexts == null) return EvaluationStatusCode.INVALID_CONTEXT
            if (Collections.disjoint(contexts, meta.displayControl.rules.context)) {
                sdkInstance.logger.log(LogLevel.INFO) { "$tag isCampaignEligibleForDisplay() : Cannot show campaign: ${meta.campaignId} reason: current contextList not as" }
                return EvaluationStatusCode.INVALID_CONTEXT
            }
        }
        sdkInstance.logger.log { "$tag isCampaignEligibleForDisplay(): Context check has passed." }
        if ((
            meta.deliveryControl.frequencyCapping.maxCount > 0 &&
                campaignState.showCount >= meta.deliveryControl.frequencyCapping.maxCount
            )
        ) {
            sdkInstance.logger.log(LogLevel.INFO) {
                "$tag isCampaignEligibleForDisplay() : Cannot show campaign: ${meta.campaignId}" +
                    "reason: already shown max times"
            }
            return EvaluationStatusCode.MAX_COUNT
        }
        sdkInstance.logger.log { "$tag isCampaignEligibleForDisplay(): Max count check passed." }
        if (campaignState.lastShowTime + meta.deliveryControl.frequencyCapping.minimumDelay > globalState.currentDeviceTime) {
            sdkInstance.logger.log(LogLevel.INFO) {
                "$tag isCampaignEligibleForDisplay() : Cannot show campaign: ${meta.campaignId} " +
                    "reason: minimum delay between same campaign"
            }
            return EvaluationStatusCode.CAMPAIGN_DELAY
        }
        sdkInstance.logger.log { "$tag isCampaignEligibleForDisplay(): Minimum delay between same campaign check passed." }
        return EvaluationStatusCode.SUCCESS
    }

    fun getEligibleCampaignFromList(
        campaignList: List<InAppCampaign>,
        globalState: InAppGlobalState,
        appContext: Set<String>?,
        context: Context
    ): InAppCampaign? {
        sdkInstance.logger.log {
            "$tag getEligibleCampaignFromList(): Exclude already scheduled delay in-app " +
                "campaigns with campaignIds=${InAppInstanceProvider.getCacheForInstance(sdkInstance).scheduledCampaigns.keys}"
        }
        val filteredCampaigns = campaignList.filter { inAppCampaign ->
            !InAppInstanceProvider.getCacheForInstance(sdkInstance).scheduledCampaigns.contains(
                inAppCampaign.campaignMeta.campaignId
            )
        }

        InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance).logCampaignAttempted(filteredCampaigns)

        var suitableCampaign: InAppCampaign? = null
        var index = 0
        val activityName = InAppModuleManager.getCurrentActivityName() ?: return null

        while (index < filteredCampaigns.size) {
            val campaign = filteredCampaigns[index]
            val statusCode = isCampaignEligibleForDisplay(
                campaign,
                appContext,
                activityName,
                globalState,
                getCurrentOrientation(context),
                isNotificationEnabled(context)
            )
            when (statusCode) {
                EvaluationStatusCode.SUCCESS -> {
                    suitableCampaign = campaign
                    break
                }
                EvaluationStatusCode.CAMPAIGN_PURPOSE_SERVED -> {
                    sdkInstance.logger.log(LogLevel.INFO) {
                        "$tag getEligibleCampaignFromList() : Cannot show campaign: " +
                            "${campaign.campaignMeta.campaignId} reason: ${statusCode.name}"
                    }
                }
                else -> {
                    InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance)
                        .logPriorityStageFailure(campaign, statusCode)
                }
            }
            index++
        }
        if (suitableCampaign != null) {
            index++
            val timestamp = currentISOTime()
            while (index < filteredCampaigns.size) {
                val campaign = filteredCampaigns[index]
                InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance)
                    .updateStatForCampaign(
                        campaign,
                        timestamp,
                        PRIORITY_STAGE_HIGHER_PRIORITY_CAMPAIGN_AVAILABLE
                    )
                index++
            }
        }
        return suitableCampaign
    }

    /**
     * In-App will be shown on the same screen only for the orientation on which the show in-app
     * method was processed for the first time on that screen.
     */
    fun canShowInAppOnScreen(
        lastScreenData: ScreenData?,
        currentScreenName: String?,
        currentOrientation: Int
    ): Boolean {
        if (lastScreenData == null) return true
        return (lastScreenData.screenName == null && lastScreenData.screenOrientation == -1) ||
            (lastScreenData.screenName == currentScreenName && lastScreenData.screenOrientation == currentOrientation)
    }

    /**
     * Evaluate the [Trigger] object against the performed event i.e. [Event] object
     *
     * @param trigger instance of [Trigger]
     * @param event instance of [Event]
     * @param enrichAttribute event attributes with formatted wrt condition evaluator library
     * @return [Boolean] - true if condition matched else false
     * @since 7.1.0
     */
    fun evaluateTriggerForEvent(trigger: Trigger, event: Event, enrichAttribute: JSONObject): Boolean {
        try {
            sdkInstance.logger.log { "$tag evaluateTriggerForEvent() : Event - $event" }
            sdkInstance.logger.log { "$tag evaluateTriggerForEvent() : Trigger - $trigger" }
            for (triggerCondition in trigger.primaryConditions) {
                if (triggerCondition.eventName == event.name &&
                    evaluateTriggerCondition(triggerCondition, enrichAttribute)
                ) {
                    return true
                }
            }
            sdkInstance.logger.log { "$tag evaluateTriggerForEvent() : Trigger condition not met for provided event" }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag evaluateTriggerForEvent() : " }
        }

        return false
    }
}