/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal

import android.app.Activity
import android.content.Context
import com.moengage.core.internal.executor.Job
import com.moengage.core.internal.model.Event
import com.moengage.core.internal.model.SdkInstance
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.enums.StateUpdateType
import com.moengage.inapp.internal.model.meta.InAppCampaign
import com.moengage.inapp.internal.tasks.AppOpenHandler
import com.moengage.inapp.internal.tasks.ShowTestInApp
import com.moengage.inapp.internal.tasks.UpdateCampaignState
import com.moengage.inapp.listeners.SelfHandledAvailableListener
import com.moengage.inapp.model.enums.InAppPosition

/**
 * @author Umang Chamaria
 */

// Job Tags
private const val TAG_IN_APP_PREVIEW_TASK: String = "INAPP_PREVIEW_TASK"
private const val TAG_IN_APP_SHOW_TASK: String = "INAPP_SHOW_TASK"
private const val TAG_IN_APP_SHOW_SELF_HANDLED_TASK: String = "INAPP_SHOW_SELF_HANDLED_TASk"
private const val TAG_IN_APP_CHECK_TRIGGER_BASED_TASK: String =
    "INAPP_CHECK_AND_SHOW_TRIGGER_BASED_TASK"
private const val TAG_IN_APP_SHOW_TEST_IN_APP_TASK: String = "INAPP_SHOW_TEST_INAPP_TASK"
private const val TAG_IN_APP_UPDATE_CAMPAIGN_STATE_TASK: String = "INAPP_UPDATE_CAMPAIGN_STATE_TASK"
private const val TAG_IN_APP_UPLOAD_STATS_TASK: String = "INAPP_UPLOAD_STATS_TASK"
private const val TAG_APP_OPEN_TASK = "FETCH_IN_APP_META_TASK"
private const val TAG_RE_RENDER_IN_APP_ON_ORIENTATION_CHANGE =
    "RE_RENDER_INAPP_ON_ORIENTATION_CHANGE"
private const val TAG_SHOW_DELAYED_IN_APP = "SHOW_DELAYED_IN_APP_TASK"
private const val TAG_IN_APP_SHOW_NUDGE_TASK: String = "INAPP_SHOW_NUDGE_TASK"

public fun getShowInAppJob(context: Context, sdkInstance: SdkInstance): Job {
    return Job(TAG_IN_APP_SHOW_TASK, true) {
        ViewBuilder(context, sdkInstance).showGeneralInApp()
    }
}

public fun getPreviewInAppJob(
    context: Context,
    sdkInstance: SdkInstance,
    campaign: InAppCampaign
): Job {
    return Job(TAG_IN_APP_PREVIEW_TASK, true) {
        ViewBuilder(context, sdkInstance).showInAppPreview(campaign)
    }
}

public fun getSelfHandledInAppJob(
    context: Context,
    sdkInstance: SdkInstance,
    listener: SelfHandledAvailableListener
): Job {
    return Job(TAG_IN_APP_SHOW_SELF_HANDLED_TASK, true) {
        ViewBuilder(context, sdkInstance).getSelfHandledInApp(listener)
    }
}

public fun getShowTriggerJob(
    context: Context,
    sdkInstance: SdkInstance,
    event: Event,
    listener: SelfHandledAvailableListener?
): Job {
    return Job(TAG_IN_APP_CHECK_TRIGGER_BASED_TASK, false) {
        ViewBuilder(context, sdkInstance).showTriggeredInApp(event, listener)
    }
}

public fun getShowTestInAppJob(context: Context, sdkInstance: SdkInstance, campaignId: String):
    Job {
    return Job(TAG_IN_APP_SHOW_TEST_IN_APP_TASK, true) {
        ShowTestInApp(context, sdkInstance, campaignId).show()
    }
}

public fun getUpdateCampaignStatusJob(
    context: Context,
    sdkInstance: SdkInstance,
    updateType: StateUpdateType,
    campaignId: String
): Job {
    return Job(TAG_IN_APP_UPDATE_CAMPAIGN_STATE_TASK, false) {
        UpdateCampaignState(
            context,
            sdkInstance,
            updateType,
            campaignId,
            false
        ).update()
    }
}

public fun getUpdateSelfHandledCampaignStatusJob(
    context: Context,
    sdkInstance: SdkInstance,
    updateType: StateUpdateType,
    campaignId: String
): Job {
    return Job(TAG_IN_APP_UPDATE_CAMPAIGN_STATE_TASK, false) {
        UpdateCampaignState(
            context,
            sdkInstance,
            updateType,
            campaignId,
            true
        ).update()
    }
}

public fun getUploadStatsJob(context: Context, sdkInstance: SdkInstance): Job {
    return Job(TAG_IN_APP_UPLOAD_STATS_TASK, true) {
        InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance).uploadStats(context)
    }
}

public fun getAppOpenJob(context: Context, sdkInstance: SdkInstance): Job {
    return Job(TAG_APP_OPEN_TASK, true) {
        AppOpenHandler(context, sdkInstance).onAppOpen()
    }
}

internal fun getReRenderInAppJob(
    activity: Activity,
    sdkInstance: SdkInstance
): Job {
    return Job(TAG_RE_RENDER_IN_APP_ON_ORIENTATION_CHANGE, false) {
        ConfigurationChangeHandler.getInstance()
            .showInAppOnConfigurationChange(activity, sdkInstance)
    }
}

internal fun reRenderInApp(
    activity: Activity,
    sdkInstance: SdkInstance
) {
    sdkInstance.taskHandler.execute(getReRenderInAppJob(activity, sdkInstance))
}

internal fun showTestInApp(context: Context, sdkInstance: SdkInstance, campaignId: String) {
    sdkInstance.taskHandler.execute(getShowTestInAppJob(context, sdkInstance, campaignId))
}

internal fun previewInApp(context: Context, sdkInstance: SdkInstance, campaign: InAppCampaign) {
    sdkInstance.taskHandler.execute(getPreviewInAppJob(context, sdkInstance, campaign))
}

/**
 * Provides a Job for displaying delay in-app campaign after the specified delay.
 *
 * @param context instance of activity [Context]
 * @param sdkInstance instance of [SdkInstance]
 * @param campaign instance of [InAppCampaign]
 * @param payload instance of [CampaignPayload]
 * @param listener instance of [SelfHandledAvailableListener]
 * @return an instance of [Job]
 */
internal fun getDelayInAppJob(
    context: Context,
    sdkInstance: SdkInstance,
    campaign: InAppCampaign,
    payload: CampaignPayload,
    listener: SelfHandledAvailableListener?
): Job {
    return Job(TAG_SHOW_DELAYED_IN_APP, false) {
        ViewBuilder(context, sdkInstance).showDelayInApp(campaign, payload, listener)
    }
}

public fun getShowNudgeJob(
    context: Context,
    sdkInstance: SdkInstance,
    inAppPosition: InAppPosition
): Job {
    return Job(TAG_IN_APP_SHOW_NUDGE_TASK, false) {
        ViewBuilder(context, sdkInstance).showNudgeInApp(inAppPosition)
    }
}