/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal

import android.app.Activity
import android.content.Context
import android.os.Bundle
import com.moengage.core.LogLevel
import com.moengage.core.internal.CoreInternalHelper
import com.moengage.core.internal.exception.NetworkRequestDisabledException
import com.moengage.core.internal.model.Event
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.accountMetaForInstance
import com.moengage.core.internal.utils.currentSeconds
import com.moengage.core.internal.utils.getDeviceType
import com.moengage.core.internal.utils.isNotificationEnabled
import com.moengage.core.internal.utils.postOnMainThread
import com.moengage.inapp.MoEInAppHelper
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.enums.EvaluationStatusCode
import com.moengage.inapp.internal.model.enums.LifecycleType
import com.moengage.inapp.internal.model.enums.StateUpdateType
import com.moengage.inapp.internal.model.meta.InAppCampaign
import com.moengage.inapp.listeners.SelfHandledAvailableListener
import com.moengage.inapp.model.CampaignData
import com.moengage.inapp.model.InAppBaseData
import com.moengage.inapp.model.InAppData
import com.moengage.inapp.model.SelfHandledCampaignData
import com.moengage.inapp.model.enums.InAppPosition
import java.lang.ref.WeakReference
import java.util.concurrent.ScheduledExecutorService

/**
 * @author Umang Chamaria
 */
internal class InAppController(private val sdkInstance: SdkInstance) {

    private val tag = "${MODULE_TAG}InAppController"

    var isInAppSynced = false
        private set

    val viewHandler = ViewHandler(sdkInstance)

    var isShowInAppPending = false
    var isSelfHandledPending = false

    var scheduledExecutorService: ScheduledExecutorService? = null

    val syncObservable = SyncCompleteObservable()
    var isShowNudgePending = false

    /**
     * Synchronized lock for cancel delay in-app jobs
     */
    private val cancelDelayInAppLock = Any()

    fun onLogout(context: Context) {
        sdkInstance.logger.log { "$tag onLogout() : " }
        isInAppSynced = false
        cancelScheduledCampaigns()
        InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance).writeStatsToStorage(context)
        InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance).onLogout()
    }

    fun onSyncSuccess(context: Context) {
        sdkInstance.logger.log { "$tag onSyncSuccess() : " }
        isInAppSynced = true
        if (isShowInAppPending) {
            sdkInstance.logger.log { "$tag onSyncSuccess() : Processing pending showInApp()" }
            isShowInAppPending = false
            MoEInAppHelper.getInstance().showInApp(context, sdkInstance.instanceMeta.instanceId)
        }
        if (isSelfHandledPending) {
            sdkInstance.logger.log { "$tag onSyncSuccess() : Processing pending getSelfHandledInApp()" }
            isSelfHandledPending = false
            InAppInstanceProvider.getCacheForInstance(sdkInstance).pendingSelfHandledListener
                .get()?.let { selfHandledAvailableListener ->
                    getSelfHandledInApp(context, selfHandledAvailableListener)
                    InAppInstanceProvider.getCacheForInstance(sdkInstance).pendingSelfHandledListener.clear()
                }
        }
        if (isShowNudgePending) {
            isShowNudgePending = false
            processPendingNudgeDisplayRequest(context)
        }
        syncObservable.onSyncSuccess(sdkInstance)
    }

    fun showInAppIfPossible(context: Context) {
        try {
            // To Avoid Memory Leak , we need to use application context
            val appContext = context.applicationContext ?: context
            sdkInstance.logger.log { "$tag showInAppIfPossible() : " }
            if (!CoreInternalHelper.getInstanceState(sdkInstance).isInitialized) {
                sdkInstance.logger.log(LogLevel.INFO) { "$tag showInAppIfPossible() : Sdk Instance is not initialised. Cannot process showInApp()." }
                sdkInstance.taskHandler.submitRunnable {
                    showInAppIfPossible(appContext)
                }
                return
            }
            // Get Current Activity to read current orientation
            val currentActivity = InAppModuleManager.getActivity() ?: run {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag showInAppIfPossible() : Cannot show in-app, activity is null" }
                return
            }
            if (!Evaluator(sdkInstance).canShowInAppOnScreen(
                    InAppInstanceProvider.getCacheForInstance(sdkInstance).lastScreenData,
                    InAppModuleManager
                        .getCurrentActivityName(),
                    getCurrentOrientation(currentActivity)
                )
            ) {
                sdkInstance.logger.log {
                    "$tag showInAppIfPossible() :  showInApp() cannot processed, will be " +
                        "processed only when the current orientation is same as the " +
                        "orientation when the in-app show method was called for the first " +
                        "time on the current screen."
                }
                return
            }
            // Store the screen name and orientation
            InAppInstanceProvider.getCacheForInstance(sdkInstance)
                .updateLastScreenData(
                    ScreenData(
                        InAppModuleManager.getCurrentActivityName(),
                        getCurrentOrientation(currentActivity)
                    )
                )

            if (InAppModuleManager.isInAppVisible) {
                sdkInstance.logger.log { "$tag showInAppIfPossible() : Another in-app visible, cannot show campaign" }
                return
            }
            val repository = InAppInstanceProvider.getRepositoryForInstance(appContext, sdkInstance)
            if (!repository.isModuleEnabled()) return
            if (!isInAppSynced) {
                sdkInstance.logger.log { "$tag showInAppIfPossible() : InApp sync pending." }
                isShowInAppPending = true
                return
            }

            sdkInstance.taskHandler.execute(getShowInAppJob(appContext, sdkInstance))
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag showInAppIfPossible() : " }
        }
    }

    fun getSelfHandledInApp(context: Context, listener: SelfHandledAvailableListener) {
        sdkInstance.logger.log { "$tag getSelfHandledInApp() : " }
        val repository = InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance)
        if (!repository.isModuleEnabled()) return
        if (!isInAppSynced) {
            sdkInstance.logger.log { "$tag getSelfHandledInApp() : InApp sync pending." }
            isSelfHandledPending = true
            InAppInstanceProvider.getCacheForInstance(sdkInstance).pendingSelfHandledListener =
                WeakReference(listener)
            return
        }
        // To Avoid Memory Leak , we need to use application context
        val applicationContext = context.applicationContext ?: context
        sdkInstance.taskHandler.execute(
            getSelfHandledInAppJob(
                applicationContext,
                sdkInstance,
                listener
            )
        )
    }

    fun onAppOpen(context: Context) {
        sdkInstance.logger.log { "$tag onAppOpen() : " }
        sdkInstance.taskHandler.execute(getAppOpenJob(context, sdkInstance))
    }

    fun onAppBackground(context: Context) {
        try {
            sdkInstance.logger.log { "$tag onAppBackground() : " }
            cancelScheduledCampaigns()
            with(InAppInstanceProvider.getCacheForInstance(sdkInstance)) {
                pendingTriggerEvents.clear()
                hasHtmlCampaignSetupFailed = false
                clearPendingNudgesCalls()
            }
            scheduledExecutorService?.shutdown()

            sdkInstance.taskHandler.execute(getUploadStatsJob(context, sdkInstance))
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag onAppClose() : " }
        }
    }

    fun showTriggerInAppIfPossible(context: Context, event: Event) {
        sdkInstance.logger.log { "$tag showTriggerInAppIfPossible() : Event: ${event.name}" }
        if (!isInAppSynced) {
            InAppInstanceProvider.getCacheForInstance(sdkInstance).pendingTriggerEvents.add(event)
            sdkInstance.logger.log { "$tag showTriggerInAppIfPossible() : InApp meta not synced, cannot process trigger event now. Will wait for sync to finish." }
            return
        }
        if (InAppInstanceProvider.getCacheForInstance(sdkInstance).primaryTriggerEvents.contains(event.name)) {
            sdkInstance.taskHandler.execute(
                getShowTriggerJob(
                    context,
                    sdkInstance,
                    event,
                    InAppInstanceProvider.getCacheForInstance(sdkInstance).selfHandledListener
                )
            )
        }
    }

    fun selfHandledShown(context: Context, data: SelfHandledCampaignData) {
        try {
            sdkInstance.logger.log { "$tag selfHandledShown() : Campaign: ${data.campaignData.campaignId}" }
            trackInAppShown(context, sdkInstance, data.campaignData)
            sdkInstance.taskHandler.execute(
                getUpdateSelfHandledCampaignStatusJob(
                    context,
                    sdkInstance,
                    StateUpdateType.SHOWN,
                    data.campaignData.campaignId
                )
            )
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag selfHandledShown() : " }
        }
    }

    fun notifyLifecycleChange(payload: CampaignPayload, lifecycleType: LifecycleType) {
        sdkInstance.logger.log { "$tag notifyLifecycleChange() : Will try to notify listeners, campaignId: ${payload.campaignId}, lifecycle event: $lifecycleType" }
        val activity = InAppModuleManager.getActivity() ?: run {
            sdkInstance.logger.log(LogLevel.ERROR) { "$tag notifyLifecycleChange() : Cannot notify listeners, activity instance is null" }
            return
        }
        val data = InAppData(
            activity,
            InAppBaseData(
                CampaignData(payload.campaignId, payload.campaignName, payload.campaignContext),
                accountMetaForInstance(sdkInstance)
            )
        )
        sdkInstance.logger.log { "$tag notifyLifecycleChange() : Notifying Listener with data: $data" }
        for (listener in InAppInstanceProvider.getCacheForInstance(sdkInstance).lifeCycleListeners) {
            postOnMainThread {
                when (lifecycleType) {
                    LifecycleType.DISMISS -> listener.onDismiss(data)
                    LifecycleType.SHOWN -> listener.onShown(data)
                }
            }
        }
    }

    fun onInAppShown(activity: Activity, payload: CampaignPayload) {
        sdkInstance.logger.log { "$tag onInAppShown() : ${payload.campaignId}" }
        val context = activity.applicationContext
        ConfigurationChangeHandler.getInstance().saveLastInAppShownData(payload, sdkInstance)
        trackInAppShown(
            context,
            sdkInstance,
            CampaignData(payload.campaignId, payload.campaignName, payload.campaignContext)
        )
        sdkInstance.taskHandler.submit(
            getUpdateCampaignStatusJob(
                context,
                sdkInstance,
                StateUpdateType.SHOWN,
                payload.campaignId
            )
        )
        notifyLifecycleChange(payload, LifecycleType.SHOWN)
    }

    fun showInAppFromPush(context: Context, pushPayload: Bundle) {
        try {
            sdkInstance.logger.log { "$tag showInAppFromPush() : " }
            PushToInAppHandler(sdkInstance).shownInApp(context, pushPayload)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag showInAppFromPush() : " }
        }
    }

    /**
     * Writes stats to storage and syncs
     * @since 6.7.0
     */
    fun syncData(context: Context, sdkInstance: SdkInstance) {
        try {
            sdkInstance.logger.log { "$tag syncData() : " }
            sdkInstance.logger.log { "$tag syncData() : syncing stats" }
            InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance)
                .writeStatsToStorage(context)
            InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance).uploadStats()
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag syncData() : " }
        }
    }

    /**
     * Clears data and cache
     * @since 6.7.0
     */
    fun clearData(context: Context, sdkInstance: SdkInstance) {
        try {
            sdkInstance.logger.log { "$tag clearData() : " }
            InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance)
                .clearDataAndUpdateCache()
        } catch (t: Throwable) {
            sdkInstance.logger.log { "$tag clearData() : " }
        }
    }

    /**
     * Schedules an in-app campaign to be displayed after specified delay.
     *
     * @param context instance of activity [Context]
     * @param campaign instance of [InAppCampaign]
     * @param payload instance of [CampaignPayload]
     * @param listener instance of [SelfHandledAvailableListener]
     * @since 6.9.0
     */
    fun scheduleInApp(
        context: Context,
        campaign: InAppCampaign,
        payload: CampaignPayload,
        listener: SelfHandledAvailableListener?
    ) {
        try {
            sdkInstance.logger.log {
                "$tag scheduleInApp(): Try to schedule an in-app campaign for campaignId: " +
                    "${payload.campaignId} after delay: ${campaign.campaignMeta.displayControl.delay}"
            }
            val scheduledFuture =
                DelayedInAppHandler.schedule(
                    campaign.campaignMeta.displayControl.delay
                ) {
                    sdkInstance.taskHandler.execute(
                        getDelayInAppJob(context, sdkInstance, campaign, payload, listener)
                    )
                }
            sdkInstance.logger.log {
                "$tag scheduleInApp(): Add campaignId: ${payload.campaignId} to scheduled in-app cache"
            }
            InAppInstanceProvider.getCacheForInstance(sdkInstance).scheduledCampaigns[payload.campaignId] =
                DelayedInAppData(payload, scheduledFuture)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) {
                "$tag scheduleInApp(): Unable to schedule an in-app campaign for campaignId: ${payload.campaignId}"
            }
        }
    }

    /**
     * Cancels the execution of a scheduled delay in-app campaign
     *
     * @param campaignId [String]
     * @since 6.9.0
     */
    private fun cancelScheduledCampaign(
        campaignId: String
    ) {
        try {
            val scheduledCampaigns =
                InAppInstanceProvider.getCacheForInstance(sdkInstance).scheduledCampaigns
            val delayedInAppData = scheduledCampaigns[campaignId] ?: return
            sdkInstance.logger.log {
                "$tag cancelScheduledCampaign(): Will try to cancel delayed in-app task for campaignId: ${delayedInAppData.payload.campaignId}"
            }
            delayedInAppData.scheduledFuture.cancel(true)
            if (delayedInAppData.scheduledFuture.isCancelled) {
                InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance)
                    .logImpressionStageFailure(
                        delayedInAppData.payload,
                        EvaluationStatusCode.CANCELLED_BEFORE_DELAY
                    )
                sdkInstance.logger.log {
                    "$tag cancelScheduledCampaign(): Successfully cancelled delayed in-app task for campaignId: ${delayedInAppData.payload.campaignId}"
                }
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag cancelScheduledCampaign(): " }
        }
    }

    /**
     * Cancels all the scheduled delay in-app campaigns
     * @since 6.9.0
     */
    private fun cancelScheduledCampaigns() {
        synchronized(cancelDelayInAppLock) {
            try {
                sdkInstance.logger.log { "$tag cancelScheduledCampaigns(): will try to cancel the scheduled in-app campaigns" }
                val scheduledCampaigns =
                    InAppInstanceProvider.getCacheForInstance(sdkInstance).scheduledCampaigns
                for (campaign in scheduledCampaigns) {
                    cancelScheduledCampaign(campaign.key)
                }
            } catch (t: Throwable) {
                sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag cancelScheduledCampaigns():" }
            } finally {
                InAppInstanceProvider.getCacheForInstance(sdkInstance).scheduledCampaigns.clear()
            }
        }
    }

    fun onLogoutComplete(context: Context) {
        sdkInstance.logger.log { "$tag onLogoutComplete() : " }
        syncMeta(context)
    }

    @Synchronized
    fun syncMeta(context: Context) {
        try {
            sdkInstance.logger.log { "$tag syncMeta() : " }
            val repository = InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance)
            if (!Evaluator(sdkInstance).isServerSyncRequired(
                    repository.getLastSyncTime(),
                    currentSeconds(),
                    repository.getApiSyncInterval(),
                    isInAppSynced
                )
            ) {
                sdkInstance.logger.log { "$tag syncMeta() : sync not required." }
                return
            }
            with(repository) {
                fetchInAppCampaignMeta(getDeviceType(context), isNotificationEnabled(context))
                deleteExpiredCampaigns()
                updateCache()
            }
            onSyncSuccess(context)
            for (event in InAppInstanceProvider.getCacheForInstance(sdkInstance).pendingTriggerEvents) {
                showTriggerInAppIfPossible(context, event)
            }
            InAppInstanceProvider.getCacheForInstance(sdkInstance).pendingTriggerEvents.clear()
        } catch (t: Throwable) {
            if (t is NetworkRequestDisabledException) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag syncMeta() : Account or SDK Disabled." }
            } else {
                sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag syncMeta() : " }
            }
        }
    }

    /**
     * Sync Data and Reset Data
     * @since 7.0.0
     */
    fun syncAndResetData(context: Context, sdkInstance: SdkInstance) {
        try {
            sdkInstance.logger.log { "$tag syncAndResetData() : " }
            isInAppSynced = false
            InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance).writeStatsToStorage(context)
            InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance).syncAndResetData()
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag syncAndResetData() : " }
        }
    }

    /**
     * Submits a [Job] to show non-intrusive nudge campaign.
     */
    fun showNudgeIfPossible(context: Context, inAppPosition: InAppPosition) {
        try {
            sdkInstance.logger.log { "$tag showNudgeIfPossible() : Position: $inAppPosition" }
            // NOTE: This is required to avoid memory leak
            val applicationContext = context.applicationContext
            if (!CoreInternalHelper.getInstanceState(sdkInstance).isInitialized) {
                sdkInstance.logger.log(LogLevel.INFO) {
                    "$tag showNudgeIfPossible() : Sdk " +
                        "Instance is not initialised. Cannot process showNudge()."
                }
                sdkInstance.taskHandler.submitRunnable {
                    showNudgeIfPossible(applicationContext, inAppPosition)
                }
                return
            }

            val repository =
                InAppInstanceProvider.getRepositoryForInstance(applicationContext, sdkInstance)
            if (!repository.isModuleEnabled()) return
            if (!isInAppSynced) {
                sdkInstance.logger.log {
                    "$tag showNudgeIfPossible() : InApp sync pending. " +
                        "Queueing the call, will be processed after sync is successful."
                }
                isShowNudgePending = true
                InAppInstanceProvider.getCacheForInstance(sdkInstance)
                    .addToPendingNudgeCall(inAppPosition)
                return
            }
            sdkInstance.logger.log { "$tag showNudgeIfPossible() : will schedule a show nudge request." }
            sdkInstance.taskHandler.execute(
                getShowNudgeJob(
                    applicationContext,
                    sdkInstance,
                    inAppPosition
                )
            )
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag showNudgeIfPossible() : " }
        }
    }

    /**
     * Executes if there are any pending nudge calls are present in the queue.
     */
    internal fun processPendingNudgeDisplayRequest(context: Context) {
        try {
            sdkInstance.logger.log(LogLevel.INFO) { "$tag processPendingNudgeCalls() : " }
            val cache = InAppInstanceProvider.getCacheForInstance(sdkInstance)
            if (cache.getPendingNudgeCalls().isEmpty()) return
            val position = cache.getPendingNudgeCalls()[0]
            cache.getPendingNudgeCalls().remove(position)

            sdkInstance.logger.log(LogLevel.INFO) { "$tag processPendingNudgeCalls() :  will process for position: $position" }
            showNudgeIfPossible(context, position)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag processPendingNudgeCalls() : " }
        }
    }
}