/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.inapp.internal

import android.app.Activity
import android.content.Context
import android.os.Bundle
import com.moengage.core.internal.inapp.InAppHandler
import com.moengage.core.internal.model.Event
import com.moengage.core.internal.model.InAppV2Meta
import com.moengage.core.internal.model.InAppV3Meta
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.storage.database.DbAdapter
import com.moengage.inapp.internal.model.enums.CampaignSubType
import com.moengage.inapp.internal.model.meta.CampaignMeta
import com.moengage.inapp.internal.model.meta.CampaignState
import com.moengage.inapp.internal.model.meta.DeliveryControl
import com.moengage.inapp.internal.model.meta.DisplayControl
import com.moengage.inapp.internal.model.meta.FrequencyCapping
import com.moengage.inapp.internal.model.meta.Rules
import com.moengage.inapp.internal.repository.PayloadMapper
import com.moengage.inapp.internal.repository.local.DatabaseMigrationHandler

/**
 * @author Umang Chamaria
 */
public class InAppHandlerImpl : InAppHandler {

    override fun showTriggerInAppIfPossible(
        context: Context,
        sdkInstance: SdkInstance,
        event: Event
    ) {
        InAppInstanceProvider.getControllerForInstance(sdkInstance)
            .showTriggerInAppIfPossible(context, event)
    }

    override fun onStop(currentActivity: Activity) {
        InAppModuleManager.unRegisterActivity(currentActivity)
    }

    override fun onStart(currentActivity: Activity) {
        InAppModuleManager.registerActivity(currentActivity)
        ConfigurationChangeHandler.getInstance().onConfigurationChanged(false)
    }

    override fun onLogout(context: Context, sdkInstance: SdkInstance) {
        InAppInstanceProvider.getControllerForInstance(sdkInstance).onLogout(context)
    }

    override fun generateMetaForV2Campaign(inAppV2Meta: InAppV2Meta): InAppV3Meta {
        val meta = CampaignMeta(
            inAppV2Meta.campaignId,
            "",
            inAppV2Meta.expiry,
            0,
            DisplayControl(Rules(null, null), -1L),
            "",
            DeliveryControl(
                inAppV2Meta.priority,
                FrequencyCapping(false, 0, 0)
            ),
            null,
            null,
            null,
            null,
            CampaignSubType.GENERAL,
            null
        )
        val state = CampaignState(
            inAppV2Meta.showCount,
            inAppV2Meta.lastShowTime / 1000,
            inAppV2Meta.isClicked == 1
        )
        val mapper = PayloadMapper()
        return InAppV3Meta(CampaignMeta.toJson(meta), mapper.campaignStateToJson(state))
    }

    override fun showInAppFromPush(
        context: Context,
        sdkInstance: SdkInstance,
        pushPayload: Bundle
    ) {
        InAppInstanceProvider.getControllerForInstance(sdkInstance)
            .showInAppFromPush(context, pushPayload)
    }

    override fun onAppOpen(context: Context, sdkInstance: SdkInstance) {
        InAppInstanceProvider.getControllerForInstance(sdkInstance).onAppOpen(context)
    }

    override fun initialiseModule(context: Context) {
        InAppModuleManager.initialiseModule()
    }

    override fun onResume(currentActivity: Activity) {
        InAppModuleManager.checkAndRegisterActivity(currentActivity)
    }

    override fun onPause(currentActivity: Activity) {
        // TODO - decide whether to unregister activity
    }

    override fun onDatabaseMigration(
        context: Context,
        unencryptedSdkInstance: SdkInstance,
        encryptedSdkInstance: SdkInstance,
        unencryptedDbAdapter: DbAdapter,
        encryptedDbAdapter: DbAdapter
    ) {
        DatabaseMigrationHandler(
            context = context,
            unencryptedSdkInstance = unencryptedSdkInstance,
            encryptedSdkInstance = encryptedSdkInstance,
            unencryptedDbAdapter = unencryptedDbAdapter,
            encryptedDbAdapter = encryptedDbAdapter
        ).migrate()
    }

    /**
     * @since 6.7.0
     */
    override fun syncData(context: Context, sdkInstance: SdkInstance) {
        InAppInstanceProvider.getControllerForInstance(sdkInstance).syncData(context, sdkInstance)
    }

    /**
     * @since 6.7.0
     */
    override fun clearData(context: Context, sdkInstance: SdkInstance) {
        InAppInstanceProvider.getControllerForInstance(sdkInstance).clearData(context, sdkInstance)
    }

    override fun onLogoutComplete(context: Context, sdkInstance: SdkInstance) {
        InAppInstanceProvider.getControllerForInstance(sdkInstance).onLogoutComplete(context)
    }

    /**
     * @since 7.0.0
     */
    override fun syncAndResetData(context: Context, sdkInstance: SdkInstance) {
        InAppInstanceProvider.getControllerForInstance(sdkInstance).syncAndResetData(context, sdkInstance)
    }

    /**
     * @since 7.0.0
     */
    override fun onDestroy(activity: Activity) {
        InAppModuleManager.onDestroyActivity(activity)
    }

    /**
     * @since 7.0.0
     */
    override fun onCreate(activity: Activity) {
        InAppModuleManager.onCreateActivity(activity)
    }
}