/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal

import android.content.Context
import com.moengage.core.internal.CoreInternalHelper
import com.moengage.core.internal.model.SdkInstance
import com.moengage.inapp.internal.repository.InAppCache
import com.moengage.inapp.internal.repository.InAppRepository
import com.moengage.inapp.internal.repository.local.LocalRepositoryImpl
import com.moengage.inapp.internal.repository.remote.ApiManager
import com.moengage.inapp.internal.repository.remote.RemoteRepositoryImpl

/**
 * @author Arshiya Khanum
 */
internal object InAppInstanceProvider {

    private var deliveryLoggerCache = mutableMapOf<String, DeliveryLogger>()

    internal fun getDeliveryLoggerForInstance(sdkInstance: SdkInstance): DeliveryLogger {
        return deliveryLoggerCache[sdkInstance.instanceMeta.instanceId]
            ?: synchronized(InAppInstanceProvider::class.java) {
                val logger =
                    deliveryLoggerCache[sdkInstance.instanceMeta.instanceId] ?: DeliveryLogger(
                        sdkInstance
                    )
                deliveryLoggerCache[sdkInstance.instanceMeta.instanceId] = logger
                logger
            }
    }

    internal val controllerCache = mutableMapOf<String, InAppController>()

    internal fun getControllerForInstance(sdkInstance: SdkInstance): InAppController {
        return controllerCache[sdkInstance.instanceMeta.instanceId]
            ?: synchronized(InAppInstanceProvider::class.java) {
                val controller =
                    controllerCache[sdkInstance.instanceMeta.instanceId] ?: InAppController(
                        sdkInstance
                    )
                controllerCache[sdkInstance.instanceMeta.instanceId] = controller
                controller
            }
    }

    private val repositoryCache = mutableMapOf<String, InAppRepository>()

    internal fun getRepositoryForInstance(
        context: Context,
        sdkInstance: SdkInstance
    ): InAppRepository {
        val appContext = context.applicationContext ?: context
        return repositoryCache[sdkInstance.instanceMeta.instanceId]
            ?: synchronized(InAppInstanceProvider::class.java) {
                val repository =
                    repositoryCache[sdkInstance.instanceMeta.instanceId] ?: InAppRepository(
                        LocalRepositoryImpl(
                            appContext,
                            CoreInternalHelper.getDataAccessor(appContext, sdkInstance),
                            sdkInstance
                        ),
                        RemoteRepositoryImpl(
                            sdkInstance,
                            ApiManager(
                                sdkInstance,
                                CoreInternalHelper.getAuthorizationHandlerInstance(
                                    appContext,
                                    sdkInstance
                                )
                            )
                        ),
                        sdkInstance
                    )
                repositoryCache[sdkInstance.instanceMeta.instanceId] = repository
                repository
            }
    }

    internal val caches = mutableMapOf<String, InAppCache>()

    internal fun getCacheForInstance(sdkInstance: SdkInstance): InAppCache {
        return caches[sdkInstance.instanceMeta.instanceId]
            ?: synchronized(InAppInstanceProvider::class.java) {
                val cache = caches[sdkInstance.instanceMeta.instanceId] ?: InAppCache()
                caches[sdkInstance.instanceMeta.instanceId] = cache
                cache
            }
    }
}