/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal

import android.app.Activity
import android.content.Context
import android.view.View
import android.widget.FrameLayout
import com.moengage.core.LogLevel
import com.moengage.core.internal.lifecycle.LifecycleManager
import com.moengage.core.internal.listeners.AppBackgroundListenerInternal
import com.moengage.core.internal.logger.Logger
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.NativeCampaignPayload
import com.moengage.inapp.model.enums.InAppPosition
import java.lang.ref.WeakReference
import java.util.Collections

/**
 * @author Umang Chamaria
 */
private const val MAX_NUDGES_DISPLAY_COUNT = 3

internal object InAppModuleManager : AppBackgroundListenerInternal {

    private val lock = Any()
    private val showInAppLock = Any()
    private val resetInAppCacheLock = Any()

    private const val tag = "${MODULE_TAG}InAppModuleManager"

    private var currentActivity: WeakReference<Activity>? = null
    private val showInAppStateForInstanceCache = hashSetOf<String>()

    var isInAppVisible = false
        private set

    private var shouldRegisterActivityOnResume: Boolean = false

    /**
     * Cache of processing Non-Intrusive nudge positions
     */
    private val processingNonIntrusiveNudgePositions =
        Collections.synchronizedMap(hashMapOf<String, MutableSet<InAppPosition>>())

    /**
     * Cache of visible Non-Intrusive nudge positions
     */
    private val visibleNonIntrusiveNudgePositions =
        Collections.synchronizedMap(hashMapOf<String, MutableSet<InAppPosition>>())

    /**
     * Lock for synchronously filtering the nudge campaigns to find suitable campaign
     * NOTE: This is required since multiple show nudge tasks are executed in parallel
     */
    internal val showNudgeLock = Any()

    fun registerActivity(activity: Activity) {
        Logger.print { "$tag registerActivity() : ${activity.javaClass.name}" }
        clearIsShowInAppCalledForInstanceCache(activity)
        updateCurrentActivityContext(activity)
    }

    fun unRegisterActivity(activity: Activity) {
        try {
            Logger.print { "$tag unRegisterActivity() : ${activity.javaClass.name}" }
            if (currentActivity?.get() == activity) {
                Logger.print { "$tag unRegisterActivity() : setting null" }
                updateCurrentActivityContext(null)
            }
            // Remove auto dismiss runnable after activity got destroyed to avoid memory Leak
            for (controller in InAppInstanceProvider.controllerCache.values) {
                controller.viewHandler.clearAutoDismissRunnables()
            }
        } catch (t: Throwable) {
            Logger.print(LogLevel.ERROR, t) { "$tag unRegisterActivity() : " }
        }
    }

    private fun updateCurrentActivityContext(activity: Activity?) {
        currentActivity = if (activity == null) null else WeakReference(activity)
    }

    fun getCurrentActivityName(): String? {
        val activity = currentActivity?.get() ?: return null
        val activityName = activity.javaClass.name
        Logger.print { "$tag getCurrentActivityName() : Activity Name: $activityName" }
        return activityName
    }

    fun getActivity(): Activity? {
        return currentActivity?.get()
    }

    fun updateInAppVisibility(isVisible: Boolean) {
        Logger.print { "$tag updateInAppVisibility(): Visibility State: $isVisible" }
        synchronized(lock) {
            isInAppVisible = isVisible
        }
    }

    fun addInAppToViewHierarchy(
        root: FrameLayout,
        view: View,
        payload: CampaignPayload,
        isShowOnConfigChange: Boolean
    ) {
        synchronized(showInAppLock) {
            Logger.print {
                "$tag addInAppToViewHierarchy(): showing campaign ${payload.campaignId} "
            }

            if (payload.templateType != InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE &&
                isInAppVisible && !isShowOnConfigChange
            ) {
                Logger.print {
                    "$tag addInAppToViewHierarchy(): another campaign visible, cannot " +
                        "show campaign ${payload.campaignId}"
                }
                return
            }
            if (payload.templateType == InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE) {
                val position = (payload as NativeCampaignPayload).position
                if (isNudgePositionVisible(position) || hasMaxNudgeDisplayLimitReached()) {
                    Logger.print {
                        "$tag addInAppToViewHierarchy(): another campaign visible at the " +
                            "position: $position, cannot show campaign ${payload.campaignId}"
                    }
                    return
                }
            }
            root.addView(view)
            if (payload.templateType == InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE) {
                val position = (payload as NativeCampaignPayload).position
                updateVisibleNudgePosition(position)
            } else {
                updateInAppVisibility(true)
            }
        }
    }

    private fun clearIsShowInAppCalledForInstanceCache(currentActivity: Activity) {
        Logger.print { "$tag clearIsShowInAppCalledForInstanceCache() : " }
        val lastActivityName = getCurrentActivityName()
        val currentActivityName = currentActivity.javaClass.name
        if (lastActivityName != currentActivityName) {
            Logger.print {
                "$tag clearIsShowInAppCalledForInstanceCache() : clearing.. $showInAppStateForInstanceCache"
            }
            resetShowInAppShowState()
        }
    }

    private fun resetShowInAppShowState() {
        try {
            synchronized(resetInAppCacheLock) {
                Logger.print { "$tag resetShowInAppShowState() : Resetting show state." }
                for (cache in InAppInstanceProvider.caches.values) {
                    cache.updateLastScreenData(ScreenData(null, -1))
                }
            }
        } catch (t: Throwable) {
            Logger.print(LogLevel.ERROR, t) { "$tag resetShowInAppShowState(): " }
        }
    }

    private var hasInitialised = false

    fun initialiseModule() {
        Logger.print { "$tag initialiseModule() : Will initialise module if needed." }
        if (hasInitialised) return
        synchronized(lock) {
            if (hasInitialised) return
            Logger.print { "$tag initialiseModule() : Initialising InApp module" }
            LifecycleManager.addBackgroundListener(this)
        }
    }

    fun enableActivityRegistrationOnResume() {
        Logger.print { "$tag enableOnResumeActivityRegister() : " }
        shouldRegisterActivityOnResume = true
    }

    fun disableActivityRegistrationOnResume() {
        Logger.print { "$tag disableOnResumeActivityRegister() : " }
        shouldRegisterActivityOnResume = false
    }

    fun checkAndRegisterActivity(currentActivity: Activity) {
        Logger.print { "$tag checkAndRegisterActivity() : ${currentActivity.javaClass.name}" }
        if (!shouldRegisterActivityOnResume) return
        registerActivity(currentActivity)
    }

    override fun onAppBackground(context: Context) {
        Logger.print { "$tag onAppBackground() : " }
        ConfigurationChangeHandler.getInstance().clearData()
        resetShowInAppShowState()
        for (controller in InAppInstanceProvider.controllerCache.values) {
            controller.onAppBackground(context)
        }
    }

    /**
     * returns true if the nudge is visible at the position, else false
     */
    fun isNudgePositionVisible(position: InAppPosition): Boolean {
        Logger.print { "$tag isNudgePositionVisible(): $position" }
        return visibleNonIntrusiveNudgePositions[getNotNullCurrentActivityName()]?.contains(position)
            ?: false
    }

    /**
     * return true if the nudge is processing for the position, else false
     */
    fun isNudgePositionProcessing(position: InAppPosition): Boolean {
        Logger.print { "$tag isNudgePositionProcessing(): $position" }
        return processingNonIntrusiveNudgePositions[getNotNullCurrentActivityName()]?.contains(
            position
        ) ?: false
    }

    /**
     * Returns true if the current visible or processing nudges count has reached the max limit
     */
    fun hasMaxNudgeDisplayLimitReached(): Boolean {
        return visibleNonIntrusiveNudgePositions.size >= MAX_NUDGES_DISPLAY_COUNT
    }

    /**
     * Updates the shown nudge position in visible or processing nudges
     */
    private fun updateVisibleNudgePosition(position: InAppPosition) {
        val currentActivityName = getNotNullCurrentActivityName()
        Logger.print { "$tag updateVisibleNudgePosition(): position: $position activity: $currentActivityName" }

        processingNonIntrusiveNudgePositions[currentActivityName]?.remove(position)
        if (visibleNonIntrusiveNudgePositions.containsKey(currentActivityName)) {
            visibleNonIntrusiveNudgePositions[currentActivityName]?.add(position)
        } else {
            visibleNonIntrusiveNudgePositions[currentActivityName] = mutableSetOf(position)
        }
    }

    /**
     * Adds a position to processing nudge positions cache
     */
    fun addProcessingNudgePosition(position: InAppPosition) {
        val currentActivityName = getNotNullCurrentActivityName()
        Logger.print { "$tag addProcessingNudgePosition(): position: $position activity: $currentActivityName" }
        if (processingNonIntrusiveNudgePositions.containsKey(currentActivityName)) {
            processingNonIntrusiveNudgePositions[getNotNullCurrentActivityName()]?.add(position)
        } else {
            processingNonIntrusiveNudgePositions[getNotNullCurrentActivityName()] =
                mutableSetOf(position)
        }
    }

    /**
     * Removes a position from visible nudge positions cache
     */
    fun removeVisibleNudgePosition(position: InAppPosition) {
        Logger.print { "$tag removeVisibleNudgePosition() : Removing from position: $position" }
        visibleNonIntrusiveNudgePositions[getNotNullCurrentActivityName()]?.remove(position)
    }

    /**
     * Removes a position from processing nudge positions cache
     *
     * @param position
     */
    fun removeProcessingNudgePosition(position: InAppPosition) {
        Logger.print { "$tag removeProcessingNudgePosition() : Removing from position: $position" }
        processingNonIntrusiveNudgePositions[getNotNullCurrentActivityName()]?.remove(position)
    }

    fun onCreateActivity(activity: Activity) {
        Logger.print { "$tag onCreateActivity(): ${activity.javaClass.name}" }
        clearNudgesCache(activity.javaClass.name)
    }

    fun onDestroyActivity(activity: Activity) {
        Logger.print { "$tag onDestroyActivity(): ${activity.javaClass.name}" }
        clearNudgesCache(activity.javaClass.name)
    }

    /**
     * Clear Nudges cache for a specified activity.
     *
     * @param activityName [String] name of the activity
     */
    private fun clearNudgesCache(activityName: String) {
        Logger.print { "$tag clearNudgesCache() : Activity name: $activityName" }
        // Remove positions from processing nudges positions
        if (processingNonIntrusiveNudgePositions.containsKey(activityName)) {
            processingNonIntrusiveNudgePositions.remove(activityName)
        }
        // Remove Positions from visible nudges positions
        if (visibleNonIntrusiveNudgePositions.containsKey(activityName)) {
            visibleNonIntrusiveNudgePositions.remove(activityName)
        }
        // Remove campaigns from instance specific cache
        for (inAppCache in InAppInstanceProvider.caches) {
            inAppCache.value.activityBasedVisibleOrProcessingCampaigns.remove(activityName)
                ?.forEach { campaignId ->
                    inAppCache.value.visibleOrProcessingCampaigns.remove(campaignId)
                }
        }
    }

    /**
     * Provides current activity name if not-null
     *
     * @return [String] Current activity name
     * @throws IllegalStateException if current activity name is null
     */
    @Throws(IllegalStateException::class)
    fun getNotNullCurrentActivityName(): String {
        return getCurrentActivityName()
            ?: throw IllegalStateException("Current activity name is null")
    }
}