/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal

import android.content.Context
import android.content.res.Configuration
import android.graphics.Bitmap
import android.view.View
import android.widget.ImageView
import com.bumptech.glide.Glide
import com.bumptech.glide.load.MultiTransformation
import com.bumptech.glide.load.resource.bitmap.RoundedCorners
import com.moengage.core.LogLevel
import com.moengage.core.Properties
import com.moengage.core.internal.GENERIC_PARAM_V2_VALUE_OS
import com.moengage.core.internal.MOE_CAMPAIGN_ID
import com.moengage.core.internal.MOE_CAMPAIGN_NAME
import com.moengage.core.internal.global.GlobalResources.mainThread
import com.moengage.core.internal.logger.Logger
import com.moengage.core.internal.model.AppMeta
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.model.ViewDimension
import com.moengage.core.internal.utils.getSdkVersion
import com.moengage.core.internal.utils.isNotificationEnabled
import com.moengage.core.internal.utils.isoStringFromSeconds
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.NativeCampaignPayload
import com.moengage.inapp.internal.model.ViewCreationMeta
import com.moengage.inapp.internal.model.enums.EvaluationStatusCode
import com.moengage.inapp.internal.model.enums.InAppType
import com.moengage.inapp.internal.model.enums.ScreenOrientation
import com.moengage.inapp.internal.model.meta.InAppCampaign
import com.moengage.inapp.model.CampaignContext
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject

/**
 * @author Arshiya Khanum
 */

private const val tag = "${InAppConstants.MODULE_TAG}Utils"

internal fun isModuleEnabled(context: Context, sdkInstance: SdkInstance): Boolean {
    if (!InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance).isModuleEnabled()) {
        Logger.print { "$tag isModuleEnabled() : InApp Module is disabled. Cannot show in-app." }
        return false
    }
    return true
}

public fun isValidJavaScriptString(string: String?): Boolean =
    (string != "undefined" && string != "null" && !string.isNullOrBlank())

@Throws(JSONException::class)
public fun screenOrientationFromJson(jsonArray: JSONArray): Set<ScreenOrientation> {
    val screenOrientations: MutableSet<ScreenOrientation> = mutableSetOf()
    for (i in 0 until jsonArray.length()) {
        screenOrientations.add(ScreenOrientation.valueOf(jsonArray.getString(i).uppercase()))
    }
    return screenOrientations
}

public fun canShowInAppInCurrentOrientation(
    orientation: Int,
    supportedOrientations: Set<ScreenOrientation>
): Boolean {
    return supportedOrientations.contains(screenOrientationMapper[orientation])
}

public fun getCurrentOrientation(context: Context): Int {
    return context.resources.configuration.orientation
}

public fun addAttributesToProperties(
    properties: Properties,
    campaignId: String,
    campaignName: String,
    campaignContext: CampaignContext? = null
) {
    properties.addAttribute(MOE_CAMPAIGN_ID, campaignId)
        .addAttribute(MOE_CAMPAIGN_NAME, campaignName)
    if (campaignContext != null) {
        for ((key, value) in campaignContext.attributes) {
            properties.addAttribute(key, value)
        }
    }
}

private val screenOrientationMapper: Map<Int, ScreenOrientation> = mapOf(
    Configuration.ORIENTATION_PORTRAIT to ScreenOrientation.PORTRAIT,
    Configuration.ORIENTATION_LANDSCAPE to ScreenOrientation.LANDSCAPE
)

internal fun isValidJavaScriptValue(string: Any?): Boolean =
    (string != "undefined" && string != "null")

internal fun getStatusBarHeight(context: Context): Int {
    var marginTop = 0
    val resource = context.resources.getIdentifier("status_bar_height", "dimen", "android")
    if (resource > 0) {
        marginTop = context.resources.getDimensionPixelSize(resource)
    }
    return marginTop
}

internal fun getScreenDimension(context: Context): ViewDimension {
    val displayMetrics = context.resources.displayMetrics
    return ViewDimension(displayMetrics.widthPixels, displayMetrics.heightPixels)
}

internal fun getViewCreationMeta(context: Context): ViewCreationMeta {
    return ViewCreationMeta(
        getScreenDimension(
            context
        ),
        getStatusBarHeight(context),
        getNavigationBarHeight(context)
    )
}

internal fun canShowInApp(context: Context, sdkInstance: SdkInstance): Boolean {
    val canShowInApp = isModuleEnabled(context, sdkInstance) &&
        InAppInstanceProvider.getControllerForInstance(sdkInstance).isInAppSynced
    sdkInstance.logger.log { "$tag canShowInApp() : Can show InApp? $canShowInApp" }
    return canShowInApp
}

internal fun logCurrentInAppState(context: Context, sdkInstance: SdkInstance) {
    sdkInstance.logger.log { "$tag logCurrentInAppState() : Current Activity: ${InAppModuleManager.getCurrentActivityName()}" }
    sdkInstance.logger.log {
        "$tag logCurrentInAppState() : InApp-Context: ${
        InAppInstanceProvider.getCacheForInstance(sdkInstance).inAppContext
        }"
    }
    val globalState =
        InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance).getGlobalState()
    sdkInstance.logger.log {
        "$tag logCurrentInAppState() : \n Global Delay: ${globalState.globalDelay} " +
            "\n Last campaign show at: ${isoStringFromSeconds(globalState.lastShowTime)}" +
            "\n Current Time: ${isoStringFromSeconds(globalState.currentDeviceTime)}"
    }
}

internal fun getUnspecifiedViewDimension(view: View): ViewDimension {
    view.measure(View.MeasureSpec.UNSPECIFIED, View.MeasureSpec.UNSPECIFIED)
    return ViewDimension(view.measuredWidth, view.measuredHeight)
}

internal fun isInAppExceedingScreen(context: Context, view: View): Boolean {
    val screenDimension = getScreenDimension(context)
    val inAppDimension = getUnspecifiedViewDimension(view)
    return screenDimension.height < inAppDimension.height
}

/**
 * Checks if the campaign is eligible to display and logs the impression failure in case
 * eligibility checks failed.
 *
 * @param context instance of activity [Context]
 * @param sdkInstance instance of [SdkInstance]
 * @param campaign instance of [InAppCampaign]
 * @param payload instance of [CampaignPayload]
 * @return true if the the campaign is eligible to display otherwise false.
 */
internal fun isCampaignEligibleForDisplay(
    context: Context,
    sdkInstance: SdkInstance,
    campaign: InAppCampaign,
    payload: CampaignPayload
): Boolean {
    val statusCode = Evaluator(sdkInstance).isCampaignEligibleForDisplay(
        campaign,
        InAppInstanceProvider.getCacheForInstance(sdkInstance).inAppContext,
        InAppModuleManager.getCurrentActivityName() ?: "",
        InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance).getGlobalState(),
        getCurrentOrientation(context),
        isNotificationEnabled(context)
    )
    if (statusCode != EvaluationStatusCode.SUCCESS) {
        sdkInstance.logger.log(LogLevel.INFO) { "$tag isCampaignEligibleForDisplay() : Cannot show in-app, conditions don't satisfy." }
        InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance)
            .logImpressionStageFailure(payload, statusCode)
        return false
    }
    return true
}

/**
 * Checks if a campaign is a delay in-app campaign.
 *
 * @param campaign instance of [InAppCampaign]
 * @return true if the campaign is a delay in-app campaign otherwise false.
 */
internal fun isDelayedInApp(
    campaign: InAppCampaign
): Boolean {
    if (campaign.campaignMeta.displayControl.delay != -1L) {
        return true
    }
    return false
}

internal fun getNavigationBarHeight(context: Context): Int {
    var marginBottom = 0
    val resource = context.resources.getIdentifier("navigation_bar_height", "dimen", "android")
    if (resource > 0) {
        marginBottom = context.resources.getDimensionPixelSize(resource)
    }
    return marginBottom
}

/**
 * Adding some of the default attributes to events tracked event attributes. This is only
 * used for evaluation and not actually tracked.
 *
 * @param attributes event attributes which will be used for evaluation
 * @param appMeta instance of [AppMeta]
 * @return [JSONObject] - enriched JSONObject
 * @since 7.1.0
 */
internal fun enrichAttributesForTriggeredEvaluation(
    attributes: JSONObject,
    appMeta: AppMeta
): JSONObject {
    val enrichedAttributes = JSONObject(attributes.toString())
    enrichedAttributes.put(ENRICH_EVENT_ATTRIBUTE_SDK_VERSION, getSdkVersion().toString())
    enrichedAttributes.put(ENRICH_EVENT_ATTRIBUTE_OS, GENERIC_PARAM_V2_VALUE_OS)
    enrichedAttributes.put(ENRICH_EVENT_ATTRIBUTE_APP_VERSION_CODE, appMeta.versionCode.toString())
    enrichedAttributes.put(ENRICH_EVENT_ATTRIBUTE_APP_VERSION_NAME, appMeta.versionName)

    return enrichedAttributes
}

/**
 * Loads background image and sets the border radius for the background image
 *
 * @param context instance of [Context]
 * @param sdkInstance instance of [SdkInstance]
 * @param borderRadius border radius of the container
 * @param src background source
 * @param imageView target [ImageView]
 * @param isGif true if the [src] is of type gif, else false
 * @since 7.1.1
 */
public fun loadContainerImageBackground(
    context: Context,
    sdkInstance: SdkInstance,
    borderRadius: Int,
    src: Any,
    imageView: ImageView,
    isGif: Boolean = false
) {
    sdkInstance.logger.log { "$tag loadContainerImageBackground () : will load bitmap. borderRadius: $borderRadius" }
    mainThread.post {
        try {
            val requestManager = Glide.with(context)
            var requestBuilder = when {
                src is Bitmap -> {
                    sdkInstance.logger.log {
                        "$tag loadContainerImageBackground () : src is Bitmap"
                    }
                    requestManager.asBitmap()
                }

                isGif -> {
                    sdkInstance.logger.log {
                        "$tag loadContainerImageBackground () : src is Gif"
                    }
                    requestManager.asGif()
                }

                else -> {
                    throw Exception("loadContainerImageBackground(): src type is not supported")
                }
            }
            if (borderRadius > 0) {
                sdkInstance.logger.log {
                    "$tag loadContainerImageBackground () : applying borderRadius: " +
                        "${borderRadius}px"
                }
                requestBuilder = requestBuilder.transform(
                    MultiTransformation(
                        RoundedCorners(
                            borderRadius
                        )
                    )
                )
            }
            requestBuilder
                .load(src)
                .into(imageView)
            sdkInstance.logger.log { "$tag loadContainerImageBackground () : completed" }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag loadContainerImageBackground () : " }
        }
    }
}

/**
 * Get Container View ID of the InApp. Returns -1 if Container ID Not found
 * @param campaignPayload - Instance of [CampaignPayload]
 * @since 7.1.2
 */
internal fun getContainerIdFromCampaignPayload(
    campaignPayload: CampaignPayload
): Int {
    return try {
        if (campaignPayload.inAppType === InAppType.NATIVE) {
            (InAppConstants.CONTAINER_BASE_ID + (campaignPayload as NativeCampaignPayload).primaryContainer!!.id)
        } else {
            InAppConstants.HTML_CONTAINER_ID
        }
    } catch (t: Throwable) {
        Logger.print { "getContainerIdFromCampaignPayload() : $campaignPayload" }
        -1
    }
}