/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal

import android.content.Context
import com.moengage.core.LogLevel
import com.moengage.core.internal.data.events.transformEventAttributesForEvaluationPackage
import com.moengage.core.internal.global.GlobalCache
import com.moengage.core.internal.model.Event
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.accountMetaForInstance
import com.moengage.core.internal.utils.currentISOTime
import com.moengage.core.internal.utils.getDeviceType
import com.moengage.core.internal.utils.postOnMainThread
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.NativeCampaignPayload
import com.moengage.inapp.internal.model.TriggerRequestMeta
import com.moengage.inapp.internal.model.meta.InAppCampaign
import com.moengage.inapp.listeners.SelfHandledAvailableListener
import com.moengage.inapp.model.CampaignData
import com.moengage.inapp.model.SelfHandledCampaign
import com.moengage.inapp.model.SelfHandledCampaignData
import com.moengage.inapp.model.enums.InAppPosition

/**
 * @author Umang Chamaria
 */
internal class ViewBuilder(private val context: Context, private val sdkInstance: SdkInstance) {

    private val tag = InAppConstants.MODULE_TAG + "ViewBuilder"

    private val controller = InAppInstanceProvider.getControllerForInstance(sdkInstance)
    private val repository = InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance)

    fun showGeneralInApp() {
        try {
            sdkInstance.logger.log { "$tag showGeneralInApp() : Will try to show general inapp" }
            if (!canShowInApp(context, sdkInstance)) return
            logCurrentInAppState(context, sdkInstance)
            val suitableInApp =
                getSuitableInApp(InAppInstanceProvider.getCacheForInstance(sdkInstance).generalCampaign)
                    ?: run {
                        sdkInstance.logger.log(LogLevel.ERROR) { "$tag showGeneralInApp() : No suitable campaign found" }
                        return
                    }
            sdkInstance.logger.log { "$tag showGeneralInApp() : Suitable InApp $suitableInApp" }
            val payload = getPayloadForCampaign(suitableInApp) ?: run {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag showGeneralInApp() : Campaign payload empty" }
                return
            }
            if (isDelayedInApp(suitableInApp)) {
                sdkInstance.logger.log { "$tag showGeneralInApp() : Delayed campaign, scheduling campaign" }
                InAppInstanceProvider.getControllerForInstance(sdkInstance)
                    .scheduleInApp(context, suitableInApp, payload, null)
            } else {
                controller.viewHandler.buildAndShowInApp(context, suitableInApp, payload)
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag showGeneralInApp() : " }
        }
    }

    fun getSelfHandledInApp(listener: SelfHandledAvailableListener) {
        try {
            sdkInstance.logger.log { "$tag getSelfHandledInApp() : Will try to get self handled inapp" }
            if (!canShowInApp(context, sdkInstance)) {
                onSelfHandledAvailable(null, listener)
                return
            }
            logCurrentInAppState(context, sdkInstance)
            val suitableInApp =
                getSuitableInApp(InAppInstanceProvider.getCacheForInstance(sdkInstance).selfHandledCampaign)
            if (suitableInApp == null) {
                onSelfHandledAvailable(null, listener)
                return
            }
            sdkInstance.logger.log { "$tag getSelfHandledInApp() : Suitable InApp: $suitableInApp" }
            val payload = getPayloadForCampaign(suitableInApp)
            if (payload == null) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag getSelfHandledInApp() : Payload null" }
                onSelfHandledAvailable(null, listener)
                return
            }
            if (isDelayedInApp(suitableInApp)) {
                sdkInstance.logger.log { "$tag getSelfHandledInApp() : Delayed campaign, scheduling campaign" }
                InAppInstanceProvider.getControllerForInstance(sdkInstance)
                    .scheduleInApp(context, suitableInApp, payload, listener)
            } else {
                onSelfHandledAvailable(payload as NativeCampaignPayload, listener)
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getSelfHandledInApp() : " }
        }
    }

    fun showTriggeredInApp(event: Event, listener: SelfHandledAvailableListener?) {
        try {
            sdkInstance.logger.log { "$tag showTriggeredInApp() : Event: $event" }
            if (!canShowInApp(context, sdkInstance)) return
            logCurrentInAppState(context, sdkInstance)
            val triggerCampaigns = repository.getCampaignsForEvent(event.name)
            val attributes = transformEventAttributesForEvaluationPackage(event.attributes)
            sdkInstance.logger.log { "$tag showTriggeredInApp() : Transformed event attributes - $attributes" }
            val filteredCampaigns = triggerCampaigns.filter { campaign ->
                campaign.campaignMeta.trigger != null &&
                    Evaluator(sdkInstance).evaluateTriggerForEvent(
                        campaign.campaignMeta.trigger,
                        event,
                        enrichAttributesForTriggeredEvaluation(
                            attributes,
                            GlobalCache.getAppMeta(context)
                        )
                    )
            }
            val suitableCampaign = getSuitableInApp(filteredCampaigns) ?: run {
                sdkInstance.logger.log { "$tag showTriggeredInApp() : Not suitable campaign found." }
                return
            }
            sdkInstance.logger.log { "$tag showTriggeredInApp() : suitable campaign: $suitableCampaign, will fetch payload" }
            val payload = getPayloadForCampaign(
                suitableCampaign,
                TriggerRequestMeta(
                    event.name,
                    transformEventAttributesForEvaluationPackage(event.attributes),
                    currentISOTime()
                )
            ) ?: run {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag showTriggeredInApp() : Campaign payload is null" }
                return
            }
            if (isDelayedInApp(suitableCampaign)) {
                sdkInstance.logger.log { "$tag showTriggeredInApp() : Delayed campaign, scheduling campaign" }
                InAppInstanceProvider.getControllerForInstance(sdkInstance)
                    .scheduleInApp(context, suitableCampaign, payload, listener)
            } else {
                if (payload.templateType == InAppConstants.IN_APP_TEMPLATE_TYPE_SELF_HANDLED) {
                    sdkInstance.logger.log { "$tag showTriggeredInApp() : Self handled campaign, will try to notify listener" }
                    onSelfHandledAvailable(payload as NativeCampaignPayload, listener)
                } else {
                    sdkInstance.logger.log { "$tag showTriggeredInApp() : Will build in-app." }
                    controller.viewHandler.buildAndShowInApp(context, suitableCampaign, payload)
                }
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag showTriggeredInApp() : " }
        }
    }

    fun showInAppPreview(campaign: InAppCampaign) {
        try {
            sdkInstance.logger.log { "$tag showInAppPreview() : $campaign" }
            val payload = getPayloadForCampaign(campaign) ?: return
            if (payload.templateType == InAppConstants.IN_APP_TEMPLATE_TYPE_SELF_HANDLED) {
                val listener =
                    InAppInstanceProvider.getCacheForInstance(sdkInstance).selfHandledListener
                        ?: return
                onSelfHandledAvailable(payload as NativeCampaignPayload, listener)
                return
            }
            val activity = InAppModuleManager.getActivity() ?: return
            val view =
                controller.viewHandler.buildInApp(payload, getViewCreationMeta(context))
                    ?: return
            controller.viewHandler.addInAppToViewHierarchy(activity, view, payload)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag showInAppPreview() : " }
        }
    }

    private fun getSuitableInApp(campaigns: List<InAppCampaign>): InAppCampaign? {
        if (campaigns.isEmpty()) {
            sdkInstance.logger.log { "$tag getSuitableInApp() : Not active campaigns passed, no suitable campaign." }
            return null
        }

        return Evaluator(sdkInstance).getEligibleCampaignFromList(
            campaigns,
            repository.getGlobalState(),
            InAppInstanceProvider.getCacheForInstance(sdkInstance).inAppContext,
            context
        )
    }

    private fun getPayloadForCampaign(
        campaign: InAppCampaign,
        triggerMeta: TriggerRequestMeta? = null
    ): CampaignPayload? {
        val payload = repository.fetchCampaignPayload(
            campaign,
            InAppModuleManager.getCurrentActivityName() ?: "",
            InAppInstanceProvider.getCacheForInstance(sdkInstance).inAppContext,
            getDeviceType(context),
            triggerMeta
        )
        sdkInstance.logger.log { "$tag getPayloadForCampaign() : Campaign Payload: $payload" }
        return payload
    }

    /**
     * Builds and shows a delayed in-app campaign if the campaign all passes the eligibility checks.
     *
     * @param campaign instance of [InAppCampaign]
     * @param payload instance of [CampaignPayload]
     * @param listener instance of [SelfHandledAvailableListener]
     */
    fun showDelayInApp(
        campaign: InAppCampaign,
        payload: CampaignPayload,
        listener: SelfHandledAvailableListener?
    ) {
        try {
            sdkInstance.logger.log { "showDelayInApp(): Executing for campaignId:${payload.campaignId}" }
            if (!InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance).isModuleEnabled()) {
                sdkInstance.logger.log {
                    "showDelayInApp(): Module disabled. Cannot show campaign: ${payload.campaignId}."
                }
                return
            }
            if (isCampaignEligibleForDisplay(context, sdkInstance, campaign, payload)) {
                if (payload.templateType == InAppConstants.IN_APP_TEMPLATE_TYPE_SELF_HANDLED) {
                    onSelfHandledAvailable(payload as NativeCampaignPayload, listener)
                } else {
                    ViewHandler(sdkInstance).buildAndShowInApp(context, campaign, payload)
                }
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) {
                "showDelayInApp(): Executing for campaignId: ${payload.campaignId}"
            }
        } finally {
            sdkInstance.logger.log { "showDelayInApp(): Remove campaignId:${payload.campaignId} from cache" }
            InAppInstanceProvider.getCacheForInstance(sdkInstance).scheduledCampaigns.remove(payload.campaignId)
        }
    }

    /**
     * Sends a callback to SelfHandledAvailableListener.
     *
     * @param payload instance of [NativeCampaignPayload]
     * @param listener instance of [SelfHandledAvailableListener]
     */
    private fun onSelfHandledAvailable(
        payload: NativeCampaignPayload?,
        listener: SelfHandledAvailableListener?
    ) {
        sdkInstance.logger.log { "$tag onSelfHandledAvailable() : Payload: $payload, listener:$listener" }
        if (listener == null) {
            sdkInstance.logger.log(LogLevel.ERROR) { "$tag onSelfHandledAvailable() : Listener is null, cannot pass callback" }
            return
        }
        val data = if (payload?.customPayload == null) {
            sdkInstance.logger.log(LogLevel.ERROR) { "$tag onSelfHandledAvailable() : Payload is null" }
            null
        } else {
            SelfHandledCampaignData(
                CampaignData(payload.campaignId, payload.campaignName, payload.campaignContext),
                accountMetaForInstance(sdkInstance),
                SelfHandledCampaign(payload.customPayload, payload.dismissInterval)
            )
        }
        sdkInstance.logger.log { "$tag onSelfHandledAvailable() : Notifying listener, data: $data" }
        postOnMainThread {
            listener.onSelfHandledAvailable(data)
        }
    }

    /**
     * Executes the show non-intrusive nudge Job.
     */
    fun showNudgeInApp(inAppPosition: InAppPosition) {
        try {
            sdkInstance.logger.log { "$tag showNudgeInApp() : " }
            if (!canShowInApp(context, sdkInstance)) return
            logCurrentInAppState(context, sdkInstance)

            val inAppCache = InAppInstanceProvider.getCacheForInstance(sdkInstance)
            val campaigns = if (inAppPosition != InAppPosition.ANY) {
                inAppCache.nonIntrusiveNudgeCampaigns[inAppPosition]
            } else {
                inAppCache.nonIntrusiveNudgeCampaigns.filter { entry -> entry.value.isNotEmpty() }
                    .flatMap { it.value }
            }

            if (campaigns.isNullOrEmpty()) {
                sdkInstance.logger.log { "$tag showNudgeInApp() : No Non-intrusive nudges to process" }
                return
            }
            val suitableInApp: InAppCampaign
            synchronized(InAppModuleManager.showNudgeLock) {
                /*
                Filter out visible or processing nudges present in:
                1. Module Level cache for Visible or Processing nudge positions
                3. Instance Level cache for visible or processing nudge positions
                */

                sdkInstance.logger.log { "$tag showNudgeInApp() : filtering nudges start" }
                val filteredCampaignList = campaigns.filter { campaign ->
                    !(
                        InAppModuleManager.isNudgePositionVisible(campaign.campaignMeta.position) ||
                            InAppModuleManager.isNudgePositionProcessing(campaign.campaignMeta.position) ||
                            inAppCache.isCampaignVisibleOrProcessing(campaign.campaignMeta.campaignId)
                        )
                }

                if (filteredCampaignList.isEmpty()) {
                    sdkInstance.logger.log {
                        "$tag showNudgeInApp() : filteredCampaignList " +
                            "is empty, cannot process further."
                    }
                    return
                }
                suitableInApp = getSuitableInApp(filteredCampaignList) ?: return
                val position = suitableInApp.campaignMeta.position ?: return

                // Update instance level cache for processing nudge campaign.
                inAppCache.addVisibleOrProcessingNonIntrusiveNudge(suitableInApp.campaignMeta.campaignId)
                // Update module level cache for processing position.
                InAppModuleManager.addProcessingNudgePosition(position)
                sdkInstance.logger.log { "$tag showNudgeInApp() : filtering nudges end" }
            }

            sdkInstance.logger.log { "$tag showNudgeInApp() : Suitable InApp $suitableInApp" }
            val payload = getPayloadForCampaign(suitableInApp)

            if (payload == null) {
                inAppCache.removeFromVisibleOrProcessingNonIntrusiveNudge(suitableInApp.campaignMeta.campaignId)
                return
            }

            // Process any pending show nudge calls if any.
            InAppInstanceProvider.getControllerForInstance(sdkInstance)
                .processPendingNudgeDisplayRequest(context)
            if (isDelayedInApp(suitableInApp)) {
                InAppInstanceProvider.getControllerForInstance(sdkInstance)
                    .scheduleInApp(context, suitableInApp, payload, null)
            } else {
                controller.viewHandler.buildAndShowInApp(context, suitableInApp, payload)
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag showNudgeInApp() : " }
        }
    }
}