/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.view.View
import android.view.ViewGroup
import android.view.animation.AnimationUtils
import android.widget.FrameLayout
import com.moengage.core.LogLevel
import com.moengage.core.internal.global.GlobalResources
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.currentISOTime
import com.moengage.inapp.internal.engine.HtmlViewEngine
import com.moengage.inapp.internal.engine.ViewEngine
import com.moengage.inapp.internal.engine.removeNonIntrusiveNudgeFromCache
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.HtmlCampaignPayload
import com.moengage.inapp.internal.model.NativeCampaignPayload
import com.moengage.inapp.internal.model.ViewCreationMeta
import com.moengage.inapp.internal.model.configmeta.InAppConfigMeta
import com.moengage.inapp.internal.model.enums.InAppType
import com.moengage.inapp.internal.model.enums.LifecycleType
import com.moengage.inapp.internal.model.meta.InAppCampaign
import com.moengage.inapp.internal.model.style.ContainerStyle

/**
 * @author Umang Chamaria
 */
internal class ViewHandler(private val sdkInstance: SdkInstance) {

    private val tag = "${MODULE_TAG}ViewHandler"
    private val autoDismissRunnables = mutableMapOf<String, Runnable>()

    fun buildAndShowInApp(context: Context, campaign: InAppCampaign, payload: CampaignPayload) {
        sdkInstance.logger.log { "$tag buildAndShowInApp() : Building campaign, campaignId: ${payload.campaignId}" }
        val viewMeta = getViewCreationMeta(context)
        val view = buildInApp(payload, viewMeta)
        if (view == null) {
            sdkInstance.logger.log {
                "$tag buildAndShowInApp() : Could not create view for in-app campaign ${campaign.campaignMeta.campaignId})"
            }
            removeNonIntrusiveNudgeFromCache(sdkInstance, payload)
            return
        }

        if (canShowInApp(context, campaign, view, payload)) {
            showInApp(view, viewMeta, payload)
        } else {
            removeNonIntrusiveNudgeFromCache(sdkInstance, payload)
        }
    }

    fun buildInApp(payload: CampaignPayload, viewCreationMeta: ViewCreationMeta): View? {
        val activity = InAppModuleManager.getActivity()

        if (activity == null) {
            sdkInstance.logger.log {
                "$tag buildAndShowInApp() : Could not create view for in-app campaign " +
                    "${payload.campaignId},reason: Activity is null."
            }
            return null
        }
        return getInAppView(activity, payload, viewCreationMeta)
    }

    private fun getInAppView(
        activity: Activity,
        campaignPayload: CampaignPayload,
        viewCreationMeta: ViewCreationMeta
    ): View? {
        return when (campaignPayload.inAppType) {
            InAppType.NATIVE -> ViewEngine(
                activity,
                sdkInstance,
                campaignPayload as NativeCampaignPayload,
                viewCreationMeta
            ).createInApp()

            InAppType.HTML -> HtmlViewEngine(
                activity,
                sdkInstance,
                (campaignPayload as HtmlCampaignPayload),
                viewCreationMeta
            ).createInApp()
        }
    }

    private fun showInApp(
        view: View,
        viewCreationMeta: ViewCreationMeta,
        campaignPayload: CampaignPayload
    ) {
        sdkInstance.logger.log { "$tag showInApp() : Will try to show in-app. Campaign id: ${campaignPayload.campaignId}" }
        val activity = InAppModuleManager.getActivity() ?: return
        addInAppToViewHierarchy(activity, view, campaignPayload)
    }

    fun addInAppToViewHierarchy(activity: Activity, view: View, payload: CampaignPayload) {
        addInAppToViewHierarchy(activity, view, payload, false)
    }

    fun addInAppToViewHierarchy(
        activity: Activity,
        view: View,
        payload: CampaignPayload,
        isShowOnConfigChange: Boolean
    ) {
        sdkInstance.logger.log { "$tag addInAppToViewHierarchy() : Attaching campaign: ${payload.campaignId}" }
        GlobalResources.mainThread.post {
            try {
                if (InAppInstanceProvider.getCacheForInstance(sdkInstance)
                    .hasHtmlCampaignSetupFailed
                ) {
                    InAppInstanceProvider.getCacheForInstance(sdkInstance)
                    sdkInstance.logger.log { "$tag addInAppToViewHierarchy() : HTML InApp Creation failed." }
                    return@post
                }
                val root = getWindowRoot(activity)
                InAppModuleManager.addInAppToViewHierarchy(
                    root,
                    view,
                    payload,
                    isShowOnConfigChange
                )
                autoDismissInAppIfRequired(root, payload, view, activity)
                if (!isShowOnConfigChange) {
                    InAppInstanceProvider.getControllerForInstance(sdkInstance)
                        .onInAppShown(activity, payload)
                }
            } catch (t: Throwable) {
                sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag addInAppToViewHierarchy() : " }
            }
        }
    }

    private fun autoDismissInAppIfRequired(
        rootView: FrameLayout,
        payload: CampaignPayload,
        view: View,
        activity: Activity
    ) {
        if (payload.dismissInterval > 0) {
            val runnable = getAutoDismissRunnableForCampaign(rootView, payload, view, activity)
            autoDismissRunnables[payload.campaignId] = runnable
            GlobalResources.mainThread.postDelayed(
                runnable,
                payload.dismissInterval * 1000
            )
        }
    }

    private fun getAutoDismissRunnableForCampaign(
        root: FrameLayout,
        payload: CampaignPayload,
        view: View,
        activity: Activity
    ): Runnable {
        return Runnable {
            if (root.indexOfChild(view) == -1) {
                sdkInstance.logger.log { "$tag getAutoDismissRunnableForCampaign() : InApp was closed  before being dismissed." }
            } else {
                removeViewFromHierarchy(activity, view, payload)
                onAutoDismiss(activity.applicationContext, payload)
            }
        }
    }

    private fun getWindowRoot(activity: Activity): FrameLayout {
        return activity.window
            .decorView
            .findViewById<View>(android.R.id.content)
            .rootView as FrameLayout
    }

    private fun canShowInApp(
        context: Context,
        campaign: InAppCampaign,
        view: View,
        payload: CampaignPayload
    ): Boolean {
        val statsLogger = InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance)

        if (campaign.campaignMeta.templateType != InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE &&
            InAppModuleManager.isInAppVisible
        ) {
            sdkInstance.logger.log(LogLevel.INFO) { "$tag canShowInApp(): Another campaign visible,cannot show campaign ${payload.campaignId}" }
            statsLogger.updateStatForCampaign(
                payload,
                currentISOTime(),
                IMPRESSION_STAGE_ANOTHER_CAMPAIGN_VISIBLE
            )
            return false
        }
        sdkInstance.logger.log(LogLevel.INFO) { "$tag canShowInApp(): will evaluate for campaign ${payload.campaignId}" }
        if (!isCampaignEligibleForDisplay(context, sdkInstance, campaign, payload)) return false
        if (isInAppExceedingScreen(context, view)) {
            sdkInstance.logger.log(LogLevel.INFO) { "$tag canShowInApp() : Cannot show in-app, view dimensions exceed device dimensions." }
            statsLogger.updateStatForCampaign(
                payload,
                currentISOTime(),
                IMPRESSION_STAGE_HEIGHT_EXCEEDS_DEVICE
            )
            return false
        }
        sdkInstance.logger.log(LogLevel.INFO) { "$tag canShowInApp(): success for campaign ${payload.campaignId}" }
        return true
    }

    private fun onAutoDismiss(context: Context, payload: CampaignPayload) {
        sdkInstance.logger.log { "$tag onAutoDismiss() : campaignId: ${payload.campaignId}" }
        handleDismiss(payload)
        trackAutoDismiss(context, sdkInstance, payload)
    }

    @SuppressLint("ResourceType")
    fun removeViewFromHierarchy(
        context: Context,
        inAppView: View,
        campaignPayload: CampaignPayload
    ) {
        try {
            if (campaignPayload.inAppType === InAppType.NATIVE) {
                val nativeCampaignPayload = campaignPayload as NativeCampaignPayload
                val primaryContainer = nativeCampaignPayload.primaryContainer ?: return
                val style = primaryContainer.style as ContainerStyle
                if (style.animation != null &&
                    style.animation.exit != -1
                ) {
                    val animation = AnimationUtils.loadAnimation(context, style.animation.exit)
                    inAppView.animation = animation
                }
            }
            (inAppView.parent as ViewGroup).removeView(inAppView)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag removeViewFromHierarchy() : " }
        }
    }

    fun handleDismiss(campaignPayload: CampaignPayload) {
        if (campaignPayload.templateType == InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE) {
            removeNonIntrusiveNudgeFromCache(sdkInstance, campaignPayload, true)
            ConfigurationChangeHandler.getInstance().removeNudgeInAppFromConfigCache(campaignPayload)
        } else {
            InAppModuleManager.updateInAppVisibility(false)
            ConfigurationChangeHandler.getInstance().clearGeneralInAppFromConfigCache()
        }

        InAppInstanceProvider.getControllerForInstance(sdkInstance)
            .notifyLifecycleChange(campaignPayload, LifecycleType.DISMISS)
    }

    /**
     * Dismiss InApp On Configuration Change
     * @param inAppConfigMeta instance of [InAppConfigMeta]
     */
    fun dismissOnConfigurationChange(inAppConfigMeta: InAppConfigMeta) {
        try {
            sdkInstance.logger.log { "$tag dismissOnConfigurationChange() : $inAppConfigMeta" }
            val view =
                InAppModuleManager.getActivity()?.window?.findViewById<View>(inAppConfigMeta.containerId)
            if (view != null) {
                (view.parent as ViewGroup).removeView(view)
                removeNonIntrusiveNudgeFromCache(sdkInstance, inAppConfigMeta)
            }
            InAppInstanceProvider.getControllerForInstance(sdkInstance).viewHandler
                .removeAutoDismissRunnable(inAppConfigMeta.campaignId)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag dismissOnConfigurationChange() : " }
        }
    }

    fun removeAutoDismissRunnable(campaignId: String) {
        sdkInstance.logger.log { "$tag removeAutoDismissRunnable() : Campaign-id: $campaignId" }
        autoDismissRunnables[campaignId]?.let {
            GlobalResources.mainThread.removeCallbacks(it)
        }
        // Need to Assign null to get Garbage Collected
        autoDismissRunnables.remove(campaignId)
    }

    /**
     * Clears auto-dismiss runnables
     */
    fun clearAutoDismissRunnables() {
        try {
            sdkInstance.logger.log {
                "$tag clearAutoDismissRunnables() : will clear all auto dimiss " +
                    "runnables"
            }

            autoDismissRunnables.let {
                try {
                    for ((campaignId, runnable) in it) {
                        sdkInstance.logger.log { "$tag clearAutoDismissRunnables() : removing auto dismiss runnable for Campaign-id: $campaignId" }
                        GlobalResources.mainThread.removeCallbacks(runnable)
                    }
                } catch (t: Throwable) {
                    sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag clearAutoDismissRunnables() : " }
                }
            }
            autoDismissRunnables.clear()
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag clearAutoDismissRunnables() : " }
        }
    }
}