/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.engine

import android.annotation.SuppressLint
import android.app.Activity
import android.graphics.Color
import android.view.KeyEvent
import android.view.View
import android.view.ViewGroup
import android.widget.RelativeLayout
import androidx.annotation.MainThread
import androidx.annotation.Nullable
import androidx.annotation.WorkerThread
import androidx.core.view.ViewCompat
import com.moengage.core.LogLevel
import com.moengage.core.internal.global.GlobalCache
import com.moengage.core.internal.logger.Logger
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.canUseWebView
import com.moengage.inapp.internal.ActionHandler
import com.moengage.inapp.internal.IMPRESSION_STAGE_FILE_DOWNLOAD_FAILURE
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.InAppInstanceProvider
import com.moengage.inapp.internal.html.HtmlJavaScriptInterface
import com.moengage.inapp.internal.html.InAppWebView
import com.moengage.inapp.internal.html.InAppWebViewClient
import com.moengage.inapp.internal.model.HtmlCampaignPayload
import com.moengage.inapp.internal.model.ViewCreationMeta
import com.moengage.inapp.internal.model.actions.DismissAction
import com.moengage.inapp.internal.repository.InAppFileManager
import com.moengage.inapp.model.enums.ActionType

/**
 * @author Arshiya Khanum
 */
private const val INTERFACE_NAME = "moengageInternal"
private const val HTML_MIME_TYPE = "text/html"
private const val HTML_ENCODING = "utf-8"
private const val FILE_URI_SCHEME_PREFIX = "file://"
private const val FILE_PATH_SEPARATOR = "/"

@Suppress("PrivatePropertyName")
public class HtmlViewEngine(
    activity: Activity,
    private val sdkInstance: SdkInstance,
    private val payload: HtmlCampaignPayload,
    viewCreationMeta: ViewCreationMeta
) : BaseViewEngine(activity, payload, viewCreationMeta) {

    private val tag = "${InAppConstants.MODULE_TAG}HtmlViewEngine"

    private var inAppView: View? = null
    private val statusBarHeight: Int = viewCreationMeta.statusBarHeight
    private val parentViewDimensions = viewCreationMeta.deviceDimensions

    @WorkerThread
    @Nullable
    override fun createInApp(): View? {
        sdkInstance.logger.log { "$tag createInApp() : Will try to create in-app view for campaign-id: ${payload.campaignId}" }
        sdkInstance.logger.log { "$tag createInApp() : Device Dimensions: $parentViewDimensions, status bar height: $statusBarHeight" }
        if (!canUseWebView(activity.applicationContext)) {
            sdkInstance.logger.log { "$tag createInApp() : Web-view cannot be used, either does not exist or is disabled." }
            return null
        }
        if (downloadAssets()) {
            inAppView = createPrimaryContainer()
        }

        return inAppView
    }

    private fun downloadAssets(): Boolean {
        payload.htmlAssets ?: return true

        val assets = payload.htmlAssets.assets
        val count =
            InAppFileManager(activity, sdkInstance)
                .downloadAndSaveHtmlAssets(payload.campaignId, assets)
        if (count != assets.size) {
            updateStatForCampaign(
                campaignPayload,
                IMPRESSION_STAGE_FILE_DOWNLOAD_FAILURE,
                sdkInstance
            )
            sdkInstance.logger.log(LogLevel.ERROR) { "$tag downloadAssets() : Assets download failed, cannot show inapp." }
            return false
        }
        return true
    }

    private fun createPrimaryContainer(): View {
        val containerLayout = RelativeLayout(activity)
        containerLayout.id = InAppConstants.HTML_CONTAINER_ID
        // The Container layout is not part of the payload for HTML In-Apps, so the size, style and
        // margins will not be a part of the payload received. HTML In-Apps will be full screen by
        // default.
        val layoutParams: RelativeLayout.LayoutParams = RelativeLayout.LayoutParams(
            parentViewDimensions.width,
            RelativeLayout.LayoutParams.MATCH_PARENT
        )
        layoutParams.setMargins(0, statusBarHeight, 0, 0)
        containerLayout.layoutParams = layoutParams

        createWebView(
            containerLayout,
            InAppFileManager(activity, sdkInstance).getHtmlAssetsPath(payload.campaignId)
        )
        handleBackPress(containerLayout)
        return containerLayout
    }

    @MainThread
    @SuppressLint("SetJavaScriptEnabled", "AddJavascriptInterface")
    private fun createWebView(containerLayout: ViewGroup, assetsPath: String) {
        activity.runOnUiThread {
            setUpWebView(assetsPath, containerLayout)
        }
    }

    private fun getBaseUrlPath(assetsPath: String): String {
        return "$FILE_URI_SCHEME_PREFIX$assetsPath$FILE_PATH_SEPARATOR"
    }

    private fun dismissInApp() {
        val view = inAppView ?: return
        ActionHandler(activity, sdkInstance).onActionPerformed(
            view,
            DismissAction(ActionType.DISMISS),
            payload
        )
    }

    private fun handleBackPress(inAppView: View) {
        sdkInstance.logger.log { "$tag handleBackPress() : " }
        inAppView.isClickable = true
        inAppView.isFocusable = true
        inAppView.isFocusableInTouchMode = true
        inAppView.requestFocus()
        inAppView.setOnFocusChangeListener { v, hasFocus ->
            sdkInstance.logger.log { "$tag onFocusChanged() : ${v.id} : $hasFocus ${v.findFocus()?.id}" }
        }
        inAppView.setOnKeyListener { _, keyCode, event ->
            try {
                sdkInstance.logger.log { "$tag inAppView() : onKey() : $keyCode ${event.action}" }
                if (event.action == KeyEvent.ACTION_DOWN && keyCode == KeyEvent.KEYCODE_BACK) {
                    sdkInstance.logger.log { "$tag handleBackPress() : on back button pressed" }
                    dismissInApp()
                    return@setOnKeyListener true
                }
            } catch (e: Exception) {
                sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag onKey() : " }
            }
            false
        }
    }

    private fun setUpWebView(assetsPath: String, containerLayout: ViewGroup) {
        try {
            sdkInstance.logger.log { "$tag setUpWebView() : will create web view." }
            val webView = InAppWebView(activity)
            webView.id = ViewCompat.generateViewId()
            with(webView.settings) {
                javaScriptEnabled = GlobalCache.jsConfig.isJavaScriptEnabled
                useWideViewPort = true
                loadWithOverviewMode = true
                displayZoomControls = false
                domStorageEnabled = true
                // Needed for loading relative path resources from local storage
                allowFileAccess = true
            }
            val layoutParams = RelativeLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT
            )

            with(webView) {
                webViewClient = InAppWebViewClient(payload)
                addJavascriptInterface(
                    HtmlJavaScriptInterface(
                        activity,
                        payload,
                        inAppView,
                        sdkInstance
                    ),
                    INTERFACE_NAME
                )
                loadDataWithBaseURL(
                    getBaseUrlPath(assetsPath),
                    payload.htmlPayload,
                    HTML_MIME_TYPE,
                    HTML_ENCODING,
                    null
                )
                this.layoutParams = layoutParams
                // NOTE: Background color is set explicitly to support in-apps with transparent
                // background color due to limitation in web view wherein the transparent color
                // specified in CSS is ignored.
                setBackgroundColor(Color.TRANSPARENT)
            }
            containerLayout.addView(webView)
        } catch (t: Throwable) {
            Logger.print(LogLevel.ERROR, t) { "$tag createWebView() : " }
            InAppInstanceProvider.getCacheForInstance(sdkInstance).hasHtmlCampaignSetupFailed = true
        }
    }
}