/*
 * Copyright (c) 2014-2023 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.engine

import android.animation.Animator
import android.animation.AnimatorSet
import android.animation.ValueAnimator
import android.content.Context
import android.graphics.Bitmap
import android.media.MediaMetadataRetriever
import android.media.MediaPlayer
import android.net.Uri
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.animation.AccelerateDecelerateInterpolator
import android.view.animation.AnimationUtils
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.RelativeLayout
import androidx.annotation.DrawableRes
import androidx.annotation.GravityInt
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.model.ViewDimension
import com.moengage.inapp.R
import com.moengage.inapp.internal.MODULE_TAG
import com.moengage.inapp.internal.exceptions.CouldNotCreateViewException
import com.moengage.inapp.internal.exceptions.VideoNotFoundException
import com.moengage.inapp.internal.getUnspecifiedViewDimension
import com.moengage.inapp.internal.listeners.OnInAppDisplaySizeChangeListener
import com.moengage.inapp.internal.listeners.VideoPlaybackListener
import com.moengage.inapp.internal.model.InAppWidget
import com.moengage.inapp.internal.model.Margin
import com.moengage.inapp.internal.model.MediaMeta
import com.moengage.inapp.internal.model.NativeCampaignPayload
import com.moengage.inapp.internal.model.Spacing
import com.moengage.inapp.internal.model.ViewCreationMeta
import com.moengage.inapp.internal.model.enums.DisplaySize
import com.moengage.inapp.internal.model.enums.Orientation
import com.moengage.inapp.internal.model.style.ContainerStyle
import com.moengage.inapp.internal.model.style.ImageStyle
import com.moengage.inapp.internal.model.style.InAppStyle
import com.moengage.inapp.internal.repository.InAppFileManager
import com.moengage.inapp.internal.widgets.MoEVideoView
import com.moengage.inapp.model.enums.InAppPosition

private const val CONTROLLER_BUTTON_SIZE = 48
private const val CONTROLLER_BUTTON_PADDING = 8
private const val CONTROLLER_HIDE_INTERVAL = 1500L
private const val RESIZEABLE_NUDGES_DEFAULT_ANIMATION_DURATION = 300L
private const val IS_AUDIO_ENABLED_STATE_YES = "yes"

/**
 * ViewEngine helper class for Non-Intrusive Nudges
 */
internal class NudgesViewEngineHelper(
    private val context: Context,
    private val sdkInstance: SdkInstance,
    private val viewCreationMeta: ViewCreationMeta,
    private val payload: NativeCampaignPayload,
    private val mediaManager: InAppFileManager,
    private val densityScale: Float
) {

    private val tag = "${MODULE_TAG}NudgesViewEngineHelper"
    private lateinit var mediaPlayer: MediaPlayer
    private var onInAppDisplaySizeChangeListener: OnInAppDisplaySizeChangeListener? = null

    /**
     * Provides [Spacing] for non-intrusive nudge according to [position]
     */
    @Throws(CouldNotCreateViewException::class)
    fun transformMarginForInAppPosition(margin: Margin, position: InAppPosition): Spacing {
        val transformedMargin =
            transformMargin(sdkInstance, viewCreationMeta.deviceDimensions, margin)
        sdkInstance.logger.log { "$tag transformMarginForInAppPosition() : Position: $position" }
        sdkInstance.logger.log { "$tag transformMarginForInAppPosition() : viewCreationMeta: $viewCreationMeta" }
        return when (position) {
            InAppPosition.TOP -> {
                Spacing(
                    transformedMargin.left,
                    transformedMargin.right,
                    transformedMargin.top + viewCreationMeta.statusBarHeight,
                    transformedMargin.bottom
                )
            }

            InAppPosition.BOTTOM, InAppPosition.BOTTOM_LEFT, InAppPosition.BOTTOM_RIGHT -> {
                Spacing(
                    transformedMargin.left,
                    transformedMargin.right,
                    transformedMargin.top,
                    transformedMargin.bottom + viewCreationMeta.navigationBarHeight
                )
            }

            else -> throw CouldNotCreateViewException("Unsupported InApp position: $position")
        }
    }

    /**
     * Creates video widget for non-intrusive nudge
     */
    @Throws(CouldNotCreateViewException::class, VideoNotFoundException::class)
    fun createVideoView(
        widget: InAppWidget,
        parentOrientation: Orientation,
        primaryContainerLayout: RelativeLayout,
        toExclude: ViewDimension
    ): View {
        sdkInstance.logger.log { "$tag createVideoView() : Will create this widget: $widget" }
        val primaryContainer = payload.primaryContainer
            ?: throw CouldNotCreateViewException("Primary container is not defined.")

        val primaryContainerStyle = primaryContainer.style as ContainerStyle
        if (primaryContainerStyle.displaySize == null) {
            throw CouldNotCreateViewException("'displaySize' is not defined for primary container.")
        }

        // Media container [FrameLayout] which hold MoEVideoView and media controller
        val videoContainer = FrameLayout(context)

        val videoView = MoEVideoView(context)
        val videoUri =
            mediaManager.getVideoFromUrl(widget.component.content, payload.campaignId)
                ?: throw VideoNotFoundException("Error while fetching video from url: ${widget.component.content}")

        // Add Video to video view
        videoView.setVideoURI(videoUri)
        // Retrieve video related meta
        val mediaMeta = getVideoMeta(videoUri)

        val campaignViewDimension = getViewDimensionsFromPercentage(
            viewCreationMeta.deviceDimensions,
            primaryContainerStyle
        )
        sdkInstance.logger.log { "$tag createVideoView(): Campaign Dimension: $campaignViewDimension" }

        sdkInstance.logger.log { "$tag createVideoView(): Video Dimension: ${mediaMeta.dimension}" }

        // Calculate initial media container dimension according to displaySize
        when (primaryContainerStyle.displaySize) {
            DisplaySize.FULLSCREEN -> {
                val fullScreenDimension = getFullScreenViewDimension(primaryContainerStyle)
                sdkInstance.logger.log {
                    "$tag createVideoView(): fullScreen dimension: $fullScreenDimension"
                }
                campaignViewDimension.width = fullScreenDimension.width
                campaignViewDimension.height =
                    mediaMeta.dimension.height * campaignViewDimension.width / mediaMeta.dimension.width
            }

            DisplaySize.MINIMISED ->
                campaignViewDimension.height =
                    mediaMeta.dimension.height * campaignViewDimension.width / mediaMeta.dimension.width
        }
        sdkInstance.logger.log {
            "$tag createVideoView(): final computed dimension: $campaignViewDimension"
        }

        // NOTE: This is only for nudges in order to fix the background image with border and
        // rounded corners issue
        campaignViewDimension.width -= toExclude.width

        val videoViewLayoutParams = FrameLayout.LayoutParams(
            campaignViewDimension.width,
            campaignViewDimension.height
        )
        videoViewLayoutParams.gravity = Gravity.CENTER
        videoView.layoutParams = videoViewLayoutParams
        videoContainer.addView(videoView)

        // Media controller [FrameLayout] which holds Video controllers Play, Pause and DisplaySize
        // controllers [DisplaySize.FULL_SCREEN], [DisplaySize.MINIMISED]
        val controllerLayout = getVideoController(
            videoView,
            primaryContainerLayout,
            videoContainer,
            mediaMeta,
            primaryContainerStyle.displaySize
        )
        videoView.setOnPreparedListener { mediaPlayer ->
            sdkInstance.logger.log {
                (
                    "$tag createVideoView(): onPrepareListener(): currentPosition= " + mediaPlayer
                        .currentPosition +
                        " videoHeight= " + mediaPlayer.videoHeight +
                        " videoWidth= " + mediaPlayer.videoWidth +
                        " aspectRatio= " + mediaPlayer.videoWidth / mediaPlayer.videoHeight.toFloat()
                    )
            }
            mediaPlayer.setVideoScalingMode(MediaPlayer.VIDEO_SCALING_MODE_SCALE_TO_FIT_WITH_CROPPING)
            this.mediaPlayer = mediaPlayer

            // Set volume if the video has audio
            updateVolume(!mediaMeta.hasAudio)
            showMediaController(controllerLayout, true)
            // NOTE: set the height and width of the videView
            // The android:height value can be different than the layout_height
            // (android:height < layout_height) initially and the system will not resize it
            // according to the view group dimension. This can lead a gap between the parent
            // layout and the videoView.
            when (primaryContainerStyle.displaySize) {
                DisplaySize.FULLSCREEN -> {
                    val fullScreenDimension = getFullScreenViewDimension(primaryContainerStyle)
                    videoView.layoutParams.width = fullScreenDimension.width
                    videoView.layoutParams.height =
                        mediaPlayer.videoHeight * fullScreenDimension.width / mediaPlayer.videoWidth
                }

                DisplaySize.MINIMISED -> {
                    videoView.layoutParams.width = mediaPlayer.videoWidth
                    videoView.layoutParams.height = mediaPlayer.videoHeight
                }
            }
        }
        videoView.addOnAttachStateChangeListener(object : View.OnAttachStateChangeListener {
            override fun onViewAttachedToWindow(view: View) {
                sdkInstance.logger.log {
                    "$tag createVideoView(): view attached to window now " +
                        "playing video"
                }
                videoView.start()
            }

            override fun onViewDetachedFromWindow(view: View) {
                sdkInstance.logger.log {
                    "$tag createVideoView(): view detached from window now " +
                        "pausing video"
                }
                videoView.pause()
            }
        })
        videoContainer.setOnClickListener { showMediaController(controllerLayout, true) }
        videoView.setOnCompletionListener {
            videoView.pause()
            showMediaController(controllerLayout, false)
        }
        videoContainer.addView(controllerLayout)

        val videoContainerParams = LinearLayout.LayoutParams(
            LinearLayout.LayoutParams.MATCH_PARENT,
            LinearLayout.LayoutParams.WRAP_CONTENT
        )
        videoContainerParams.weight = 0.9f
        setLayoutGravity(videoContainerParams, parentOrientation)
        videoContainer.layoutParams = videoContainerParams
        sdkInstance.logger.log { "$tag createVideoView() : created widget: $widget" }
        return videoContainer
    }

    /**
     * Provides Video widget controller layout containing video controllers (Play/Pause)
     * and [displaySize] controllers.
     *
     * @return [FrameLayout]
     */
    private fun getVideoController(
        videoView: MoEVideoView,
        primaryContainer: RelativeLayout,
        videoContainer: FrameLayout,
        mediaMeta: MediaMeta,
        displaySize: DisplaySize
    ): FrameLayout {
        sdkInstance.logger.log { "$tag getVideoController() : Will create video controller" }
        val controllerContainer = FrameLayout(context)

        // Prepare Play Button controller
        val playButton = getControllerButton(
            Gravity.CENTER,
            R.drawable.moengage_inapp_play
        )
        playButton.setOnClickListener { videoView.start() }
        playButton.visibility = View.GONE
        controllerContainer.addView(playButton)

        // Prepare Pause Button controller
        val pauseButton = getControllerButton(
            Gravity.CENTER,
            R.drawable.moengage_inapp_pause
        )

        pauseButton.setOnClickListener { videoView.pause() }
        pauseButton.visibility = View.GONE
        controllerContainer.addView(pauseButton)

        // Handle visibility of play and pause controllers on onStart() and onPause() of videoView
        videoView.setVideoPlaybackListener(
            object : VideoPlaybackListener {
                override fun onStart() {
                    if (!videoView.isPlaying) return
                    playButton.visibility = View.GONE
                    pauseButton.visibility = View.VISIBLE
                }

                override fun onPause() {
                    if (videoView.isPlaying) return
                    pauseButton.visibility = View.GONE
                    playButton.visibility = View.VISIBLE
                }
            }
        )

        // Handle visibility of the audio controller according to hasAudio
        if (mediaMeta.hasAudio) {
            // Prepare mute button controller
            val muteButton = getControllerButton(
                Gravity.BOTTOM or Gravity.START,
                R.drawable.moengage_inapp_mute
            )

            // Prepares unMute button controller
            val unmuteButton = getControllerButton(
                Gravity.BOTTOM or Gravity.START,
                R.drawable.moengage_inapp_unmute
            )

            // Add listener only if the video has audio
            muteButton.setOnClickListener {
                updateVolume(true)
                handleAudioController(true, muteButton, unmuteButton)
            }
            unmuteButton.setOnClickListener {
                updateVolume(false)
                handleAudioController(false, muteButton, unmuteButton)
            }

            controllerContainer.addView(muteButton)
            controllerContainer.addView(unmuteButton)
            handleAudioController(false, muteButton, unmuteButton)
        }

        // Attach displaySize controllers
        attachDisplaySizeControllers(
            primaryContainer,
            videoContainer,
            mediaMeta.dimension,
            displaySize,
            controllerContainer,
            videoView
        )

        val controllerContainerLayoutParams = FrameLayout.LayoutParams(
            FrameLayout.LayoutParams.MATCH_PARENT,
            FrameLayout.LayoutParams.MATCH_PARENT
        )
        controllerContainerLayoutParams.gravity = Gravity.CENTER
        controllerContainer.layoutParams = controllerContainerLayoutParams
        controllerContainer.visibility = View.GONE
        sdkInstance.logger.log { "$tag getVideoController() : completed" }
        return controllerContainer
    }

    /**
     * Creates Image widget for Non-Intrusive nudges for Gif template support
     */
    fun createContainerForResizeableImageView(
        primaryContainerLayout: RelativeLayout,
        imageView: ImageView,
        imageStyle: ImageStyle,
        displaySize: DisplaySize
    ): FrameLayout {
        sdkInstance.logger.log { "$tag createContainerForResizeableImageView() : will create image view" }
        // Media container [FrameLayout] which hold imageView for gif template and media controller
        val imageViewContainer = FrameLayout(context)

        imageViewContainer.addView(imageView)

        // Media controller [FrameLayout] containing displaySize controllers.
        val mediaController: FrameLayout = getImageController(
            primaryContainerLayout,
            imageViewContainer,
            ViewDimension(imageStyle.realWidth.toInt(), imageStyle.realHeight.toInt()),
            displaySize,
            imageView
        )
        imageViewContainer.addView(mediaController)
        imageViewContainer.setOnClickListener {
            showMediaController(mediaController, true)
        }
        // show media controller
        showMediaController(mediaController, true)
        sdkInstance.logger.log { "$tag createContainerForResizeableImageView() : completed" }
        return imageViewContainer
    }

    /**
     * Provides image controller layout for non-intrusive nudge with gif template
     * containing [displaySize] controllers.
     *
     * @return [FrameLayout]
     */
    private fun getImageController(
        primaryContainer: RelativeLayout,
        imageContainer: FrameLayout,
        imageDimension: ViewDimension,
        displaySize: DisplaySize,
        imageView: ImageView
    ): FrameLayout {
        sdkInstance.logger.log { "$tag createImageController(): Will create the image/gif controller" }
        val controllerContainer = FrameLayout(context)
        attachDisplaySizeControllers(
            primaryContainer,
            imageContainer,
            imageDimension,
            displaySize,
            controllerContainer,
            imageView
        )
        val layoutParams = LinearLayout.LayoutParams(
            ViewGroup.LayoutParams.MATCH_PARENT,
            ViewGroup.LayoutParams.MATCH_PARENT
        )
        layoutParams.gravity = Gravity.CENTER
        controllerContainer.layoutParams = layoutParams
        controllerContainer.visibility = View.GONE
        sdkInstance.logger.log {
            "$tag createImageController(): Will create the image/gif controller"
        }
        return controllerContainer
    }

    /**
     * Adds minimized and fullscreen controllers to the media controller
     */
    private fun attachDisplaySizeControllers(
        primaryContainer: RelativeLayout,
        mediaContainer: FrameLayout,
        mediaDimension: ViewDimension,
        displaySize: DisplaySize,
        controllerContainer: FrameLayout,
        mediaView: View
    ) {
        sdkInstance.logger.log {
            "$tag addScreenControllers(): Will try to add displaySize controllers to media controller"
        }

        // Prepares fullScreen controller
        val fullscreenController = getControllerButton(
            Gravity.BOTTOM or Gravity.END,
            R.drawable.moengage_inapp_fullscreen
        )

        // Prepares minimise controller
        val minimiseController = getControllerButton(
            Gravity.BOTTOM or Gravity.END,
            R.drawable.moengage_inapp_minimise
        )

        fullscreenController.setOnClickListener {
            val animator = getResizeValueAnimator(
                primaryContainer,
                mediaContainer,
                mediaDimension,
                DisplaySize.FULLSCREEN,
                mediaView
            )
            animator.addListener(object : Animator.AnimatorListener {
                override fun onAnimationStart(animation: Animator) {
                    onInAppDisplaySizeChangeListener?.onDisplaySizeChangeStart(DisplaySize.MINIMISED)
                }

                override fun onAnimationEnd(animation: Animator) {
                    val primaryContainerLayoutParams =
                        primaryContainer.layoutParams as FrameLayout.LayoutParams
                    primaryContainerLayoutParams.gravity =
                        primaryContainerLayoutParams.gravity or Gravity.TOP
                    primaryContainerLayoutParams.height = FrameLayout.LayoutParams.MATCH_PARENT
                    primaryContainer.layoutParams = primaryContainerLayoutParams

                    val mediaParent = mediaContainer.parent as View
                    val mediaParentLayoutParams = mediaParent.layoutParams
                    mediaParentLayoutParams.height = FrameLayout.LayoutParams.MATCH_PARENT
                    mediaParent.layoutParams = mediaParentLayoutParams
                    // Toggle screen controller visibility. Hide fullscreen controller and show
                    // minimize controller.
                    fullscreenController.visibility = View.GONE
                    minimiseController.visibility = View.VISIBLE
                    animator.removeListener(this)
                    onInAppDisplaySizeChangeListener?.onDisplaySizeChangeEnd(DisplaySize.FULLSCREEN)
                }

                override fun onAnimationCancel(animation: Animator) {}
                override fun onAnimationRepeat(animation: Animator) {}
            })
            animator.start()
        }
        controllerContainer.addView(fullscreenController)

        minimiseController.setOnClickListener {
            val animator = getResizeValueAnimator(
                primaryContainer,
                mediaContainer,
                mediaDimension,
                DisplaySize.MINIMISED,
                mediaView
            )
            animator.addListener(object : Animator.AnimatorListener {
                override fun onAnimationStart(animation: Animator) {
                    setLayoutGravity(
                        sdkInstance,
                        primaryContainer.layoutParams as FrameLayout.LayoutParams,
                        payload.position
                    )
                    onInAppDisplaySizeChangeListener?.onDisplaySizeChangeStart(DisplaySize.FULLSCREEN)
                }

                override fun onAnimationEnd(animation: Animator) {
                    // Toggle screen controller visibility. Hide minimize controller and show
                    // fullscreen controller
                    minimiseController.visibility = View.GONE
                    fullscreenController.visibility = View.VISIBLE
                    animator.removeListener(this)
                    onInAppDisplaySizeChangeListener?.onDisplaySizeChangeEnd(DisplaySize.MINIMISED)
                }

                override fun onAnimationCancel(animation: Animator) {}
                override fun onAnimationRepeat(animation: Animator) {}
            })
            animator.start()
        }
        controllerContainer.addView(minimiseController)

        // Handle initial visibility of display size controller according to [displaySize] set.
        when (displaySize) {
            DisplaySize.FULLSCREEN -> {
                minimiseController.visibility = View.VISIBLE
                fullscreenController.visibility = View.GONE
            }

            DisplaySize.MINIMISED -> {
                minimiseController.visibility = View.GONE
                fullscreenController.visibility = View.VISIBLE
            }
        }
        sdkInstance.logger.log {
            "$tag addScreenControllers(): displaySize controllers added successfully. Default " +
                "displaySize: $displaySize"
        }
    }

    /**
     * Provides animator for animating non-intrusive nudge to [displaySize]
     *
     * @return [ValueAnimator]
     */
    @Throws(CouldNotCreateViewException::class)
    private fun getResizeValueAnimator(
        primaryContainerLayout: RelativeLayout,
        mediaContainer: FrameLayout,
        mediaDimension: ViewDimension,
        displaySize: DisplaySize,
        mediaView: View
    ): AnimatorSet {
        sdkInstance.logger.log {
            "$tag getResizeValueAnimator(): will try build animator " +
                "according to displaySize=$displaySize"
        }

        val primaryContainer = payload.primaryContainer
            ?: throw CouldNotCreateViewException("primary container not defined")

        val initialContainerDimension = ViewDimension(
            primaryContainerLayout.layoutParams.width,
            primaryContainerLayout.layoutParams.height
        )
        if (initialContainerDimension.height == FrameLayout.LayoutParams.WRAP_CONTENT) {
            initialContainerDimension.height =
                getUnspecifiedViewDimension(primaryContainerLayout).height
        }
        sdkInstance.logger.log { "$tag getResizeValueAnimator(): initial view dimension=$initialContainerDimension" }

        val fullScreenMediaDimension = getFullScreenViewDimension(primaryContainer.style)
        fullScreenMediaDimension.height =
            mediaDimension.height * fullScreenMediaDimension.width / mediaDimension.width
        sdkInstance.logger.log { "$tag getResizeValueAnimator(): fullscreen video dimension=$fullScreenMediaDimension" }

        val minimisedMediaDimension = getViewDimensionsFromPercentage(
            viewCreationMeta.deviceDimensions,
            primaryContainer.style
        )
        sdkInstance.logger.log { "$tag getResizeValueAnimator(): minimised video dimension=$minimisedMediaDimension" }
        minimisedMediaDimension.height =
            mediaDimension.height * minimisedMediaDimension.width / mediaDimension.width

        // Calculate target view dimension according to [displaySize]
        val targetContainerDimension = when (displaySize) {
            DisplaySize.FULLSCREEN -> getFullScreenViewDimension(primaryContainer.style)
            DisplaySize.MINIMISED -> getViewDimensionsFromPercentage(
                viewCreationMeta.deviceDimensions,
                primaryContainer.style
            )
        }

        sdkInstance.logger.log { "$tag getResizeValueAnimator(): target view dimension=$targetContainerDimension" }

        val containerValueAnimator = ValueAnimator.ofFloat(0f, 1f)
        containerValueAnimator.addUpdateListener { animation: ValueAnimator ->
            updateContainerAnimatedDimension(
                primaryContainerLayout,
                mediaContainer,
                initialContainerDimension,
                targetContainerDimension,
                animation.animatedFraction,
                displaySize
            )
        }

        val videoValueAnimator = ValueAnimator.ofFloat(0f, 1f)
        videoValueAnimator.addUpdateListener { animation: ValueAnimator ->
            when (displaySize) {
                DisplaySize.FULLSCREEN -> {
                    updateViewAnimatedDimension(
                        mediaView,
                        minimisedMediaDimension,
                        fullScreenMediaDimension,
                        animation.animatedFraction
                    )
                }

                DisplaySize.MINIMISED -> {
                    updateViewAnimatedDimension(
                        mediaView,
                        fullScreenMediaDimension,
                        minimisedMediaDimension,
                        animation.animatedFraction
                    )
                }
            }
        }
        sdkInstance.logger.log { "$tag getResizeValueAnimator(): completed" }

        val animatorSet = AnimatorSet()
        animatorSet.playTogether(containerValueAnimator, videoValueAnimator)
        animatorSet.interpolator = AccelerateDecelerateInterpolator()
        animatorSet.duration = RESIZEABLE_NUDGES_DEFAULT_ANIMATION_DURATION
        return animatorSet
    }

    /**
     * Handles the updates for [ViewGroup.LayoutParams] for primary container and media container
     * of non-intrusive nudges according to the [ValueAnimator.getAnimatedFraction] and [DisplaySize]
     */
    private fun updateContainerAnimatedDimension(
        primaryContainer: RelativeLayout,
        mediaContainer: FrameLayout,
        initialViewDimension: ViewDimension,
        targetViewDimension: ViewDimension,
        fraction: Float,
        animateToDisplaySize: DisplaySize
    ) {
        sdkInstance.logger.log {
            "$tag updateContainerAnimatedDimension(): will update the dimension for " +
                "fraction=$fraction and animating to displaySize: $animateToDisplaySize"
        }

        // Calculate current width and height of the media container according to fraction.
        val currentWidth =
            (initialViewDimension.width + (targetViewDimension.width - initialViewDimension.width) * fraction).toInt()
        val currentHeight =
            (initialViewDimension.height + (targetViewDimension.height - initialViewDimension.height) * fraction).toInt()
        sdkInstance.logger.log { "$tag updateContainerAnimatedDimension(): currentWidth= $currentWidth currentHeight=$currentHeight" }

        // Update LayoutParams of media container to resize the media container.
        val mediaContainerLayoutParams = mediaContainer.layoutParams
        mediaContainerLayoutParams.width = currentWidth
        mediaContainerLayoutParams.height = currentHeight

        /*
        Update LayoutParams of parent of media container. So, that it can wrap the media
        container.
         */
        val mediaParent = mediaContainer.parent as View
        val mediaParentLayoutParams = mediaParent.layoutParams
        mediaParentLayoutParams.width = currentWidth
        if (animateToDisplaySize == DisplaySize.FULLSCREEN) {
            mediaParentLayoutParams.height = currentHeight
        } else {
            mediaParentLayoutParams.height = ViewGroup.LayoutParams.WRAP_CONTENT
        }

        /*
        Finally update the LayoutParams for primary container. So that it can wrap the media
        container.
         */
        val primaryContainerLayoutParams = primaryContainer.layoutParams
        primaryContainerLayoutParams.width = currentWidth
        if (animateToDisplaySize == DisplaySize.FULLSCREEN) {
            primaryContainerLayoutParams.height = currentHeight
        } else {
            primaryContainerLayoutParams.height = RelativeLayout.LayoutParams.WRAP_CONTENT
        }
        sdkInstance.logger.log {
            "$tag updateContainerAnimatedDimension(): updated dimensions for " +
                "fraction=$fraction and animating to displaySize: $animateToDisplaySize"
        }
    }

    /**
     * Handles media controller visibility for provided [controllerView].
     */
    private fun showMediaController(controllerView: View, autoDismiss: Boolean) {
        sdkInstance.logger.log { "$tag showMediaController(): " }
        if (controllerView.visibility == View.VISIBLE) return

        val fadeInAnimation = AnimationUtils.loadAnimation(context, R.anim.fade_in)
        fadeInAnimation.duration = RESIZEABLE_NUDGES_DEFAULT_ANIMATION_DURATION
        fadeInAnimation.fillAfter = true
        controllerView.animation = fadeInAnimation
        controllerView.visibility = View.VISIBLE

        if (autoDismiss) {
            try {
                // To make controller fade out after delay = [CONTROLLER_HIDE_INTERVAL]
                controllerView.postDelayed({
                    if (controllerView.visibility == View.GONE) return@postDelayed
                    val fadeoutAnimation = AnimationUtils.loadAnimation(context, R.anim.fade_out)
                    fadeoutAnimation.duration = RESIZEABLE_NUDGES_DEFAULT_ANIMATION_DURATION
                    fadeoutAnimation.fillAfter = false
                    controllerView.animation = fadeoutAnimation
                    controllerView.visibility = View.GONE
                }, CONTROLLER_HIDE_INTERVAL)
            } catch (t: Throwable) {
                sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag showMediaController(): " }
            }
        }
        sdkInstance.logger.log { "$tag showMediaController(): completed" }
    }

    /**
     * Provide view dimension containing width and height of a non-intrusive nudge for
     * [DisplaySize.FULLSCREEN]
     *
     * @return [ViewDimension]
     */
    fun getFullScreenViewDimension(
        primaryContainerStyle: InAppStyle
    ): ViewDimension {
        val spacingMargin: Spacing = transformMargin(
            sdkInstance,
            viewCreationMeta.deviceDimensions,
            primaryContainerStyle.margin
        )

        return ViewDimension(
            viewCreationMeta.deviceDimensions.width - spacingMargin.left - spacingMargin.right,
            viewCreationMeta.deviceDimensions.height - spacingMargin.top - spacingMargin.bottom - viewCreationMeta.statusBarHeight
        )
    }

    /**
     * Provide view for Media Controller button with defined [gravity] and [res]
     *
     * @return [ImageView]
     */
    private fun getControllerButton(
        @GravityInt gravity: Int,
        @DrawableRes resId: Int
    ): ImageView {
        sdkInstance.logger.log { "$tag getControllerButton() : " }
        val imageBitmap: Bitmap? = generateBitmapFromRes(sdkInstance, context, resId)

        check(imageBitmap != null) {
            "getControllerButton() : Couldn't create controller button, imageBitmap is null."
        }
        val imageView = ImageView(context)
        val dimension: Int = (CONTROLLER_BUTTON_SIZE * densityScale).toInt()
        val imageDimension = ViewDimension(
            dimension,
            dimension
        )
        imageView.setImageBitmap(imageBitmap)
        val layoutParams = FrameLayout.LayoutParams(
            imageDimension.width,
            imageDimension.height
        )
        layoutParams.gravity = gravity

        val padding = (CONTROLLER_BUTTON_PADDING * densityScale).toInt()
        imageView.setPadding(padding, padding, padding, padding)
        imageView.layoutParams = layoutParams
        imageView.isClickable = true

        sdkInstance.logger.log { "$tag getControllerButton() : completed" }
        return imageView
    }

    /**
     * Provides [MediaMeta]  retrieved from [MediaMetadataRetriever]. Container video dimension
     * and hasAudio
     */
    @Throws(CouldNotCreateViewException::class)
    private fun getVideoMeta(uri: Uri): MediaMeta {
        sdkInstance.logger.log { "$tag getVideoMeta() : " }
        val mediaMetadataRetriever = MediaMetadataRetriever()
        try {
            mediaMetadataRetriever.setDataSource(context, uri)
            val videoWidth = mediaMetadataRetriever.extractMetadata(
                MediaMetadataRetriever.METADATA_KEY_VIDEO_WIDTH
            ) ?: throw CouldNotCreateViewException("unable to parse video width")

            val videoHeight = mediaMetadataRetriever.extractMetadata(
                MediaMetadataRetriever.METADATA_KEY_VIDEO_HEIGHT
            ) ?: throw CouldNotCreateViewException("unable to parse video height")

            val hasAudio =
                mediaMetadataRetriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_HAS_AUDIO)
                    ?.equals(IS_AUDIO_ENABLED_STATE_YES, true) ?: false

            val metadata =
                MediaMeta(ViewDimension(videoWidth.toInt(), videoHeight.toInt()), hasAudio)
            sdkInstance.logger.log { "$tag getVideoMeta() : metadata: $metadata" }
            return metadata
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag unable to fetch video dimensions" }
            throw CouldNotCreateViewException("unable to fetch video dimensions")
        } finally {
            mediaMetadataRetriever.release()
        }
    }

    /**
     * Starts audio playback if [isMute] is false, otherwise stops the audio playback
     */
    @Suppress("SENSELESS_COMPARISON")
    private fun updateVolume(isMute: Boolean) {
        sdkInstance.logger.log {
            "$tag setVolume(): will try to update the media state to isMute=$isMute"
        }
        if (mediaPlayer == null) {
            sdkInstance.logger.log { "$tag setVolume(): media player not initialized" }
            return
        }
        if (isMute) {
            mediaPlayer.setVolume(0f, 0f)
        } else {
            mediaPlayer.setVolume(1f, 1f)
        }
        sdkInstance.logger.log {
            "$tag setVolume(): updated media state to isMute=$isMute"
        }
    }

    /**
     * Handles the visibility of audio controller
     */
    private fun handleAudioController(isMute: Boolean, muteButton: View, unmuteButton: View) {
        if (isMute) {
            muteButton.visibility = View.GONE
            unmuteButton.visibility = View.VISIBLE
        } else {
            muteButton.visibility = View.VISIBLE
            unmuteButton.visibility = View.GONE
        }
    }

    @Throws(CouldNotCreateViewException::class)
    fun setPrimaryContainerDimensions(
        containerLayout: RelativeLayout,
        containerStyle: ContainerStyle,
        campaignDimensions: ViewDimension
    ) {
        sdkInstance.logger.log {
            "$tag setPrimaryContainerDimensions() : will set primary " +
                "container dimensions for nudges."
        }
        val nudgeLayoutParams: FrameLayout.LayoutParams

        when (containerStyle.displaySize) {
            DisplaySize.FULLSCREEN -> {
                val fullScreenDimension = getFullScreenViewDimension(containerStyle)
                sdkInstance.logger.log {
                    "$tag setPrimaryContainerDimensions() : fullScreen dimension: $fullScreenDimension"
                }
                campaignDimensions.width = fullScreenDimension.width
                campaignDimensions.height = fullScreenDimension.height
                nudgeLayoutParams = FrameLayout.LayoutParams(
                    campaignDimensions.width,
                    ViewGroup.LayoutParams.MATCH_PARENT
                )
            }

            else -> {
                nudgeLayoutParams = FrameLayout.LayoutParams(
                    campaignDimensions.width,
                    ViewGroup.LayoutParams.WRAP_CONTENT
                )
            }
        }

        val spacingMargin: Spacing = transformMarginForInAppPosition(
            containerStyle.margin,
            payload.position
        )
        // Update primaryContainer FrameLayout.LayoutParams.gravity according to InAppPosition
        setLayoutGravity(sdkInstance, nudgeLayoutParams, payload.position)

        // Update the margin according to the displaySize
        when (containerStyle.displaySize) {
            DisplaySize.FULLSCREEN -> {
                nudgeLayoutParams.gravity = nudgeLayoutParams.gravity or Gravity.TOP
                nudgeLayoutParams.setMargins(
                    spacingMargin.left,
                    spacingMargin.top + viewCreationMeta.statusBarHeight,
                    spacingMargin.right,
                    spacingMargin.bottom
                )
            }

            DisplaySize.MINIMISED -> {
                nudgeLayoutParams.setMargins(
                    spacingMargin.left,
                    spacingMargin.top + viewCreationMeta.statusBarHeight,
                    spacingMargin.right,
                    spacingMargin.bottom
                )
            }

            else -> {
                nudgeLayoutParams.setMargins(
                    spacingMargin.left,
                    spacingMargin.top,
                    spacingMargin.right,
                    spacingMargin.bottom
                )
            }
        }
        containerLayout.layoutParams = nudgeLayoutParams
        sdkInstance.logger.log { "$tag setPrimaryContainerDimensions() : " }
    }

    /**
     * Animates the dimensions of a view.
     *
     * @param view A [View] to be animated
     * @param initialViewDimension initial view dimensions [ViewDimension]
     * @param targetViewDimension target view dimensions[ViewDimension]
     * @param fraction animation progress in [Float]
     */
    private fun updateViewAnimatedDimension(
        view: View,
        initialViewDimension: ViewDimension,
        targetViewDimension: ViewDimension,
        fraction: Float
    ) {
        // Calculate current width and height of the media view according to fraction.
        val currentWidth =
            (initialViewDimension.width + (targetViewDimension.width - initialViewDimension.width) * fraction).toInt()
        val currentHeight =
            (initialViewDimension.height + (targetViewDimension.height - initialViewDimension.height) * fraction).toInt()
        sdkInstance.logger.log { "$tag updateViewAnimatedDimension(): currentWidth= $currentWidth currentHeight=$currentHeight" }

        val viewLayoutParams = view.layoutParams
        viewLayoutParams.width = currentWidth
        viewLayoutParams.height = currentHeight
        view.layoutParams = viewLayoutParams
    }

    /**
     * Registers/Unregisters a callback to be notified for InApp [DisplaySize] change event.
     * @param listener instance of [OnInAppDisplaySizeChangeListener]
     * @since 7.1.1
     */
    fun setOnInAppDisplaySizeChangeListener(listener: OnInAppDisplaySizeChangeListener?) {
        this.onInAppDisplaySizeChangeListener = listener
    }

    /**
     * Handles the background image display for resizeable nudges
     *
     * @param containerStyle instance of [ContainerStyle]
     * @param imageView container background [ImageView]
     */
    fun handleBackgroundImageForResizeableNudge(
        containerStyle: ContainerStyle,
        imageView: ImageView
    ) {
        sdkInstance.logger.log { "$tag handleBackgroundImageForResizeableNudge() : " }
        if (containerStyle.displaySize === DisplaySize.MINIMISED) {
            imageView.visibility = View.GONE
        }

        // set the listener to handle the background image visibility
        setOnInAppDisplaySizeChangeListener(object :
                OnInAppDisplaySizeChangeListener {
                override fun onDisplaySizeChangeEnd(currentDisplaySize: DisplaySize) {
                    sdkInstance.logger.log {
                        "$tag handleBackgroundImageForResizeableNudge() : " +
                            "onDisplaySizeChangeEnd(): currentDisplaySize: $currentDisplaySize"
                    }
                    if (currentDisplaySize === DisplaySize.MINIMISED) {
                        imageView.visibility = View.GONE
                    }
                }

                override fun onDisplaySizeChangeStart(currentDisplaySize: DisplaySize) {
                    sdkInstance.logger.log {
                        "$tag handleBackgroundImageForResizeableNudge() : " +
                            "onDisplaySizeChangeStart(): currentDisplaySize: $currentDisplaySize"
                    }
                    imageView.visibility =
                        if (currentDisplaySize === DisplaySize.MINIMISED) View.VISIBLE else View.GONE
                }
            })
        sdkInstance.logger.log { "$tag handleBackgroundImageForResizeableNudge() : completed" }
    }
}