/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.engine;

import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.addContentToSetTextAction;
import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.applyBackgroundToView;
import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.filterRatingChangeActionFromList;
import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.getBorder;
import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.getColor;
import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.getScaledBitmap;
import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.getViewDimensionsFromPercentage;
import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.handleDismiss;
import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.setLayoutGravity;
import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.transformMargin;
import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.transformViewDimension;
import static com.moengage.inapp.internal.engine.ViewEngineUtilsKt.updateContainerPaddingIfRequired;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Typeface;
import android.graphics.drawable.GradientDrawable;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.ImageView.ScaleType;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.RelativeLayout.LayoutParams;
import android.widget.TextView;

import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;
import androidx.core.content.res.ResourcesCompat;

import com.bumptech.glide.Glide;
import com.moengage.core.LogLevel;
import com.moengage.core.internal.global.GlobalResources;
import com.moengage.core.internal.model.SdkInstance;
import com.moengage.core.internal.model.ViewDimension;
import com.moengage.core.internal.utils.CoreUtils;
import com.moengage.core.internal.utils.MoEUtils;
import com.moengage.inapp.R;
import com.moengage.inapp.internal.ActionHandler;
import com.moengage.inapp.internal.DeliveryLoggerKt;
import com.moengage.inapp.internal.InAppConstants;
import com.moengage.inapp.internal.UtilsKt;
import com.moengage.inapp.internal.exceptions.CouldNotCreateViewException;
import com.moengage.inapp.internal.exceptions.ImageNotFoundException;
import com.moengage.inapp.internal.exceptions.VideoNotFoundException;
import com.moengage.inapp.internal.model.InAppComponent;
import com.moengage.inapp.internal.model.InAppContainer;
import com.moengage.inapp.internal.model.InAppWidget;
import com.moengage.inapp.internal.model.Margin;
import com.moengage.inapp.internal.model.NativeCampaignPayload;
import com.moengage.inapp.internal.model.Padding;
import com.moengage.inapp.internal.model.Spacing;
import com.moengage.inapp.internal.model.ViewCreationMeta;
import com.moengage.inapp.internal.model.Widget;
import com.moengage.inapp.internal.model.actions.RatingChangeAction;
import com.moengage.inapp.internal.model.customrating.CustomRatingComponent;
import com.moengage.inapp.internal.model.customrating.RatingIcon;
import com.moengage.inapp.internal.model.enums.DisplaySize;
import com.moengage.inapp.internal.model.enums.Orientation;
import com.moengage.inapp.internal.model.enums.TemplateAlignment;
import com.moengage.inapp.internal.model.enums.ViewType;
import com.moengage.inapp.internal.model.enums.WidgetType;
import com.moengage.inapp.internal.model.style.ButtonStyle;
import com.moengage.inapp.internal.model.style.CloseStyle;
import com.moengage.inapp.internal.model.style.ContainerStyle;
import com.moengage.inapp.internal.model.style.ImageStyle;
import com.moengage.inapp.internal.model.style.InAppStyle;
import com.moengage.inapp.internal.model.style.RatingStyle;
import com.moengage.inapp.internal.model.style.TextStyle;
import com.moengage.inapp.internal.repository.InAppFileManager;
import com.moengage.inapp.internal.widgets.ratingbar.MoECustomRatingBar;
import com.moengage.inapp.internal.widgets.ratingbar.MoERatingBar;
import com.moengage.inapp.model.actions.Action;

import java.io.File;
import java.util.List;

/**
 * @author Umang Chamaria
 */
public class ViewEngine extends BaseViewEngine {
	private static final String TAG = InAppConstants.MODULE_TAG + "ViewEngine";

	private static final int CLOSE_BUTTON_SIZE = 42;
	private static final int CLOSE_BUTTON_PADDING = 6;
	private static final int CLOSE_BUTTON_MARGIN = CLOSE_BUTTON_SIZE / 2;
	private static final int CLOSE_BUTTON_IMAGE_SIZE = 24;

	private final NativeCampaignPayload payload;
	private final Context context;
	private final InAppFileManager imageManager;
	private final ViewDimension parentViewDimensions;
	private View popUpView;
	private final int statusBarHeight;
	private final float densityScale;
	private int popUpId;
	private final Activity activity;
	private View inAppView;
	private final SdkInstance sdkInstance;
	private final NudgesViewEngineHelper nudgesViewEngineHelper;

	private final WidgetFactory widgetFactory;

	private ViewDimension primaryContainerExcludeDimen;

	public ViewEngine(Activity activity,
										SdkInstance sdkInstance,
										NativeCampaignPayload payload, ViewCreationMeta viewCreationMeta) {
		super(activity, payload, viewCreationMeta);
		this.activity = activity;
		this.sdkInstance = sdkInstance;
		this.context = activity.getApplicationContext();
		this.payload = payload;
		imageManager = new InAppFileManager(activity.getApplicationContext(), sdkInstance);
		this.parentViewDimensions = viewCreationMeta.deviceDimensions;
		this.statusBarHeight = viewCreationMeta.statusBarHeight;
		this.densityScale = activity.getResources().getDisplayMetrics().density;
		this.nudgesViewEngineHelper = new NudgesViewEngineHelper(
			context,
			sdkInstance,
			viewCreationMeta,
			payload,
			imageManager,
			densityScale
		);
		widgetFactory = new WidgetFactory(
			activity,
			sdkInstance,
			viewCreationMeta,
			payload,
			densityScale
		);
	}

	@SuppressWarnings("ConstantConditions")
	@SuppressLint("WrongThread")
	@WorkerThread
	@Nullable
	@Override
	public View createInApp() {
		try {
			sdkInstance.logger.log(() -> TAG + " createInApp() : Will try to create in-app view for "
				+ "campaign-id: " + payload.getCampaignId());

			sdkInstance.logger.log(() -> TAG
				+ " createInApp() : Device Dimensions: "
				+ parentViewDimensions
				+ " Status Bar height: "
				+ statusBarHeight);

			inAppView = createPrimaryContainer(payload.getPrimaryContainer());
			if (inAppView == null) return null;
			handleBackPress(inAppView);
			sdkInstance.logger.log(
				() -> TAG + " createInApp() : InApp creation complete, returning created view.");
			ContainerStyle style = (ContainerStyle) payload.getPrimaryContainer().style;
			if (style.animation != null && style.animation.entry != -1) {
				Animation animation = AnimationUtils.loadAnimation(context, style.animation.entry);
				animation.setFillAfter(true);
				inAppView.setAnimation(animation);
			}
			inAppView.setClickable(true);
			return inAppView;
		} catch (Throwable t) {
			sdkInstance.logger.log(LogLevel.ERROR, t, () -> TAG + " createInApp() : ");
			if (t instanceof UnsupportedOperationException) {
				updateStatForCampaign(payload, DeliveryLoggerKt.IMPRESSION_STAGE_GIF_LIBRARY_NOT_PRESENT,
					sdkInstance);
			} else if (t instanceof ImageNotFoundException) {
				updateStatForCampaign(payload, DeliveryLoggerKt.IMPRESSION_STAGE_IMAGE_DOWNLOAD_FAILURE,
					sdkInstance);
			} else if (t instanceof VideoNotFoundException) {
				updateStatForCampaign(payload, DeliveryLoggerKt.IMPRESSION_STAGE_VIDEO_DOWNLOAD_FAILURE,
					sdkInstance);
			}
		}
		return null;
	}

	private View createPrimaryContainer(InAppContainer container)
		throws CouldNotCreateViewException, ImageNotFoundException, IllegalStateException, VideoNotFoundException {
		sdkInstance.logger.log(() -> TAG + " createPrimaryContainer() : will create primary container");
		RelativeLayout containerLayout = new RelativeLayout(context);


		ContainerStyle containerStyle = (ContainerStyle) container.style;
		primaryContainerExcludeDimen = getStyleDimensionsToExclude(container, true);

		containerLayout.setId(InAppConstants.CONTAINER_BASE_ID + container.id);
		View widgetView = null;
		Widget widget = getWidgetFromList(container.widgets, WidgetType.CONTAINER);
		if (widget == null) {
			throw new IllegalStateException("Unexpected Widget type");
		}
		widgetView = createPopUp((InAppContainer) widget.inAppWidget, containerLayout);
		if (widgetView == null) {
			throw new CouldNotCreateViewException("One of the container/widget "
				+ "creation wasn't successful cannot create view further");
		}
		popUpView = widgetView;
		containerLayout.addView(widgetView);
		widget = getWidgetFromList(container.widgets, WidgetType.WIDGET);
		if (widget == null) {
			throw new IllegalStateException("Unexpected Widget type");
		}
		InAppWidget closeWidget = (InAppWidget) widget.inAppWidget;
		if (closeWidget.viewType != ViewType.CLOSE_BUTTON) {
			throw new IllegalStateException("Unexpected Widget type. Expected widget type is close "
				+ "button.");
		}

		ViewDimension campaignDimensions = getViewDimensionsFromPercentage(parentViewDimensions, containerStyle);
		sdkInstance.logger.log(() ->
			TAG + " createPrimaryContainer() : Campaign Dimension: " + campaignDimensions);

		ViewDimension unspecifiedDimension = getUnspecifiedViewDimension(containerLayout);
		sdkInstance.logger.log(() ->
			TAG + " createPrimaryContainer() : Computed Dimension: " + unspecifiedDimension);
		campaignDimensions.height = Math.max(campaignDimensions.height, unspecifiedDimension.height);

		if (closeWidget.component.style.display) {
			widgetView = createCloseButton(closeWidget, campaignDimensions);
			alignCloseButton(widgetView, (CloseStyle) closeWidget.component.style);
			containerLayout.addView(widgetView);
		}

		setPrimaryContainerDimensions(container, campaignDimensions, containerLayout);
		styleContainer(containerLayout, (ContainerStyle) container.style, campaignDimensions, true, primaryContainerExcludeDimen);

		containerLayout.setClipToOutline(true);
		sdkInstance.logger.log(() -> TAG + " createPrimaryContainer() : creation completed.");
		return containerLayout;
	}

	private Widget getWidgetFromList(List<Widget> widgetList, WidgetType widgetType) {
		for (Widget widget : widgetList) {
			if (widget.type == widgetType) return widget;
		}
		return null;
	}

	private void alignCloseButton(View widgetView, CloseStyle style)
		throws CouldNotCreateViewException {
		if (style.position == null) {
			throw new CouldNotCreateViewException("Cannot create in-app position "
				+ "of close button is missing Campaign-id:"
				+ payload.getCampaignId());
		}
		RelativeLayout.LayoutParams layoutParams =
			(RelativeLayout.LayoutParams) widgetView.getLayoutParams();
		switch (style.position) {
			case LEFT:
				if (payload.getTemplateType().equals(
					InAppConstants.IN_APP_TEMPLATE_TYPE_POP_UP)) {
					layoutParams.addRule(RelativeLayout.ALIGN_TOP, popUpView.getId());
					layoutParams.addRule(RelativeLayout.ALIGN_LEFT, popUpView.getId());
					layoutParams.leftMargin += transformViewDimension(style.margin.left,
						parentViewDimensions.width) - (CLOSE_BUTTON_MARGIN * densityScale);
				} else {
					layoutParams.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
				}
				break;
			case RIGHT:
				if (payload.getTemplateType().equals(
					InAppConstants.IN_APP_TEMPLATE_TYPE_POP_UP)) {
					layoutParams.rightMargin += transformViewDimension(style.margin.right,
						parentViewDimensions.width) - (CLOSE_BUTTON_MARGIN * densityScale);
					layoutParams.addRule(RelativeLayout.ALIGN_TOP, popUpView.getId());
					layoutParams.addRule(RelativeLayout.ALIGN_RIGHT, popUpView.getId());
				} else {
					layoutParams.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
				}
				break;
		}
		if (payload.getTemplateType().equals(InAppConstants.IN_APP_TEMPLATE_TYPE_POP_UP)) {
			layoutParams.topMargin = layoutParams.topMargin - (int) (CLOSE_BUTTON_MARGIN * densityScale);
		}
		widgetView.setLayoutParams(layoutParams);
	}

	private View createCloseButton(InAppWidget widget, ViewDimension primaryContainerDimension) {
		sdkInstance.logger
			.log(() -> TAG + " createCloseButton() : Will create close button. " + widget);
		Bitmap imageBitmap = imageManager.getImageFromUrl(context, widget.component.content,
			payload.getCampaignId());
		if (imageBitmap == null) {
			int res = R.drawable.moengage_inapp_close;
			imageBitmap = BitmapFactory.decodeResource(context.getResources(), res);
		}
		ImageView imageView = new ImageView(context);
		int dimension = (int) (CLOSE_BUTTON_SIZE * densityScale);
		ViewDimension imageDimension = new ViewDimension(dimension,
			Math.min(dimension, primaryContainerDimension.height));
		int imageSize = (int) (CLOSE_BUTTON_IMAGE_SIZE * densityScale);
		imageView.setImageBitmap(getScaledBitmap(imageBitmap, new ViewDimension(imageSize, imageSize)));
		RelativeLayout.LayoutParams layoutParams =
			new RelativeLayout.LayoutParams(imageDimension.width,
				imageDimension.height);

		int fixedPadding = (int) (CLOSE_BUTTON_PADDING * densityScale);
		Spacing padding = new Spacing(fixedPadding, fixedPadding, fixedPadding, fixedPadding);
		imageView.setPadding(padding.left, padding.top, padding.right, padding.bottom);
		imageView.setLayoutParams(layoutParams);
		imageView.setClickable(true);
		addAction(imageView, widget.actions);
		return imageView;
	}

	@SuppressLint("ResourceType")
	private View createPopUp(InAppContainer container,
													 RelativeLayout primaryContainer)
		throws CouldNotCreateViewException, ImageNotFoundException, VideoNotFoundException {
		RelativeLayout popUpLayout = new RelativeLayout(context);
		popUpId = container.id;
		View popUpView = createContainer(container, primaryContainer);
		if (popUpView == null) {
			throw new CouldNotCreateViewException(
				"One of the container/widget creation wasn't successful cannot create view further");
		}
		RelativeLayout.LayoutParams layoutParams =
			new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT,
				ViewGroup.LayoutParams.WRAP_CONTENT);
		addMarginToLayout(layoutParams, container.style);
		popUpLayout.setLayoutParams(layoutParams);
		//popUpView.he
		ViewDimension popUpDimensions = new ViewDimension(getViewDimensionsFromPercentage(parentViewDimensions, container.style).width, getUnspecifiedViewDimension(popUpView).height);
		sdkInstance.logger
			.log(() -> TAG + " createPopUp() : Pop up view Dimensions: " + popUpDimensions);
		styleContainer(popUpLayout, (ContainerStyle) container.style, popUpDimensions, false, primaryContainerExcludeDimen);
		popUpLayout.addView(popUpView);
		alignContainer(popUpLayout, payload.getAlignment());
		popUpLayout.setId(12345);
		return popUpLayout;
	}

	private void alignContainer(View view, TemplateAlignment alignment) {
		RelativeLayout.LayoutParams layoutParams = (RelativeLayout.LayoutParams) view.getLayoutParams();
		layoutParams.addRule(RelativeLayout.CENTER_VERTICAL);
		layoutParams.addRule(RelativeLayout.CENTER_HORIZONTAL);
		view.setLayoutParams(layoutParams);
	}

	private View createContainer(InAppContainer container, RelativeLayout primaryContainer)
		throws CouldNotCreateViewException, ImageNotFoundException, VideoNotFoundException {
		LinearLayout containerLayout = new LinearLayout(context);
		switch (container.orientation) {
			case VERTICAL:
				containerLayout.setOrientation(LinearLayout.VERTICAL);
				break;
			case HORIZONTAL:
				containerLayout.setOrientation(LinearLayout.HORIZONTAL);
				containerLayout.setGravity(Gravity.CENTER_HORIZONTAL);
				break;
		}
		View widgetView = null;
		for (Widget widget : container.widgets) {
			switch (widget.type) {
				case WIDGET:
					InAppWidget inAppWidget = (InAppWidget) widget.inAppWidget;
					if (!inAppWidget.component.style.display) {
						sdkInstance.logger.log(() -> TAG + " createContainer() : Display type of widget is " +
							"false. Will not create widget. " + inAppWidget);
						continue;
					}
					widgetView = createWidget(inAppWidget, container.orientation, primaryContainer, getStyleDimensionsToExclude(container, false));
					break;
				case CONTAINER:
					InAppContainer inAppContainer = (InAppContainer) widget.inAppWidget;
					if (!inAppContainer.style.display) {
						sdkInstance.logger.log(() -> TAG + " createContainer() : Display type of container is" +
							" false. Will not create container. " + inAppContainer);
						continue;
					}
					widgetView = createContainer(inAppContainer, primaryContainer);
					break;
			}
			if (widgetView == null) {
				throw new CouldNotCreateViewException("One of the container/widget "
					+ "creation wasn't successful cannot create view further");
			}
			containerLayout.addView(widgetView);
		}
		sdkInstance.logger.log(() -> TAG + " createContainer() : " + container.style);
		setViewDimensionsPopUp(containerLayout, container.style);
		if (popUpId != container.id) {
			RelativeLayout.LayoutParams layoutParams =
				(RelativeLayout.LayoutParams) containerLayout.getLayoutParams();
			addMarginToLayout(layoutParams, container.style);
			containerLayout.setLayoutParams(layoutParams);
			Spacing spacing = transformPadding(container.style.padding);
			containerLayout.setPadding(spacing.left, spacing.top, spacing.right, spacing.bottom);
			styleContainer(containerLayout, (ContainerStyle) container.style);
		}
		containerLayout.setId(InAppConstants.CONTAINER_BASE_ID + container.id);
		return containerLayout;
	}

	private void setViewDimensionsPopUp(View view, InAppStyle style) {

		ViewDimension campaignDimension = getViewDimensionsFromPercentage(parentViewDimensions, style);
		sdkInstance.logger.log(() ->
			TAG + " setViewDimensionsPopUp() : Campaign Dimension " + campaignDimension);
		ViewDimension computedDimension = getUnspecifiedViewDimension(view);
		sdkInstance.logger.log(() ->
			TAG + " setViewDimensionsPopUp() : Computed dimension: " + computedDimension);

		campaignDimension.height = Math.max(campaignDimension.height, computedDimension.height);


		if (getPrimaryContainerStyle().displaySize == DisplaySize.FULLSCREEN) {
			campaignDimension.height = RelativeLayout.LayoutParams.MATCH_PARENT;
		}

		RelativeLayout.LayoutParams layoutParams =
			new RelativeLayout.LayoutParams(campaignDimension.width,
				campaignDimension.height);
		view.setLayoutParams(layoutParams);
	}

	private View createWidget(InAppWidget widget, Orientation parentOrientation,
														RelativeLayout primaryContainer, ViewDimension toExclude)
		throws ImageNotFoundException, CouldNotCreateViewException, VideoNotFoundException {
		sdkInstance.logger.log(() -> TAG + " createWidget() : Creating widget: " + widget);
		View view = null;
		switch (widget.viewType) {
			case TEXT:
			case FEEDBACK_TEXT:
				view = createTextView(widget, parentOrientation, toExclude);
				break;
			case IMAGE:
				view = createImageView(widget, parentOrientation, primaryContainer, toExclude);
				break;
			case BUTTON:
				view = createButton(widget, parentOrientation, toExclude);
				break;
			case RATING:
				view = createRatingBar(widget, parentOrientation, toExclude);
				break;
			case VIDEO:
				view = nudgesViewEngineHelper.createVideoView(widget, parentOrientation, primaryContainer, toExclude);
				break;
			case CUSTOM_RATING:
				view = createCustomRatingBar(widget, parentOrientation, toExclude);
				break;
		}
		if (view == null) {
			throw new CouldNotCreateViewException(
				"View type not recognised. Type " + widget.viewType);
		}
		view.setId(InAppConstants.WIDGET_BASE_ID + widget.id);
		view.setClickable(true);
		addAction(view, widget.actions);
		return view;
	}

	private TextView createTextView(InAppWidget widget, Orientation parentOrientation, ViewDimension toExclude) {
		sdkInstance.logger.log(() -> TAG + " createTextView() : Will create text widget: " + widget);
		TextView textView = new TextView(context);
		setTextContent(textView, widget.component);
		TextStyle textStyle = (TextStyle) widget.component.style;

		textView.setTextSize(textStyle.font.size);

		if (textStyle.font.color != null) {
			textView.setTextColor(getColor(textStyle.font.color));
		}
		int resourceId = context.getResources().getIdentifier(textStyle.font.name, "font", context
			.getPackageName());
		if (resourceId > 0) {
			Typeface typeface = ResourcesCompat.getFont(context, resourceId);
			textView.setTypeface(typeface);
		}

		ViewDimension campaignDimension = getViewDimensionsFromPercentage(parentViewDimensions, widget.component.style);

		if (payload.getTemplateType().equals(InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE)) {
			//NOTE: This is only for nudges in order to fix the background image with border and
			// rounded corners issue
			campaignDimension.width -= toExclude.width;
		}
		sdkInstance.logger
			.log(() -> TAG + " createTextView() : Campaign Dimension: " + campaignDimension);
		campaignDimension.height = LayoutParams.WRAP_CONTENT;
		Spacing paddingSpacing = transformPadding(textStyle.padding);
		sdkInstance.logger.log(() -> TAG + " createTextView() : Padding: " + paddingSpacing);
		textView.setPadding(paddingSpacing.left, paddingSpacing.top, paddingSpacing.right,
			paddingSpacing.bottom);
		sdkInstance.logger
			.log(() -> TAG + " createTextView() : Final Dimensions: " + campaignDimension);
		LinearLayout.LayoutParams layoutParams =
			new LinearLayout.LayoutParams(campaignDimension.width,
				campaignDimension.height);
		setLayoutGravity(layoutParams, parentOrientation);
		Spacing marginSpacing = transformMargin(sdkInstance, parentViewDimensions, textStyle.margin);
		layoutParams.setMargins(marginSpacing.left, marginSpacing.top, marginSpacing.right,
			marginSpacing.bottom);
		textView.setLayoutParams(layoutParams);
		GradientDrawable drawable = new GradientDrawable();
		if (textStyle.background != null && textStyle.background.color != null) {
			drawable.setColor(getColor(textStyle.background.color));
		}
		if (textStyle.border != null) {
			getBorder(textStyle.border, drawable, densityScale);
		}
		applyBackgroundToView(textView, drawable, payload.getTemplateType());
		if (!payload.getTemplateType().equals(InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE) ||
			widget.viewType == ViewType.FEEDBACK_TEXT) {
			textView.setGravity(Gravity.CENTER);
		}
		textView.setVisibility(textStyle.visibility.toViewVisibility());
		if (textStyle.maxLines != -1) {
			textView.setMaxLines(textStyle.maxLines);
			textView.setEllipsize(TextUtils.TruncateAt.END);
		}
		return textView;
	}

	@SuppressWarnings("ConstantConditions")
	@SuppressLint("CheckResult")
	private View createImageView(InAppWidget widget,
															 Orientation parentOrientation, RelativeLayout primaryContainer, ViewDimension toExclude)
		throws ImageNotFoundException {
		sdkInstance.logger.log(() -> TAG + " createImageView() : Will create this widget: " + widget);
		boolean isGif = CoreUtils.isGif(widget.component.content);
		if (!MoEUtils.hasGlideSupport()) {
			sdkInstance.logger.log(LogLevel.WARN, () -> TAG + " createImageView() : Image is of gif type, gif dependency not present");
			throw new UnsupportedOperationException(
				"Library support not found: Image and gif require Glide library.");
		}
		final ImageView imageView = new ImageView(context);
		ImageStyle imageStyle = (ImageStyle) widget.component.style;
		ContainerStyle containerStyle = getPrimaryContainerStyle();

		// Flag to check if the image widget is for non-intrusive nudges gif template
		boolean isResizeableGif = isGif && containerStyle.displaySize != null;

		ViewDimension campaignDimension = isResizeableGif
			? getViewDimensionsFromPercentage(parentViewDimensions, containerStyle)
			: getViewDimensionsFromPercentage(parentViewDimensions, imageStyle);

		sdkInstance.logger.log(() -> TAG + " createImageView() : Campaign Dimension: " + campaignDimension);

		// Update campaign dimensions if the initial displaySize for non-intrusive nudge gif
		// template is DisplaySize.FULL_SCREEN
		if (containerStyle.displaySize == DisplaySize.FULLSCREEN) {
			ViewDimension fullscreenDimension = nudgesViewEngineHelper.getFullScreenViewDimension(containerStyle);
			sdkInstance.logger.log(() -> TAG + " createImageView(): fullscreen Dimensions: " + fullscreenDimension);
			campaignDimension.width = fullscreenDimension.width;
			campaignDimension.height = fullscreenDimension.height;
		}

		if (isGif) {
			loadGif(imageView, isResizeableGif, widget, imageStyle, campaignDimension);
		} else {
			loadBitmap(imageView, widget, campaignDimension);
		}

		ViewGroup imageviewContainer;
		LinearLayout.LayoutParams linearLayoutParams;
		if (isResizeableGif) {
			// Media container containing image view and media controller for non-intrusive nudge gif
			// template.
			imageviewContainer = nudgesViewEngineHelper.createContainerForResizeableImageView(primaryContainer, imageView, imageStyle, containerStyle.displaySize);
			linearLayoutParams = new LinearLayout.LayoutParams(campaignDimension.width, campaignDimension.height);
			linearLayoutParams.weight = 0.9f;
		} else {
			//TODO move this to FrameLayout as well, requires retesting of all in-apps
			imageviewContainer = new LinearLayout(context);
			linearLayoutParams = new LinearLayout.LayoutParams(
				imageView.getLayoutParams().width, imageView.getLayoutParams().height);
			imageviewContainer.addView(imageView);
			sdkInstance.logger.log(() -> TAG + " createImageView() : widget: " + widget +
				" creation completed.");
		}

		Spacing marginSpacing = transformMargin(sdkInstance, parentViewDimensions, imageStyle.margin);
		linearLayoutParams.setMargins(marginSpacing.left, marginSpacing.top, marginSpacing.right,
			marginSpacing.bottom);
		linearLayoutParams.leftMargin = marginSpacing.left;
		linearLayoutParams.rightMargin = marginSpacing.right;
		linearLayoutParams.topMargin = marginSpacing.top;
		linearLayoutParams.bottomMargin = marginSpacing.bottom;
		setLayoutGravity(linearLayoutParams, parentOrientation);
		imageviewContainer.setLayoutParams(linearLayoutParams);
		int borderWidth = 0;
		if (imageStyle.border != null) {
			borderWidth = transformToPx(imageStyle.border.width);
		}
		if (imageStyle.border != null) {
			applyBackgroundToView(imageviewContainer, getBorder(imageStyle.border,
					densityScale),
				payload.getTemplateType()
			);
		}
		imageviewContainer.setPadding(borderWidth, borderWidth, borderWidth, borderWidth);
		sdkInstance.logger.log(() -> TAG + " createImageView() : widget: " + widget + " creation " +
			"completed");
		return imageviewContainer;
	}

	private Button createButton(InAppWidget widget, Orientation parentOrientation, ViewDimension toExclude) {
		sdkInstance.logger.log(() -> TAG + " createButton() : Will create button widget " + widget);
		Button button = new Button(context);
		setTextContent(button, widget.component);
		ButtonStyle style = (ButtonStyle) widget.component.style;
		sdkInstance.logger.log(() -> TAG + " createButton() : Style: " + style);
		button.setTextSize(style.font.size);

		if (style.font.color != null) {
			button.setTextColor(getColor(style.font.color));
		}
		int resourceId = context.getResources().getIdentifier(style.font.name, "font", context
			.getPackageName());
		if (resourceId > 0) {
			Typeface typeface = ResourcesCompat.getFont(context, resourceId);
			button.setTypeface(typeface);
		}
		ViewDimension campaignDimension = getViewDimensionsFromPercentage(parentViewDimensions, widget.component.style);
		sdkInstance.logger
			.log(() -> TAG + " createButton() : Campaign Dimension: " + campaignDimension);
		Spacing spacing = transformPadding(style.padding);
		sdkInstance.logger.log(() -> TAG + " createButton() : Padding: " + spacing);
		button.setPadding(spacing.left, spacing.top, spacing.right, spacing.bottom);
		ViewDimension unSpecifiedDimension = getUnspecifiedViewDimension(button);
		sdkInstance.logger
			.log(() -> TAG + " createButton() : Calculated Dimensions: " + unSpecifiedDimension);
		int minimumHeight = transformToPx(style.minHeight);
		sdkInstance.logger
			.log(() -> TAG + " createButton() : Minimum height for widget: " + minimumHeight);
		if (minimumHeight > unSpecifiedDimension.height) {
			campaignDimension.height = minimumHeight;
		}

		if (payload.getTemplateType().equals(InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE)) {
			//NOTE: This is only for nudges in order to fix the background image with border and
			// rounded corners issue
			campaignDimension.width -= toExclude.width;
		}
		sdkInstance.logger.log(() -> TAG + " createButton() : Final Dimensions: " + campaignDimension);

		LinearLayout.LayoutParams layoutParams = getPrimaryContainerStyle().displaySize != null ?
			new LinearLayout.LayoutParams(LinearLayout.LayoutParams.MATCH_PARENT, campaignDimension.height) :
			new LinearLayout.LayoutParams(campaignDimension.width, campaignDimension.height);

		setLayoutGravity(layoutParams, parentOrientation);
		Spacing marginSpacing = transformMargin(sdkInstance, parentViewDimensions, style.margin);
		layoutParams.setMargins(marginSpacing.left, marginSpacing.top, marginSpacing.right,
			marginSpacing.bottom);
		button.setLayoutParams(layoutParams);

		GradientDrawable drawable = new GradientDrawable();

		if (style.background != null && style.background.color != null) {
			drawable.setColor(getColor(style.background.color));
		}
		if (style.border != null) {
			getBorder(style.border, drawable, densityScale);
		}
		applyBackgroundToView(button, drawable, payload.getTemplateType());
		button.setGravity(Gravity.CENTER);

		return button;
	}

	private MoERatingBar createRatingBar(InAppWidget widget, Orientation parentOrientation, ViewDimension toExclude) {
		sdkInstance.logger
			.log(() -> TAG + " createRatingBar() : Will create rating widget: " + widget);
		MoERatingBar ratingBar = new MoERatingBar(context);
		ratingBar.setIsIndicator(false);

		RatingStyle style = (RatingStyle) widget.component.style;
		ratingBar.setNumStars(style.numberOfStars);
		if (style.isHalfStepAllowed) {
			ratingBar.setStepSize(0.5f);
		} else {
			ratingBar.setStepSize(1.0f);
		}
		ratingBar.setColor(getColor(style.color));
		ViewDimension campaignDimension = new ViewDimension(
			getViewDimensionsFromPercentage(parentViewDimensions, style).width,
			(int) (style.realHeight * densityScale)
		);

		if (payload.getTemplateType().equals(InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE)) {
			//NOTE: This is only for nudges in order to fix the background image with border and
			//rounded corners issue
			campaignDimension.width -= toExclude.width;
		}

		sdkInstance.logger
			.log(() -> TAG + " createRatingBar() : Campaign dimensions: " + campaignDimension);
		LinearLayout.LayoutParams layoutParams =
			new LinearLayout.LayoutParams(campaignDimension.width, campaignDimension.height);
		setLayoutGravity(layoutParams, parentOrientation);
		Spacing marginSpacing = transformMargin(sdkInstance, parentViewDimensions, style.margin);
		layoutParams.setMargins(marginSpacing.left, marginSpacing.top, marginSpacing.right,
			marginSpacing.bottom);
		ratingBar.setLayoutParams(layoutParams);
		GradientDrawable drawable = new GradientDrawable();

		if (style.border != null) {
			getBorder(style.border, drawable, densityScale);
		}
		applyBackgroundToView(ratingBar, drawable, payload.getTemplateType());
		return ratingBar;
	}

	private void addAction(final View view, final List<Action> actions) {
		if (actions == null) {
			sdkInstance.logger.log(() -> TAG + " addAction() : View does not have any actionType.");
			return;
		}
		sdkInstance.logger.log(() -> TAG + " addAction() : Will try to execute actionType: " + actions);
		view.setOnClickListener(v -> {
			ActionHandler handler = new ActionHandler(activity, sdkInstance);
			for (Action action : actions) {
				sdkInstance.logger.log(() -> TAG + " onClick() : Will execute actionType: " + action);
				handler.onActionPerformed(inAppView, action, payload);
			}
		});
	}

	private void setTextContent(TextView view, InAppComponent component) {
		view.setText(component.content);
		view.setAllCaps(false);
	}

	private void addMarginToLayout(LayoutParams layoutParams, InAppStyle style) {
		Margin margin = style.margin;
		layoutParams.leftMargin = margin.left == 0 ? 0
			: transformViewDimension(margin.left, parentViewDimensions.width);
		layoutParams.rightMargin = margin.right == 0 ? 0
			: transformViewDimension(margin.right, parentViewDimensions.width);
		layoutParams.topMargin = margin.top == 0 ? 0
			: transformViewDimension(margin.top, parentViewDimensions.height);
		layoutParams.bottomMargin = margin.bottom == 0 ? 0
			: transformViewDimension(margin.bottom, parentViewDimensions.height);
	}

	private ViewDimension getUnspecifiedViewDimension(View view) {
		view.measure(View.MeasureSpec.UNSPECIFIED, View.MeasureSpec.UNSPECIFIED);
		return new ViewDimension(view.getMeasuredWidth(), view.getMeasuredHeight());
	}

	private void styleContainer(LinearLayout containerLayout, ContainerStyle style) {

		if (style.background != null && style.background.color != null) {
			containerLayout.setBackgroundColor(getColor(style.background.color));
		}
		//background without image
		if (style.border != null) {
			GradientDrawable backgroundDrawable = getBorder(style.border, densityScale);
			if (style.background != null && style.background.color != null) {
				backgroundDrawable.setColor(getColor(style.background.color));
			}
			applyBackgroundToView(containerLayout, backgroundDrawable, payload.getTemplateType());
		}
	}

	private void styleContainer(RelativeLayout containerLayout, ContainerStyle style,
															ViewDimension containerDimensions, Boolean isPrimaryContainer, ViewDimension toExclude) throws ImageNotFoundException {
		sdkInstance.logger.log(() -> TAG + " styleContainer() : will style container");
		if (style.background == null) return;
		final int borderWidth;
		int borderRadius;
		if (style.border != null) {
			borderWidth = (int) (style.border.width * densityScale);
			borderRadius = (int) (style.border.radius);
		} else {
			borderRadius = 0;
			borderWidth = 0;
		}
		sdkInstance.logger.log(() -> TAG + " styleContainer() : borderWidth: " + borderWidth + ", " +
			"borderRadius: " + borderRadius);
		updateContainerPaddingIfRequired(borderWidth, containerLayout);

		if (style.background.content != null) {
			sdkInstance.logger.log(() -> TAG + " styleContainer() : background has content.");

			if (!MoEUtils.hasGlideSupport()) {
				sdkInstance.logger.log(LogLevel.WARN, () -> TAG + " styleContainer() : Image is of gif type, gif dependency not present");
				throw new UnsupportedOperationException(
					"Library support not found: Image and gif require Glide library."
				);
			}
			//background with image
			final ImageView imageView = new ImageView(context);
			RelativeLayout.LayoutParams layoutParams;
			// If the template is a resizeable, the background image will not be shown in the minimised
			// state
			if (getPrimaryContainerStyle().displaySize != null) {
				layoutParams = new RelativeLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
				nudgesViewEngineHelper.handleBackgroundImageForResizeableNudge(style, imageView);
			} else {
				final int height;
				if (!isPrimaryContainer) {
					height = containerDimensions.height - toExclude.height;
				} else {
					height = containerDimensions.height;
				}
				layoutParams = new RelativeLayout.LayoutParams(containerDimensions.width - toExclude.width, height);
			}
			imageView.setLayoutParams(layoutParams);
			imageView.setScaleType(ScaleType.FIT_XY);

			if (CoreUtils.isGif(style.background.content)) {
				final File gifFile = imageManager.getGifFromUrl(style.background.content,
					payload.getCampaignId());
				if (gifFile == null || !gifFile.exists()) {
					throw new ImageNotFoundException("Gif Download failure");
				}
				UtilsKt.loadContainerImageBackground(context, sdkInstance, borderRadius, gifFile, imageView, true);
			} else {
				final Bitmap imageBitmap = imageManager.getImageFromUrl(context, style.background.content,
					payload.getCampaignId());
				if (imageBitmap == null) throw new ImageNotFoundException("Image Download failure");
				UtilsKt.loadContainerImageBackground(context, sdkInstance, borderRadius, imageBitmap, imageView, false);
			}
			containerLayout.addView(imageView, 0);
		}
		GradientDrawable drawable = new GradientDrawable();

		if (style.background.color != null) {
			drawable.setColor(getColor(style.background.color));
		}
		if (style.border != null) {
			getBorder(style.border, drawable, densityScale);
		}

		applyBackgroundToView(containerLayout, drawable, payload.getTemplateType());
	}


	private void handleBackPress(View inAppView) {
		sdkInstance.logger.log(() -> TAG + " handleBackPress() : will set back press handling.");
		if (payload.getTemplateType().equals(InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE)) return;
		inAppView.setFocusable(true);
		inAppView.setFocusableInTouchMode(true);
		inAppView.requestFocus();
		inAppView.setOnKeyListener((v, keyCode, event) -> {
			try {
				if (event.getAction() == KeyEvent.ACTION_DOWN && keyCode == KeyEvent.KEYCODE_BACK) {
					sdkInstance.logger.log(() -> TAG + " handleBackPress() : on back button pressed");
					ContainerStyle containerStyle =
						(ContainerStyle) payload.getPrimaryContainer().style;
					com.moengage.inapp.internal.model.Animation animation = containerStyle.animation;
					if (animation != null && animation.exit != -1) {
						Animation viewAnimation = AnimationUtils.loadAnimation(context, animation.exit);
						viewAnimation.setFillAfter(true);
						v.setAnimation(viewAnimation);
					}
					((ViewGroup) v.getParent()).removeView(v);
					handleDismiss(sdkInstance, payload);
					return true;
				}
			} catch (Throwable t) {
				sdkInstance.logger.log(LogLevel.ERROR, t, () -> TAG + " onKey() : ");
			}
			return false;
		});
	}

	private Spacing transformPadding(Padding padding) {
		Spacing spacing = new Spacing(
			padding.left == 0 ? 0
				: transformViewDimension(padding.left, parentViewDimensions.width),
			padding.right == 0 ? 0
				: transformViewDimension(padding.right, parentViewDimensions.width),
			padding.top == 0 ? 0
				: transformViewDimension(padding.top, parentViewDimensions.height),
			padding.bottom == 0 ? 0
				: transformViewDimension(padding.bottom, parentViewDimensions.height)
		);
		sdkInstance.logger.log(() -> TAG + " transformPadding() : Padding: " + spacing);
		return spacing;
	}

	private int transformToPx(double dp) {
		return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, (float) dp,
			activity.getResources().getDisplayMetrics());
	}

	private void setPrimaryContainerDimensions(InAppContainer container,
																						 ViewDimension campaignDimensions,
																						 RelativeLayout containerLayout) throws CouldNotCreateViewException {
		sdkInstance.logger.log(() ->
			TAG + " setPrimaryContainerDimensions() : will set dimensions");
		ContainerStyle containerStyle = (ContainerStyle) container.style;
		Spacing spacingMargin = transformMargin(sdkInstance, parentViewDimensions, container.style.margin);
		if (payload.getTemplateType().equals(InAppConstants.IN_APP_TEMPLATE_TYPE_POP_UP) ||
			payload.getTemplateType().equals(InAppConstants.IN_APP_TEMPLATE_TYPE_FULL_SCREEN)) {
			spacingMargin = new Spacing(
				spacingMargin.left,
				spacingMargin.right,
				spacingMargin.top + statusBarHeight,
				spacingMargin.bottom
			);
		}

		if (payload.getTemplateType().equals(InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE)) {
			nudgesViewEngineHelper.setPrimaryContainerDimensions(containerLayout, containerStyle, campaignDimensions);
		} else {
			RelativeLayout.LayoutParams layoutParams =
				new RelativeLayout.LayoutParams(campaignDimensions.width, LayoutParams.MATCH_PARENT);
			layoutParams.setMargins(spacingMargin.left, spacingMargin.top, spacingMargin.right, spacingMargin.bottom);
			containerLayout.setLayoutParams(layoutParams);
		}

		Spacing spacingPadding = transformPadding(container.style.padding);
		containerLayout.setPadding(spacingPadding.left, spacingPadding.top, spacingPadding.right, spacingPadding.bottom);
		sdkInstance.logger.log(() -> TAG + " setPrimaryContainerDimensions() : completed");
	}

	private void loadBitmap(ImageView imageView, InAppWidget widget, ViewDimension campaignDimension) throws ImageNotFoundException {
		sdkInstance.logger.log(() -> TAG + " loadBitmap() : will load bitmap in ImageView.");
		Bitmap imageBitmap = imageManager.getImageFromUrl(context, widget.component.content,
			payload.getCampaignId());
		if (imageBitmap == null) throw new ImageNotFoundException("Image Download failure");

		ViewDimension imageDimension = new ViewDimension(imageBitmap.getWidth(),
			imageBitmap.getHeight());
		sdkInstance.logger.log(() -> TAG + " loadBitmap() : Image dimensions: " + imageDimension);
		campaignDimension.height =
			(imageDimension.height * campaignDimension.width) / imageDimension.width;
		sdkInstance.logger.log(() -> TAG + " loadBitmap() : Final dimensions: " + campaignDimension);
		LinearLayout.LayoutParams layoutParams =
			new LinearLayout.LayoutParams(campaignDimension.width, campaignDimension.height);
		imageView.setLayoutParams(layoutParams);
		imageView.setImageBitmap(getScaledBitmap(imageBitmap, campaignDimension));
		sdkInstance.logger.log(() -> TAG + " loadBitmap() : completed");
	}

	/**
	 * Downloads and loads the gif in the give {@link ImageView}
	 *
	 * @param imageView         instance of {@link ImageView}
	 * @param isResizeableGif   true if the campaign is resizeable(Fullscreen/Minimise), else false
	 * @param widget            instance of {@link InAppWidget}
	 * @param imageStyle        instance of {@link ImageStyle}
	 * @param campaignDimension instance of {@link ViewDimension}
	 * @throws ImageNotFoundException
	 */
	private void loadGif(ImageView imageView,
											 boolean isResizeableGif,
											 InAppWidget widget,
											 ImageStyle imageStyle,
											 ViewDimension campaignDimension) throws ImageNotFoundException {
		sdkInstance.logger.log(() -> TAG + " loadGif() : will load gif in ImageView.");
		final File gifFile = imageManager.getGifFromUrl(widget.component.content, payload.getCampaignId());
		if (gifFile == null || !gifFile.exists()) {
			throw new ImageNotFoundException("Gif Download failure");
		}
		sdkInstance.logger.log(() -> TAG + " loadGif() : Real dimensions: " +
			new ViewDimension((int) imageStyle.realWidth, (int) imageStyle.realHeight));
		campaignDimension.height =
			(int) ((imageStyle.realHeight * campaignDimension.width) / imageStyle.realWidth);
		sdkInstance.logger.log(() -> TAG + " loadGif() : Final Dimensions: " + campaignDimension);

		if (isResizeableGif) {
			FrameLayout.LayoutParams layoutParams = new FrameLayout.LayoutParams(
				campaignDimension.width,
				campaignDimension.height
			);
			layoutParams.gravity = Gravity.CENTER;
			imageView.setLayoutParams(layoutParams);
			imageView.setScaleType(ScaleType.FIT_XY);
		} else {
			LinearLayout.LayoutParams layoutParams = new LinearLayout.LayoutParams(
				campaignDimension.width,
				campaignDimension.height
			);
			imageView.setLayoutParams(layoutParams);
			imageView.setScaleType(ScaleType.FIT_CENTER);
		}

		GlobalResources.INSTANCE.getMainThread().post(() -> {
			try {
				Glide.with(context).asGif().load(gifFile).into(imageView);
			} catch (Throwable t) {
				sdkInstance.logger.log(LogLevel.ERROR, t, () -> TAG + " styleContainer() : ");
			}
		});
		sdkInstance.logger.log(() -> TAG + " loadGif() : completed");
	}

	/**
	 * Provides the container style for primary container
	 *
	 * @return {@link ContainerStyle}
	 * @throws IllegalStateException in case the primary container is null
	 */
	private ContainerStyle getPrimaryContainerStyle() throws IllegalStateException {
		if (payload.getPrimaryContainer() == null)
			throw new IllegalStateException("no primary container found");

		return (ContainerStyle) payload.getPrimaryContainer().style;
	}

	/**
	 * Creates {@link MoECustomRatingBar} widget
	 *
	 * @param widget            instance of {@link InAppWidget}
	 * @param parentOrientation {@link Orientation}
	 * @param toExclude
	 * @return {@link MoECustomRatingBar} object
	 * @since 7.0.0
	 */
	private MoECustomRatingBar createCustomRatingBar(InAppWidget widget, Orientation parentOrientation, ViewDimension toExclude) {
		sdkInstance.logger.log(() -> TAG + " createCustomRatingBar() : Will create rating widget: " + widget);

		MoECustomRatingBar ratingBar = widgetFactory.createCustomRatingBar(widget, parentOrientation, toExclude);
		CustomRatingComponent customRatingComponent = (CustomRatingComponent) widget.component;

		ratingBar.setOnRatingBarChangeListener((ratingBar1, rating, fromUser) -> {
			try {
				sdkInstance.logger
					.log(() -> TAG + " createCustomRatingBar() : onRatingChanged() : rating: " + rating
					);
				/*
				filter the RatingChangeAction from the list of actions
				*/
				RatingChangeAction ratingChangeAction = filterRatingChangeActionFromList(widget.actions);
				if (ratingChangeAction == null) {
					sdkInstance.logger.log(() -> TAG + " createCustomRatingBar() onRatingChanged() : " +
						"ratingChangeAction is null.");
					return;
				}
				RatingIcon ratingIcon = customRatingComponent.getRatingIcons().get((int) rating);
				if (ratingIcon == null) {
					sdkInstance.logger.log(LogLevel.ERROR, () -> TAG + " createCustomRatingBar() : " +
						"Couldn't find rating icon for rating " + rating);
					return;
				}
				//Filter SetTextAction and update the SetTextAction.content value based on the selected
				// rating
				addContentToSetTextAction(
					ratingChangeAction.getActions(),
					ratingIcon.getDescription()
				);
				new ActionHandler(activity, sdkInstance).onActionPerformed(inAppView, ratingChangeAction, payload);
			} catch (Throwable t) {
				sdkInstance.logger.log(LogLevel.ERROR, t, () -> TAG + " createCustomRatingBar() : ");
			}
		});
		sdkInstance.logger.log(() -> TAG + " createCustomRatingBar() : MoECustomRatingBar created " +
			"successfully.");
		return ratingBar;
	}

	/***
	 * Calculates the dimensions of the container layout that has to be excluded from the child
	 * view dimensions.
	 * @param container {@link InAppContainer}
	 * @param isPrimaryContainer true if it is a primary container, else false
	 * @return instance of {@link ViewDimension}
	 *
	 * @since 7.0.0
	 */
	private ViewDimension getStyleDimensionsToExclude(InAppContainer container, Boolean isPrimaryContainer) {
		sdkInstance.logger.log(() ->
			TAG + " getStyleDimensionsToExclude() : will set dimensions");

		ViewDimension toExclude;
		ContainerStyle containerStyle = (ContainerStyle) container.style;

		int borderWidth;
		if (containerStyle.background != null && containerStyle.border != null) {
			borderWidth = (int) (containerStyle.border.width * densityScale);
		} else {
			borderWidth = 0;
		}
		Spacing paddingSpacing = transformPadding(container.style.padding);
		Spacing spacingMargin = transformMargin(sdkInstance, parentViewDimensions, container.style.margin);

		toExclude = new ViewDimension(
			borderWidth * 2,
			borderWidth * 2
		);
		sdkInstance.logger.log(() -> TAG + " getStyleDimensionsToExclude() : toExclude: " + toExclude);
		if (isPrimaryContainer) {
			primaryContainerExcludeDimen = toExclude;
		} else {
			toExclude.width += primaryContainerExcludeDimen.width;
			toExclude.height += primaryContainerExcludeDimen.height;
		}
		sdkInstance.logger.log(() -> TAG + " getStyleDimensionsToExclude() : completed");
		return toExclude;
	}
}
