/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.engine

import android.content.Context
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.drawable.Drawable
import android.graphics.drawable.GradientDrawable
import android.view.Gravity
import android.view.View
import android.view.ViewOutlineProvider
import android.widget.FrameLayout
import android.widget.LinearLayout
import android.widget.RelativeLayout
import androidx.annotation.GravityInt
import androidx.core.content.ContextCompat
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.model.ViewDimension
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.InAppInstanceProvider
import com.moengage.inapp.internal.InAppModuleManager
import com.moengage.inapp.internal.MODULE_TAG
import com.moengage.inapp.internal.exceptions.CouldNotCreateViewException
import com.moengage.inapp.internal.model.Border
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.Color
import com.moengage.inapp.internal.model.Margin
import com.moengage.inapp.internal.model.NativeCampaignPayload
import com.moengage.inapp.internal.model.Spacing
import com.moengage.inapp.internal.model.actions.RatingChangeAction
import com.moengage.inapp.internal.model.actions.SetTextAction
import com.moengage.inapp.internal.model.configmeta.InAppConfigMeta
import com.moengage.inapp.internal.model.configmeta.NudgeConfigMeta
import com.moengage.inapp.internal.model.enums.Orientation
import com.moengage.inapp.internal.model.style.InAppStyle
import com.moengage.inapp.model.actions.Action
import com.moengage.inapp.model.enums.InAppPosition

/**
 * Utils class for View Engines
 * @author Umang Chamaria
 */

private val TAG = "${MODULE_TAG}ViewEngineUtils"

internal fun handleDismiss(sdkInstance: SdkInstance, payload: CampaignPayload) {
    InAppInstanceProvider.getControllerForInstance(sdkInstance).viewHandler.handleDismiss(
        payload
    )
}

/**
 * Removes non-intrusive nudge from visible or processing nudges cache
 */
internal fun removeNonIntrusiveNudgeFromCache(
    sdkInstance: SdkInstance,
    campaignPayload: CampaignPayload,
    isVisible: Boolean = false
) {
    sdkInstance.logger.log { "$TAG removeNonIntrusiveNudgeFromCache() : " }
    if (campaignPayload.templateType != InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE) return
    val position = (campaignPayload as NativeCampaignPayload).position
    if (isVisible) {
        InAppModuleManager.removeVisibleNudgePosition(position)
    }
    InAppModuleManager.removeProcessingNudgePosition(position)
    InAppInstanceProvider.getCacheForInstance(sdkInstance)
        .removeFromVisibleOrProcessingNonIntrusiveNudge(campaignPayload.campaignId)
}

/**
 * Generates bitmap from the resource
 * @param context instance of [Context]
 * @param resId resource id
 */
internal fun generateBitmapFromRes(
    sdkInstance: SdkInstance,
    context: Context,
    resId: Int
): Bitmap? {
    try {
        sdkInstance.logger.log {
            "$TAG generateBitmapFromRes(): will generate bitmap for resId: " +
                "$resId"
        }
        val drawable = ContextCompat.getDrawable(context, resId) ?: return null
        drawable.setBounds(
            0,
            0,
            drawable.intrinsicWidth,
            drawable.intrinsicHeight
        )
        val bitmap: Bitmap = Bitmap.createBitmap(
            drawable.intrinsicWidth,
            drawable.intrinsicHeight,
            Bitmap.Config.ARGB_8888
        )
        val canvas = Canvas(bitmap)
        drawable.draw(canvas)
        sdkInstance.logger.log {
            "$TAG generateBitmapFromRes(): generated bitmap for resId: " +
                "$resId"
        }
        return bitmap
    } catch (t: Throwable) {
        sdkInstance.logger.log(LogLevel.ERROR, t) { "$TAG generateBitmapFromRes() : " }
    }
    return null
}

internal fun transformMargin(
    sdkInstance: SdkInstance,
    viewDimension: ViewDimension,
    margin: Margin
): Spacing {
    val spacing = Spacing(
        if (margin.left == 0.0) {
            0
        } else {
            transformViewDimension(
                margin.left,
                viewDimension.width
            )
        },
        if (margin.right == 0.0) {
            0
        } else {
            transformViewDimension(
                margin.right,
                viewDimension.width
            )
        },
        if (margin.top == 0.0) {
            0
        } else {
            transformViewDimension(
                margin.top,
                viewDimension.height
            )
        },
        if (margin.bottom == 0.0) {
            0
        } else {
            transformViewDimension(
                margin.bottom,
                viewDimension.height
            )
        }
    )
    sdkInstance.logger.log { "$TAG transformMargin() : Margin: $spacing" }
    return spacing
}

internal fun transformViewDimension(dimension: Double, containerReference: Int): Int {
    return (dimension * containerReference / 100).toInt()
}

internal fun getViewDimensionsFromPercentage(
    viewDimension: ViewDimension,
    style: InAppStyle
): ViewDimension {
    return ViewDimension(
        transformViewDimension(style.width, viewDimension.width),
        if (style.height == RelativeLayout.LayoutParams.WRAP_CONTENT.toDouble()) {
            RelativeLayout.LayoutParams.WRAP_CONTENT
        } else {
            transformViewDimension(style.height, viewDimension.height)
        }
    )
}

internal fun getScaledBitmap(imageBitmap: Bitmap, bitmapDimension: ViewDimension): Bitmap? {
    return Bitmap.createScaledBitmap(
        imageBitmap,
        bitmapDimension.width,
        bitmapDimension.height,
        true
    )
}

/**
 * Adds gravity to [LinearLayout] according to [Orientation]
 */
internal fun setLayoutGravity(
    layoutParams: LinearLayout.LayoutParams,
    parentOrientation: Orientation
) {
    if (Orientation.VERTICAL == parentOrientation) {
        layoutParams.gravity = Gravity.CENTER_HORIZONTAL
    }
}

/**
 * Adds gravity to [FrameLayout] according to [InAppPosition]
 */
internal fun setLayoutGravity(
    sdkInstance: SdkInstance,
    layoutParams: FrameLayout.LayoutParams,
    inAppPosition: InAppPosition
) {
    layoutParams.gravity = getLayoutGravityFromPosition(sdkInstance, inAppPosition)
}

/**
 * Returns gravity according to [InAppPosition]
 */
@GravityInt
@Throws(CouldNotCreateViewException::class)
private fun getLayoutGravityFromPosition(sdkInstance: SdkInstance, position: InAppPosition): Int {
    sdkInstance.logger.log {
        "$TAG getLayoutGravityFromPosition(): will try to provide gravity for position: $position"
    }
    val gravity = when (position) {
        InAppPosition.TOP -> Gravity.TOP or Gravity.CENTER_HORIZONTAL
        InAppPosition.BOTTOM -> Gravity.BOTTOM or Gravity.CENTER_HORIZONTAL
        InAppPosition.BOTTOM_LEFT -> Gravity.BOTTOM or Gravity.START
        InAppPosition.BOTTOM_RIGHT -> Gravity.BOTTOM or Gravity.END
        else -> throw CouldNotCreateViewException("Unsupported InApp position: $position")
    }
    sdkInstance.logger.log { "$TAG getLayoutGravityFromPosition(): layout gravity: $gravity" }
    return gravity
}

/**
 * Filters [RatingChangeAction] from the list of [Action]
 */
internal fun filterRatingChangeActionFromList(actions: List<Action>?): RatingChangeAction? {
    actions ?: return null
    val filteredActions = actions.filterIsInstance<RatingChangeAction>()

    return if (filteredActions.isEmpty()) {
        null
    } else {
        filteredActions[0]
    }
}

/**
 * Filters [SetTextAction] and sets value to [SetTextAction.content]
 */
internal fun addContentToSetTextAction(
    actions: List<Action>,
    content: String
): List<Action> {
    actions.filterIsInstance<SetTextAction>()
        .onEach { action: SetTextAction -> action.content = content }
    return actions
}

/**
 * Applies background to the view
 *
 * @param view instance of [View]
 * @param drawable instance of [Drawable]
 * @param templateType template type
 * @since 7.0.0
 */
internal fun applyBackgroundToView(view: View, drawable: Drawable, templateType: String) {
    view.background = drawable
    view.outlineProvider = ViewOutlineProvider.BACKGROUND
    view.clipToOutline = true
}

/**
 *
 * @param border instance of [Border]
 * @param densityScale density scale
 * @return instance of [GradientDrawable] with border properties applied
 * @since 7.0.0
 */
internal fun getBorder(border: Border, densityScale: Float): GradientDrawable {
    val drawable = GradientDrawable()
    return getBorder(border, drawable, densityScale)
}

/**
 * Applies border and radius to the [drawable]
 *
 * @param border instance of
 * @param drawable instance of [GradientDrawable]
 * @param densityScale density scale
 * @return instance of [GradientDrawable] with border properties applied
 * @since 7.0.0
 */
internal fun getBorder(
    border: Border,
    drawable: GradientDrawable,
    densityScale: Float
): GradientDrawable {
    if (border.radius != 0.0) {
        drawable.cornerRadius = border.radius.toFloat() * densityScale
    }
    if (border.color != null && border.width != 0.0) {
        drawable.setStroke((border.width * densityScale).toInt(), getColor(border.color))
    }
    return drawable
}

/***
 * Converts [Color] to ARGB int
 */
internal fun getColor(color: Color): Int {
    return android.graphics.Color.argb(
        (color.alpha * 255.0f + 0.5f).toInt(),
        color.red,
        color.green,
        color.blue
    )
}

/**
 * Updates the container padding if required.
 *
 * @param borderWidth border width of the container
 * @param containerLayout instance of [RelativeLayout]
 * @since 7.1.1
 */
internal fun updateContainerPaddingIfRequired(borderWidth: Int, containerLayout: RelativeLayout) {
    if (borderWidth != 0) {
        val paddingSpacing = Spacing(
            containerLayout.paddingLeft,
            containerLayout.paddingRight,
            containerLayout.paddingTop,
            containerLayout.paddingBottom
        )
        containerLayout.setPadding(
            paddingSpacing.left + borderWidth,
            paddingSpacing.top + borderWidth,
            paddingSpacing.right + borderWidth,
            paddingSpacing.bottom + borderWidth
        )
    }
}

/**
 * Removes non-intrusive nudge from visible or processing nudges cache
 * @param sdkInstance instance of [SdkInstance]
 * @param inAppConfigMeta instance of [InAppConfigMeta] Holds Meta Data of InApp Campaign
 * @since 7.1.2
 */
internal fun removeNonIntrusiveNudgeFromCache(
    sdkInstance: SdkInstance,
    inAppConfigMeta: InAppConfigMeta
) {
    if (inAppConfigMeta !is NudgeConfigMeta) return
    sdkInstance.logger.log { "$TAG removeNonIntrusiveNudgeFromCache() : $inAppConfigMeta" }
    InAppModuleManager.removeVisibleNudgePosition(inAppConfigMeta.position)
    InAppModuleManager.removeProcessingNudgePosition(inAppConfigMeta.position)
    InAppInstanceProvider.getCacheForInstance(sdkInstance)
        .removeFromVisibleOrProcessingNonIntrusiveNudge(inAppConfigMeta.campaignId)
}