/*
 * Copyright (c) 2014-2023 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.engine

import android.app.Activity
import android.graphics.drawable.GradientDrawable
import android.widget.LinearLayout
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.model.ViewDimension
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.MODULE_TAG
import com.moengage.inapp.internal.model.InAppWidget
import com.moengage.inapp.internal.model.NativeCampaignPayload
import com.moengage.inapp.internal.model.ViewCreationMeta
import com.moengage.inapp.internal.model.customrating.CustomRatingComponent
import com.moengage.inapp.internal.model.enums.Orientation
import com.moengage.inapp.internal.model.enums.RatingType
import com.moengage.inapp.internal.model.style.CustomRatingStyle
import com.moengage.inapp.internal.widgets.ratingbar.MoECustomRatingBar

/**
 * Utility class for creating widgets
 * @author Arshiya Khanum
 * @since 7.0.0
 */

internal class WidgetFactory(
    private val context: Activity,
    private val sdkInstance: SdkInstance,
    private val viewCreationMeta: ViewCreationMeta,
    private val payload: NativeCampaignPayload,
    private val densityScale: Float
) {

    private val tag = "${MODULE_TAG}WidgetCreator"

    /**
     * Creates [MoECustomRatingBar] widget
     * @param widget instance of [InAppWidget]
     * @param parentOrientation [Orientation]
     * @return [MoECustomRatingBar] object
     * @since 7.0.0
     */
    fun createCustomRatingBar(
        widget: InAppWidget,
        parentOrientation: Orientation,
        toExclude: ViewDimension
    ): MoECustomRatingBar {
        sdkInstance.logger
            .log { "$tag createCustomRatingBar() : Will create rating widget: $widget" }
        val ratingBar = MoECustomRatingBar(context, RatingType.STAR)
        ratingBar.setIsIndicator(false)
        val style = widget.component.style as CustomRatingStyle
        ratingBar.numStars = style.numberOfRatings
        ratingBar.stepSize = 1.0f
        val customRatingComponent = widget.component as CustomRatingComponent
        ratingBar.setRatingIcons(customRatingComponent.ratingIcons)
        val campaignDimension = ViewDimension(
            getViewDimensionsFromPercentage(
                viewCreationMeta.deviceDimensions,
                style
            ).width,
            (style.realHeight * densityScale).toInt()
        )
        if (payload.templateType === InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE) {
            // NOTE:This is only for nudges in order to fix the background image with border and
            // rounded corners issue
            campaignDimension.width -= toExclude.width
        }
        sdkInstance.logger
            .log { "$tag createCustomRatingBar() : Campaign dimensions: $campaignDimension" }
        val layoutParams =
            LinearLayout.LayoutParams(campaignDimension.width, campaignDimension.height)
        setLayoutGravity(layoutParams, parentOrientation)
        val marginSpacing =
            transformMargin(sdkInstance, viewCreationMeta.deviceDimensions, style.margin)
        layoutParams.setMargins(
            marginSpacing.left,
            marginSpacing.top,
            marginSpacing.right,
            marginSpacing.bottom
        )
        ratingBar.layoutParams = layoutParams
        val drawable = GradientDrawable()
        if (style.border != null) {
            getBorder(style.border, drawable, densityScale)
        }
        applyBackgroundToView(ratingBar, drawable, payload.templateType)

        sdkInstance.logger.log {
            "$tag createCustomRatingBar() : MoECustomRatingBar created successfully."
        }
        return ratingBar
    }
}