/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.html

import android.app.Activity
import android.view.View
import android.webkit.JavascriptInterface
import com.moengage.core.LogLevel
import com.moengage.core.Properties
import com.moengage.core.analytics.MoEAnalyticsHelper
import com.moengage.core.internal.EVENT_APP_RATED
import com.moengage.core.internal.USER_ATTRIBUTE_USER_BDAY
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.MoEUtils
import com.moengage.core.model.GeoLocation
import com.moengage.core.model.UserGender
import com.moengage.inapp.internal.ActionHandler
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.InAppConstants.IN_APP_RATING_ATTRIBUTE
import com.moengage.inapp.internal.addAttributesToProperties
import com.moengage.inapp.internal.isValidJavaScriptString
import com.moengage.inapp.internal.isValidJavaScriptValue
import com.moengage.inapp.internal.model.HtmlCampaignPayload
import com.moengage.inapp.internal.model.actions.CallAction
import com.moengage.inapp.internal.model.actions.CopyAction
import com.moengage.inapp.internal.model.actions.DismissAction
import com.moengage.inapp.internal.model.actions.NavigateToSettingsAction
import com.moengage.inapp.internal.model.actions.ShareAction
import com.moengage.inapp.internal.model.actions.SmsAction
import com.moengage.inapp.internal.trackInAppClicked
import com.moengage.inapp.internal.trackInAppDismissed
import com.moengage.inapp.model.CampaignData
import com.moengage.inapp.model.actions.Action
import com.moengage.inapp.model.actions.CustomAction
import com.moengage.inapp.model.actions.NavigationAction
import com.moengage.inapp.model.actions.RequestNotificationAction
import com.moengage.inapp.model.enums.ActionType
import com.moengage.inapp.model.enums.NavigationType
import org.json.JSONObject

/**
 *
 * @author Arshiya Khanum
 */
private const val USER_ATTRIBUTE_NAME = "name"
private const val USER_ATTRIBUTE_VALUE = "value"
private const val USER_ATTRIBUTE_WIDGET_ID = "widgetId"
private const val USER_ATTRIBUTE_RATING = "rating"
private const val USER_ATTRIBUTE_LATITUDE = "latitude"
private const val USER_ATTRIBUTE_LONGITUDE = "longitude"

internal class HtmlJavaScriptInterface(
    private val activity: Activity,
    private val payload: HtmlCampaignPayload,
    private val htmlInAppView: View?,
    private val sdkInstance: SdkInstance
) {

    private val tag = "${InAppConstants.MODULE_TAG}HtmlJavaScriptInterface"
    private val webCallbackParser = WebPayloadParser()
    private val actionHandler = ActionHandler(activity, sdkInstance)
    private val context = activity.applicationContext
    private val instanceId = sdkInstance.instanceMeta.instanceId

    @JavascriptInterface
    fun trackEvent(
        eventName: String?,
        generalAttrJson: String?,
        locationAttrJson: String?,
        dateAttrJson: String?,
        isNonInteractive: Boolean,
        shouldAttachCampaignMeta: Boolean
    ) {
        try {
            sdkInstance.logger.log {
                "$tag trackEvent() : " +
                    "eventName: $eventName," +
                    " generalAttrJson: $generalAttrJson, " +
                    "locationAttrJson: $locationAttrJson, " +
                    "dateAttrJson: $dateAttrJson, " +
                    "isNonInteractive: $isNonInteractive, " +
                    "shouldAttachCampaignMeta: $shouldAttachCampaignMeta"
            }

            if (eventName.isNullOrBlank() || !isValidJavaScriptString(eventName)) return

            val properties = webCallbackParser.toProperties(
                generalAttrJson,
                locationAttrJson,
                dateAttrJson,
                isNonInteractive
            )
            if (shouldAttachCampaignMeta) {
                addAttributesToProperties(
                    properties,
                    payload.campaignId,
                    payload.campaignName,
                    payload.campaignContext
                )
            }
            MoEAnalyticsHelper.trackEvent(
                context,
                eventName,
                properties,
                instanceId
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag trackEvent() : " }
        }
    }

    @JavascriptInterface
    fun trackClick(payload: String?) {
        try {
            sdkInstance.logger.log { "$tag trackClick() : payload: $payload" }
            if (!isValidJavaScriptValue(payload)) return

            val widgetId: Any? = if (!payload.isNullOrBlank()) {
                val widgetIdJson = JSONObject(payload)
                widgetIdJson.opt(USER_ATTRIBUTE_WIDGET_ID)
            } else {
                null
            }

            trackInAppClicked(
                context,
                sdkInstance,
                CampaignData(
                    this.payload.campaignId,
                    this.payload.campaignName,
                    this.payload.campaignContext
                ),
                widgetId
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag trackClick() : " }
        }
    }

    @JavascriptInterface
    fun trackDismiss() {
        try {
            sdkInstance.logger.log { "$tag trackDismiss() : " }
            trackInAppDismissed(
                context,
                sdkInstance,
                CampaignData(payload.campaignId, payload.campaignName, payload.campaignContext)
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag trackDismiss() : " }
        }
    }

    @JavascriptInterface
    fun trackRating(payload: String?) {
        try {
            sdkInstance.logger.log { "$tag trackRating() : $payload" }
            if (payload.isNullOrBlank() || !isValidJavaScriptString(payload) ||
                !isValidJavaScriptValue(payload)
            ) {
                return
            }
            val ratingJson = JSONObject(payload)
            val rating = ratingJson.getDouble(USER_ATTRIBUTE_RATING)
            val properties = Properties().addAttribute(IN_APP_RATING_ATTRIBUTE, rating)
            addAttributesToProperties(
                properties,
                this.payload.campaignId,
                this.payload.campaignName,
                this.payload.campaignContext
            )
            MoEAnalyticsHelper.trackEvent(
                context,
                EVENT_APP_RATED,
                properties,
                instanceId
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag trackRating() : " }
        }
    }

    @JavascriptInterface
    fun setAlias(alias: String?) {
        try {
            sdkInstance.logger.log { "$tag setAlias() : alias $alias" }
            if (alias.isNullOrBlank() || !isValidJavaScriptString(alias)) return

            MoEAnalyticsHelper
                .setAlias(context, alias, instanceId)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setAlias() : " }
        }
    }

    @JavascriptInterface
    fun setUniqueId(uniqueId: String?) {
        try {
            sdkInstance.logger.log { "$tag setUniqueId() : uniqueId: $uniqueId" }
            if (uniqueId.isNullOrBlank() || !isValidJavaScriptString(uniqueId)) return

            MoEAnalyticsHelper.setUniqueId(context, uniqueId, instanceId)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setUniqueId() : " }
        }
    }

    @JavascriptInterface
    fun setUserName(userName: String?) {
        try {
            sdkInstance.logger.log { "$tag setUserName() : username: $userName" }
            if (userName.isNullOrBlank() || !isValidJavaScriptString(userName)) return

            MoEAnalyticsHelper.setUserName(context, userName, instanceId)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setUserName() : " }
        }
    }

    @JavascriptInterface
    fun setFirstName(firstName: String?) {
        try {
            sdkInstance.logger.log { "$tag setFirstName() : first name: $firstName" }
            if (firstName.isNullOrBlank() || !isValidJavaScriptString(firstName)) return

            MoEAnalyticsHelper.setFirstName(context, firstName, instanceId)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setFirstName() : " }
        }
    }

    @JavascriptInterface
    fun setLastName(lastName: String?) {
        try {
            sdkInstance.logger.log { "$tag setLastName() : last name: $lastName" }
            if (lastName.isNullOrBlank() || !isValidJavaScriptString(lastName)) return

            MoEAnalyticsHelper.setLastName(context, lastName, instanceId)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setLastName() : " }
        }
    }

    @JavascriptInterface
    fun setEmailId(emailId: String?) {
        try {
            sdkInstance.logger.log { "$tag setEmailId() : emailId: $emailId" }
            if (emailId.isNullOrBlank() || !isValidJavaScriptString(emailId)) return

            MoEAnalyticsHelper.setEmailId(context, emailId, instanceId)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setEmailId() : " }
        }
    }

    @JavascriptInterface
    fun setMobileNumber(mobileNumber: String?) {
        try {
            sdkInstance.logger.log { "$tag setMobileNumber() : mobile number: $mobileNumber" }
            if (mobileNumber.isNullOrBlank() || !isValidJavaScriptString(mobileNumber)) return

            MoEAnalyticsHelper.setMobileNumber(context, mobileNumber, instanceId)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setMobileNumber() : " }
        }
    }

    @JavascriptInterface
    fun setGender(gender: String?) {
        try {
            sdkInstance.logger.log { "$tag setGender() : gender: $gender" }
            if (gender.isNullOrBlank() || !isValidJavaScriptString(gender)) return

            MoEAnalyticsHelper
                .setGender(context, UserGender.valueOf(gender.uppercase()), instanceId)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setGender() : " }
        }
    }

    @JavascriptInterface
    fun setBirthDate(birthDate: String?) {
        try {
            sdkInstance.logger.log { "$tag setBirthDate() : birthdate: $birthDate" }
            if (birthDate.isNullOrBlank() || !isValidJavaScriptString(birthDate)) return

            MoEAnalyticsHelper.setUserAttributeISODate(
                context,
                USER_ATTRIBUTE_USER_BDAY,
                birthDate,
                instanceId
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setBirthDate() : " }
        }
    }

    @JavascriptInterface
    fun setUserLocation(payload: String?) {
        try {
            sdkInstance.logger.log { "$tag setUserLocation() : $payload" }
            if (payload.isNullOrBlank() || !isValidJavaScriptString(payload) ||
                !isValidJavaScriptValue(payload)
            ) {
                return
            }
            val locationJson = JSONObject(payload)
            MoEAnalyticsHelper.setLocation(
                context,
                locationJson.getDouble(USER_ATTRIBUTE_LATITUDE),
                locationJson.getDouble(USER_ATTRIBUTE_LONGITUDE),
                instanceId
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setUserLocation() : " }
        }
    }

    @JavascriptInterface
    fun setUserAttribute(userAttrJson: String?) {
        try {
            sdkInstance.logger.log { "$tag setUserAttribute() : userAttrJson: $userAttrJson" }
            if (userAttrJson.isNullOrBlank() || !isValidJavaScriptString(userAttrJson) ||
                !isValidJavaScriptValue(userAttrJson)
            ) {
                return
            }
            val attributeJson = JSONObject(userAttrJson)
            val name = attributeJson.getString(USER_ATTRIBUTE_NAME)
            when (val value: Any? = attributeJson.get(USER_ATTRIBUTE_VALUE)) {
                is Int -> MoEAnalyticsHelper.setUserAttribute(
                    context,
                    name,
                    value,
                    sdkInstance.instanceMeta.instanceId
                )
                is Boolean ->
                    MoEAnalyticsHelper
                        .setUserAttribute(context, name, value, instanceId)
                is Double ->
                    MoEAnalyticsHelper
                        .setUserAttribute(context, name, value, instanceId)
                is Float ->
                    MoEAnalyticsHelper
                        .setUserAttribute(context, name, value, instanceId)
                is Long ->
                    MoEAnalyticsHelper
                        .setUserAttribute(context, name, value, instanceId)
                is String ->
                    MoEAnalyticsHelper
                        .setUserAttribute(context, name, value, instanceId)
                else -> sdkInstance.logger.log(LogLevel.ERROR) { "$tag setUserAttribute() : name: $name value: $value, unsupported data type." }
            }
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setUserAttribute() : " }
        }
    }

    @JavascriptInterface
    fun setUserAttributeDate(name: String?, isoDate: String?) {
        try {
            sdkInstance.logger.log { "$tag setUserAttributeDate() : name: $name, iso date: $isoDate" }
            if (name.isNullOrBlank() || !isValidJavaScriptString(name) || isoDate.isNullOrBlank() ||
                !isValidJavaScriptString(isoDate)
            ) {
                return
            }

            MoEAnalyticsHelper
                .setUserAttributeISODate(context, name, isoDate, instanceId)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setUserAttributeDate() : " }
        }
    }

    @JavascriptInterface
    fun setUserAttributeLocation(payload: String?) {
        try {
            sdkInstance.logger.log { "$tag setUserAttributeLocation() : $payload" }
            if (payload.isNullOrBlank() || !isValidJavaScriptString(payload)
            ) {
                return
            }
            val jsonObject = JSONObject(payload)
            val name = jsonObject.getString(USER_ATTRIBUTE_NAME)
            if (name.isNullOrBlank() || !isValidJavaScriptString(name)) return

            MoEAnalyticsHelper
                .setUserAttribute(
                    context,
                    name,
                    GeoLocation(
                        jsonObject.getDouble(USER_ATTRIBUTE_LATITUDE),
                        jsonObject.getDouble(USER_ATTRIBUTE_LONGITUDE)
                    ),
                    instanceId
                )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag setUserAttributeLocation() : " }
        }
    }

    @JavascriptInterface
    fun navigateToScreen(screenName: String?, dataJson: String?) {
        try {
            if (screenName.isNullOrBlank() || !isValidJavaScriptString(screenName)) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag navigateToScreen() : screenName: $screenName is invalid. Not processing." }
                return
            }
            processAction(
                NavigationAction(
                    ActionType.NAVIGATE,
                    NavigationType.SCREEN,
                    screenName,
                    toKeyValuePairs(dataJson)
                )
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag navigateToScreen() : " }
        }
    }

    @JavascriptInterface
    fun openDeepLink(deepLinkUrl: String?, dataJson: String?) {
        try {
            if (deepLinkUrl.isNullOrBlank() || !isValidJavaScriptString(deepLinkUrl)) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag openDeepLink() : url: $deepLinkUrl is invalid. Not processing." }
                return
            }
            processAction(
                NavigationAction(
                    ActionType.NAVIGATE,
                    NavigationType.DEEP_LINKING,
                    deepLinkUrl,
                    toKeyValuePairs(dataJson)
                )
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag openDeepLink() : " }
        }
    }

    @JavascriptInterface
    fun openRichLanding(url: String?, dataJson: String?) {
        try {
            if (url.isNullOrBlank() || !isValidJavaScriptString(url)) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag openRichLanding() : url: $url is invalid. Not processing." }
                return
            }
            processAction(
                NavigationAction(
                    ActionType.NAVIGATE,
                    NavigationType.RICH_LANDING,
                    url,
                    toKeyValuePairs(dataJson)
                )
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag openRichLanding() : " }
        }
    }

    @JavascriptInterface
    fun openWebURL(webUrl: String?, dataJson: String?) {
        try {
            if (webUrl.isNullOrBlank() || !isValidJavaScriptString(webUrl)) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag openWebURL() : $webUrl is invalid. Not processing." }
                return
            }
            processAction(
                NavigationAction(
                    ActionType.NAVIGATE,
                    NavigationType.DEEP_LINKING,
                    webUrl,
                    toKeyValuePairs(dataJson)
                )
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag openWebURL() : " }
        }
    }

    @JavascriptInterface
    fun dismissMessage() {
        try {
            activity.runOnUiThread {
                processAction(DismissAction(ActionType.DISMISS))
            }
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag dismissMessage() : " }
        }
    }

    @JavascriptInterface
    fun copyText(textToCopy: String?, message: String?) {
        try {
            sdkInstance.logger.log { "$tag copyText() : text to copy: $textToCopy, message: $message" }
            if (textToCopy.isNullOrBlank() || !isValidJavaScriptString(textToCopy)) return

            processAction(
                CopyAction(
                    ActionType.COPY_TEXT,
                    if (isValidJavaScriptString(message)) message else null,
                    textToCopy
                )
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag copyText() : " }
        }
    }

    @JavascriptInterface
    fun call(mobileNumber: String?) {
        try {
            sdkInstance.logger.log { "$tag call() : mobile number: $mobileNumber" }
            if (mobileNumber.isNullOrBlank() || !isValidJavaScriptString(mobileNumber)) return

            processAction(
                CallAction(ActionType.CALL, mobileNumber)
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag call() : " }
        }
    }

    @JavascriptInterface
    fun sms(mobileNumber: String?, message: String?) {
        try {
            sdkInstance.logger.log { "$tag sms() : mobile number: $mobileNumber, message: $message" }
            if (mobileNumber.isNullOrBlank() || !isValidJavaScriptString(mobileNumber) ||
                message.isNullOrBlank() || !isValidJavaScriptString(message)
            ) {
                return
            }

            processAction(
                SmsAction(ActionType.SMS, mobileNumber, message)
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag sms() : " }
        }
    }

    @JavascriptInterface
    fun share(content: String?) {
        try {
            sdkInstance.logger.log { "$tag share() : content: $content" }
            if (content.isNullOrBlank() || !isValidJavaScriptString(content)) return

            processAction(
                ShareAction(ActionType.SHARE, content)
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag share() : " }
        }
    }

    @JavascriptInterface
    fun customAction(dataJson: String?) {
        try {
            sdkInstance.logger.log { "$tag customAction() : DataJson: $dataJson" }
            if (!isValidJavaScriptString(dataJson)) return
            processAction(
                CustomAction(ActionType.CUSTOM_ACTION, toKeyValuePairs(dataJson))
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag customAction() : " }
        }
    }

    private fun processAction(action: Action) {
        val view = htmlInAppView ?: return
        actionHandler.onActionPerformed(
            view,
            action,
            payload
        )
    }

    private fun toKeyValuePairs(dataJson: String?): MutableMap<String, Any>? {
        return if (!isValidJavaScriptString(dataJson) || dataJson.isNullOrBlank()) {
            null
        } else {
            MoEUtils.jsonToMap(
                JSONObject(dataJson)
            )
        }
    }

    @JavascriptInterface
    fun requestNotificationPermission() {
        try {
            sdkInstance.logger.log { "$tag requestNotificationPermission() : " }
            processAction(
                RequestNotificationAction(
                    ActionType.REQUEST_NOTIFICATION_PERMISSION,
                    -1
                )
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag requestNotificationPermission() : " }
        }
    }

    @JavascriptInterface
    fun navigateToNotificationSettings() {
        try {
            sdkInstance.logger.log { "$tag navigateToNotificationSettings() : " }
            processAction(
                NavigateToSettingsAction(
                    ActionType.NAVIGATE_SETTINGS_NOTIFICATIONS
                )
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag navigateToNotificationSettings() : " }
        }
    }
}