/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.model.meta;

import static com.moengage.inapp.internal.repository.ModelToJsonMapperKt.triggerToJson;

import androidx.annotation.Nullable;

import com.moengage.core.LogLevel;
import com.moengage.core.internal.logger.Logger;
import com.moengage.core.internal.utils.ApiUtilsKt;
import com.moengage.core.internal.utils.TimeUtilsKt;
import com.moengage.inapp.internal.model.enums.CampaignSubType;
import com.moengage.inapp.internal.model.enums.InAppType;
import com.moengage.inapp.internal.model.enums.ScreenOrientation;
import com.moengage.inapp.model.CampaignContext;
import com.moengage.inapp.model.enums.InAppPosition;

import org.json.JSONObject;

import java.util.Set;

/**
 * InAppCampaign meta data.
 *
 * @author Umang Chamaria
 */
public class CampaignMeta {

  private static final String TAG = "CampaignMeta";
  /**
   * Unique identifier for the campaign.
   */
  public final String campaignId;
  /**
   * Name of the campaign.
   */
  public final String campaignName;
  /**
   * Expiry time of the campaign.
   */
  public final long expiryTime;
  /**
   * Last time the campaign was updated.
   */
  public final long lastUpdatedTime;
  /**
   * Rules on displaying or dismissing the campaign.
   */
  public final DisplayControl displayControl;
  /**
   * Type of template. Full Screen, Pop up or Self Handled or Non-Intrusive.
   */
  public final String templateType;
  /**
   * Control the delivery parameters of the campaign.
   */
  public final DeliveryControl deliveryControl;
  /**
   * Condition on which the in-app should be triggered.
   */
  @Nullable public final Trigger trigger;

  /**
   * Campaign context attributes.
   */
  @Nullable public final CampaignContext campaignContext;
  /**
   * Type of InApp. HTML or Nativ.
   */
  @Nullable public InAppType inAppType;
  /**
   * List of supported screen orientations [LANDSCAPE, PORTRAIT].
   */
  public Set<ScreenOrientation> supportedOrientations;
  /**
   * Campaign sub-type
   */
  public final CampaignSubType campaignSubType;

  /**
   * Position of Non-Intrusive InApp. [TOP, BOTTOM, BOTTOM_LEFT, BOTTOM_RIGHT]
   */
  public final InAppPosition position;

  public CampaignMeta(String campaignId,
      String campaignName,
      long expiryTime,
      long lastUpdatedTime,
      DisplayControl displayControl,
      String templateType,
      DeliveryControl deliveryControl,
      @Nullable Trigger trigger,
      @Nullable CampaignContext campaignContext,
      @Nullable InAppType inAppType,
      Set<ScreenOrientation> supportedOrientations,
      CampaignSubType campaignSubType,
      InAppPosition position) {
    this.campaignId = campaignId;
    this.campaignName = campaignName;
    this.expiryTime = expiryTime;
    this.lastUpdatedTime = lastUpdatedTime;
    this.displayControl = displayControl;
    this.templateType = templateType;
    this.deliveryControl = deliveryControl;
    this.trigger = trigger;
    this.campaignContext = campaignContext;
    this.inAppType = inAppType;
    this.supportedOrientations = supportedOrientations;
    this.campaignSubType = campaignSubType;
    this.position = position;
  }

  @Override public String toString() {
    try {
      JSONObject jsonObject = toJson(this);
      if (jsonObject != null) {
        return jsonObject.toString(4);
      }
    } catch (Throwable t) {
      Logger.print(LogLevel.ERROR, t, () -> TAG + " toString() ");
    }
    return super.toString();
  }

  public static JSONObject toJson(CampaignMeta meta) {
    try {
      JSONObject metaJson = new JSONObject();
      metaJson.put(CAMPAIGN_ID, meta.campaignId)
          .put(CAMPAIGN_NAME, meta.campaignName)
          .put(EXPIRY_TIME, TimeUtilsKt.isoStringFromSeconds(meta.expiryTime))
          .put(LAST_UPDATED_TIME, TimeUtilsKt.isoStringFromSeconds(meta.lastUpdatedTime))
          .put(DISPLAY_CONTROL, DisplayControl.toJson(meta.displayControl))
          .put(TEMPLATE_TYPE, meta.templateType)
          .put(DELIVERY_CONTROL, DeliveryControl.toJson(meta.deliveryControl))
          .put(TRIGGER, triggerToJson(meta.trigger))
          .put(CAMPAIGN_CONTEXT, meta.campaignContext)
          .put(CAMPAIGN_SUB_TYPE, meta.campaignSubType.toString().toLowerCase());

      if (meta.campaignContext != null) {
        metaJson.put(CAMPAIGN_CONTEXT, meta.campaignContext.getPayload());
      }

      if (meta.inAppType != null) {
        metaJson.put(INAPP_TYPE, meta.inAppType.toString());
      }

      if (meta.supportedOrientations != null) {
        metaJson.put(ORIENTATIONS, ApiUtilsKt.setToJsonArray(meta.supportedOrientations));
      }

      if(meta.position != null){
        metaJson.put(POSITION, meta.position.toString().toLowerCase());
      }

      return metaJson;
    } catch (Throwable t) {
      Logger.print(LogLevel.ERROR, t, () -> TAG + "toJson()");
    }
    return null;
  }

  @Override public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    CampaignMeta meta = (CampaignMeta) o;

    if (expiryTime != meta.expiryTime) return false;
    if (lastUpdatedTime != meta.lastUpdatedTime) return false;
    if (!campaignId.equals(meta.campaignId)) return false;
    if (!campaignName.equals(meta.campaignName)) return false;
    if (!displayControl.equals(meta.displayControl)) return false;
    if (!templateType.equals(meta.templateType)) return false;
    if (!deliveryControl.equals(meta.deliveryControl)) return false;
    if (campaignContext != null ? !campaignContext.equals(meta.campaignContext)
        : meta.campaignContext == null) {
      return false;
    }
    if (!(trigger != null ? trigger.equals(meta.trigger) : meta.trigger == null)) return false;
    if (inAppType != meta.inAppType) return false;
    if(position != meta.position) return false;
    return  (supportedOrientations.equals(meta.supportedOrientations));
  }

  private @Nullable static CampaignContext campaignContextFromJson(@Nullable JSONObject jsonObject) {
    if (jsonObject == null) return null;
    return CampaignContext.fromJson(jsonObject);
  }

  private static final String CAMPAIGN_ID = "campaign_id";
  private static final String CAMPAIGN_NAME = "campaign_name";
  private static final String EXPIRY_TIME = "expiry_time";
  private static final String LAST_UPDATED_TIME = "updated_time";
  private static final String DISPLAY_CONTROL = "display";
  private static final String TEMPLATE_TYPE = "template_type";
  private static final String DELIVERY_CONTROL = "delivery";
  private static final String TRIGGER = "trigger";
  private static final String CAMPAIGN_CONTEXT = "campaign_context";
  private static final String INAPP_TYPE = "inapp_type";
  private static final String ORIENTATIONS = "orientations";
  private static final String CAMPAIGN_SUB_TYPE = "campaign_sub_type";
  private static final String POSITION = "position";
}
