/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.inapp.internal.repository

import com.moengage.core.internal.model.Event
import com.moengage.inapp.internal.DelayedInAppData
import com.moengage.inapp.internal.InAppModuleManager
import com.moengage.inapp.internal.ScreenData
import com.moengage.inapp.internal.model.meta.InAppCampaign
import com.moengage.inapp.listeners.InAppLifeCycleListener
import com.moengage.inapp.listeners.OnClickActionListener
import com.moengage.inapp.listeners.SelfHandledAvailableListener
import com.moengage.inapp.model.enums.InAppPosition
import java.lang.ref.WeakReference
import java.util.Collections

/**
 * @author Umang Chamaria
 * Date: 2019-07-15
 */
internal class InAppCache {

    var generalCampaign: List<InAppCampaign> = emptyList()
        private set

    var primaryTriggerEvents: Set<String> = emptySet()
        private set

    var selfHandledCampaign: List<InAppCampaign> = emptyList()
        private set

    fun updateCache(repository: InAppRepository) {
        val mapper = PayloadMapper()
        generalCampaign = mapper.entityToCampaign(repository.getGeneralCampaigns())
        primaryTriggerEvents = repository.getPrimaryTriggerEvents()
        selfHandledCampaign = mapper.entityToCampaign(repository.getSelfHandledCampaign())
        nonIntrusiveNudgeCampaigns =
            updateNonIntrusiveNudgeCampaigns(mapper.entityToCampaign(repository.getNonIntrusiveNudgeCampaigns()))
    }

    var selfHandledListener: SelfHandledAvailableListener? = null

    var clickActionListener: OnClickActionListener? = null

    val lifeCycleListeners = mutableListOf<InAppLifeCycleListener>()

    val pendingTriggerEvents = mutableSetOf<Event>()

    var inAppContext = emptySet<String>()

    var lastScreenData: ScreenData? = null
        private set

    /**
     * Stores the screen name and orientation of the screen.
     */
    fun updateLastScreenData(screenData: ScreenData) {
        lastScreenData = screenData
    }

    var hasHtmlCampaignSetupFailed = false

    /**
     * Cached Listener for the SelfHandled if getSelfHandledInApp called before meta sync
     */
    var pendingSelfHandledListener: WeakReference<SelfHandledAvailableListener?> =
        WeakReference(null)

    /**
     * Cached scheduled delay in-app campaigns
     */
    var scheduledCampaigns: MutableMap<String, DelayedInAppData> =
        Collections.synchronizedMap(mutableMapOf())

    /**
     * Cache of available Non-Intrusive nudge campaigns
     */
    var nonIntrusiveNudgeCampaigns: Map<InAppPosition, List<InAppCampaign>> =
        Collections.synchronizedMap(mapOf<InAppPosition, List<InAppCampaign>>())
        private set

    /**
     * Cache for the visible/processing campaigns mapped to activity
     */
    val activityBasedVisibleOrProcessingCampaigns: MutableMap<String, MutableSet<String>> = Collections.synchronizedMap(
        mutableMapOf<String, MutableSet<String>>()
    )

    /**
     * Cache for all the visible or processing campaigns
     */
    val visibleOrProcessingCampaigns: MutableSet<String> = Collections.synchronizedSet(mutableSetOf<String>())

    /**
     * Cache of pending non-intrusive nudges calls
     */
    private val pendingNudgeCalls = Collections.synchronizedList(arrayListOf<InAppPosition>())

    /**
     * Build cache for available non-intrusive nudge campaigns.
     *
     * @return [Map] of [InAppPosition] to [List] of [InAppCampaign]
     */
    private fun updateNonIntrusiveNudgeCampaigns(nonIntrusiveNudgeCampaigns: List<InAppCampaign>): Map<InAppPosition, List<InAppCampaign>> {
        val map = mutableMapOf<InAppPosition, MutableList<InAppCampaign>>()
        for (nudge in nonIntrusiveNudgeCampaigns) {
            if (map.containsKey(nudge.campaignMeta.position)) {
                map[nudge.campaignMeta.position]?.add(nudge)
            } else {
                map[nudge.campaignMeta.position] = mutableListOf(nudge)
            }
        }
        return map
    }

    /**
     * Adds [campaignId] and [position] to cache of [activityBasedVisibleOrProcessingCampaigns]
     */
    fun addVisibleOrProcessingNonIntrusiveNudge(campaignId: String) {
        val currentActivityName = InAppModuleManager.getNotNullCurrentActivityName()
        if (activityBasedVisibleOrProcessingCampaigns.containsKey(currentActivityName)) {
            activityBasedVisibleOrProcessingCampaigns[currentActivityName]?.add(campaignId)
        } else {
            activityBasedVisibleOrProcessingCampaigns[currentActivityName] =
                mutableSetOf(campaignId)
        }
        visibleOrProcessingCampaigns.add(campaignId)
    }

    /**
     * Removes [campaignId] from the cache of [activityBasedVisibleOrProcessingCampaigns]
     */
    fun removeFromVisibleOrProcessingNonIntrusiveNudge(campaignId: String) {
        activityBasedVisibleOrProcessingCampaigns[InAppModuleManager.getNotNullCurrentActivityName()]?.remove(
            campaignId
        )
        visibleOrProcessingCampaigns.remove(campaignId)
    }

    /**
     * Clears all the pending nudges calls
     */
    fun clearPendingNudgesCalls() {
        pendingNudgeCalls.clear()
    }

    /**
     * Provides pending nudge position call if any, otherwise returns null
     */
    fun getPendingNudgeCalls(): MutableList<InAppPosition> {
        return pendingNudgeCalls
    }

    /**
     * Adds [position] to the pending nudges calls cache
     */
    fun addToPendingNudgeCall(position: InAppPosition) {
        pendingNudgeCalls.add(position)
    }

    /**
     * Checks if the campaign is present in visible and processing nudges cache
     *
     * @param campaignId Campaign Identifier
     * @return true if campaign is present in the cache, otherwise false.
     */
    fun isCampaignVisibleOrProcessing(campaignId: String): Boolean {
        return visibleOrProcessingCampaigns.contains(campaignId)
    }
}