/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.repository

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.net.Uri
import com.moengage.core.LogLevel
import com.moengage.core.internal.lifecycle.LifecycleManager
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.storage.FileManager
import com.moengage.core.internal.utils.downloadImageBitmap
import com.moengage.core.internal.utils.getSha256ForString
import com.moengage.inapp.internal.MODULE_TAG
import java.io.File
import java.net.URL
import java.security.NoSuchAlgorithmException
import java.util.concurrent.CountDownLatch
import java.util.concurrent.Executors

/**
 * @author Arshiya Khanum
 */

private const val FILE_PATH_SEPARATOR = "/"
private const val HTML_SUBFOLDER = "html"

public class InAppFileManager constructor(
    context: Context,
    private val sdkInstance: SdkInstance
) {

    private val tag = "${MODULE_TAG}InAppFileManager"
    private var fileManager: FileManager = FileManager(context, sdkInstance)

    public fun getImageFromUrl(context: Context, url: String, campaignId: String): Bitmap? {
        try {
            return if (isRemoteResource(url)) {
                getRemoteImage(url, campaignId)
            } else {
                getBundledImageIfPresent(context, url)
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag  getImageFromUrl() : " }
        }
        return null
    }

    public fun deleteImagesForCampaignIds(campaignIdList: Set<String>?) {
        sdkInstance.logger.log { "$tag deleteImagesForCampaignIds() : Deleting images for campaigns" }
        if (campaignIdList == null) return
        for (campaignId in campaignIdList) {
            campaignId.let { fileManager.deleteFolder(it) }
        }
    }

    @Throws(NoSuchAlgorithmException::class)
    private fun getRemoteImage(imageURI: String, campaignId: String): Bitmap? {
        val fileName = getSha256ForString(imageURI)
        return if (fileManager.fileExistsInDirectory(campaignId, fileName)) {
            BitmapFactory.decodeFile(fileManager.getPathForFile(campaignId, fileName))
        } else {
            sdkInstance.logger.log { "$tag getRemoteImage() : Downloading image, url - $imageURI" }
            val bitmap = downloadImageBitmap(imageURI) ?: return null
            fileManager.saveImageFile(campaignId, fileName, bitmap)
            bitmap
        }
    }

    private fun getBundledImageIfPresent(context: Context, imageName: String): Bitmap? {
        val res = context.resources.getIdentifier(imageName, "drawable", context.packageName)
        return if (res == 0) {
            null
        } else {
            BitmapFactory.decodeResource(context.resources, res)
        }
    }

    private fun isRemoteResource(imageUrl: String): Boolean {
        return imageUrl.startsWith("https://") || imageUrl.startsWith("http://")
    }

    public fun getGifFromUrl(url: String, campaignId: String): File? {
        try {
            val fileName = "${getSha256ForString(url)}.gif"
            if (fileManager.fileExistsInDirectory(campaignId, fileName)) {
                return fileManager.getFileByName(campaignId, fileName)
            }
            val inputStream = URL(url).openStream()
            return fileManager.saveFile(campaignId, fileName, inputStream)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag  getGifFromUrl() : " }
        }
        return null
    }

    public fun downloadAndSaveHtmlAssets(campaignId: String, assets: Map<String, String>): Int {
        sdkInstance.logger.log { "$tag  downloadAndSaveFiles() : downloading files for campaignId: $campaignId" }
        val successCount = intArrayOf(0)
        try {
            val countDownLatch = CountDownLatch(assets.size)
            val executorService = Executors.newFixedThreadPool(Math.max(assets.size, 5))
            for ((key, value) in assets) {
                executorService.submit {
                    if (downloadAndSaveFile(campaignId, key, value)) successCount[0]++
                    countDownLatch.countDown()
                }
            }
            countDownLatch.await()
            executorService.shutdown()
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag  downloadAndSaveHtmlAssets() : " }
        }
        return successCount[0]
    }

    public fun getHtmlAssetsPath(campaignId: String): String {
        return fileManager.getPathForFile(
            campaignId + FILE_PATH_SEPARATOR + HTML_SUBFOLDER,
            ""
        )
    }

    private fun downloadAndSaveFile(
        campaignId: String,
        relativePath: String,
        absoluteUrl: String
    ): Boolean {
        try {
            val fileName =
                relativePath.substring(relativePath.lastIndexOf(FILE_PATH_SEPARATOR) + 1)
            var subDir = relativePath.replace(fileName, "")
            if (subDir.isNotEmpty()) {
                subDir = (
                    campaignId +
                        FILE_PATH_SEPARATOR +
                        HTML_SUBFOLDER +
                        FILE_PATH_SEPARATOR +
                        subDir
                    )
            }
            if (fileManager.fileExistsInDirectory(subDir, fileName)) {
                sdkInstance.logger.log { "$tag  downloadAndSaveFiles() : file already exists. file:$relativePath" }
                return true
            }
            val inputStream = URL(absoluteUrl).openStream()
            val isSuccess = fileManager.saveFile(subDir, fileName, inputStream) != null
            sdkInstance.logger.log { "$tag downloadAndSaveFiles() : isDownloadSuccess: ,$isSuccess  file: $relativePath, fileUrl: $absoluteUrl" }
            inputStream.close()
            return isSuccess
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag downloadAndSaveFiles() : " }
        }
        return false
    }

    public fun deleteHtmlAssetsForCampaignIds(campaignIdList: Set<String>?) {
        if (campaignIdList == null) return
        for (campaignId in campaignIdList) {
            sdkInstance.logger.log(LogLevel.INFO) { "$tag deleteHtmlAssetsForCampaignIds() : campaignId:$campaignId" }
            fileManager.deleteFolder(campaignId + FILE_PATH_SEPARATOR + HTML_SUBFOLDER)
        }
    }

    public fun getVideoFromUrl(url: String, campaignId: String): Uri? {
        sdkInstance.logger.log {
            "$tag getVideoFromUrl(): will try fetch video from url for " +
                "campaignId: $campaignId"
        }
        try {
            return if (isRemoteResource(url)) {
                getVideo(url, campaignId)
            } else {
                null
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag  getVideoFromUrl() : " }
        }
        return null
    }

    /**
     * Get downloaded Video URI
     * @param url - [String] URL of Video
     * @param campaignId - InApp Campaign id
     * @return Instance of [Uri] , if not exists in storage returns null
     */
    @Throws(NoSuchAlgorithmException::class)
    private fun getVideo(url: String, campaignId: String): Uri? {
        sdkInstance.logger.log {
            "$tag getVideo(): will try to fetch video from the url for " +
                "campaignId: $campaignId, url: $url"
        }
        try {
            val fileName = getSha256ForString(url)
            if (fileManager.fileExistsInDirectory(campaignId, fileName)) {
                return Uri.fromFile(fileManager.getFileByName(campaignId, fileName))
            } else {
                val inputStream = URL(url).openStream()
                LifecycleManager.addBackgroundListener {
                    try {
                        inputStream.close()
                    } catch (t: Throwable) {
                        sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getVideo(): onAppBackground() : " }
                    }
                }
                val file = fileManager.saveFile(campaignId, fileName, inputStream) ?: return null
                inputStream.close()
                return Uri.fromFile(file)
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getVideo(): " }
        }
        return null
    }
}