/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.repository

import androidx.annotation.RestrictTo
import androidx.annotation.WorkerThread
import com.moengage.core.LogLevel
import com.moengage.core.internal.exception.NetworkRequestDisabledException
import com.moengage.core.internal.exception.NetworkRequestFailedException
import com.moengage.core.internal.model.DeviceType
import com.moengage.core.internal.model.NetworkResult
import com.moengage.core.internal.model.ResultFailure
import com.moengage.core.internal.model.ResultSuccess
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.currentISOTime
import com.moengage.core.internal.utils.currentSeconds
import com.moengage.inapp.internal.DELIVERY_STAGE_API_FAILURE
import com.moengage.inapp.internal.DELIVERY_STAGE_MANDATORY_PARAM_MISSING
import com.moengage.inapp.internal.ERROR_CODE_GLOBAL_CONTROL_GROUP
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.InAppInstanceProvider
import com.moengage.inapp.internal.STATISTICS_BATCH_SIZE
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.TriggerRequestMeta
import com.moengage.inapp.internal.model.meta.CampaignState
import com.moengage.inapp.internal.model.meta.InAppCampaign
import com.moengage.inapp.internal.model.network.CampaignError
import com.moengage.inapp.internal.model.network.CampaignRequest
import com.moengage.inapp.internal.model.network.InAppMetaRequest
import com.moengage.inapp.internal.model.network.MetaResponse
import com.moengage.inapp.internal.model.network.StatsUploadRequest
import com.moengage.inapp.internal.repository.local.LocalRepository
import com.moengage.inapp.internal.repository.remote.RemoteRepository
import org.json.JSONObject

/**
 * @author Arshiya Khanum
 */

private const val PARAM_ERROR_CODE = "code"
private const val ERROR_CODE_PERSONALISATION_FAILURE = 409
private const val ERROR_CODE_UPDATE_STATE = 410

internal class InAppRepository(
    private val localRepository: LocalRepository,
    private val remoteRepository: RemoteRepository,
    private val sdkInstance: SdkInstance
) : LocalRepository by localRepository,
    RemoteRepository by remoteRepository {

    private val tag = "${InAppConstants.MODULE_TAG}InAppRepository"

    @WorkerThread
    fun fetchInAppCampaignMeta(deviceType: DeviceType, hasPushPermission: Boolean): Boolean {
        sdkInstance.logger.log { "$tag fetchInAppCampaignMeta() : Fetching in-app campaign meta" }
        if (!isModuleEnabled()) throw NetworkRequestDisabledException("Account/SDK disabled.")
        val request = InAppMetaRequest(baseRequest(), deviceType, hasPushPermission)
        when (val response = fetchCampaignMeta(request)) {
            is ResultFailure<*> -> {
                sdkInstance.logger.log { "$tag fetchInAppCampaignMeta() : Meta API Failed." }
                throw NetworkRequestFailedException("Meta API failed.")
            }
            is ResultSuccess<*> -> {
                val meta = response.data as MetaResponse
                sdkInstance.logger.log { "$tag fetchInAppCampaignMeta() : Sync Interval ${meta.syncInterval}" }
                sdkInstance.logger.log { "$tag fetchInAppCampaignMeta() : Global Delay ${meta.globalDelay}" }
                val currentTime = currentSeconds()
                storeLastApiSyncTime(currentTime)
                addOrUpdateInApp(meta.campaigns)
                if (meta.syncInterval > 0) {
                    storeApiSyncInterval(meta.syncInterval)
                }
                if (meta.globalDelay >= 0) {
                    storeGlobalDelay(meta.globalDelay)
                }
            }
        }
        return true
    }

    @WorkerThread
    fun fetchCampaignPayload(
        campaign: InAppCampaign,
        screenName: String,
        appContext: Set<String>,
        deviceType: DeviceType,
        triggerMeta: TriggerRequestMeta?
    ): CampaignPayload? {
        sdkInstance.logger.log { "$tag fetchCampaignPayload() : Fetching in-app campaign payload." }
        try {
            if (!isModuleEnabled()) return null
            val request = CampaignRequest(
                baseRequest(),
                campaign.campaignMeta.campaignId,
                screenName,
                appContext,
                triggerMeta,
                campaign.campaignMeta.campaignContext,
                deviceType,
                campaign.campaignMeta.inAppType
            )
            return when (val response = fetchCampaignPayload(request)) {
                is ResultFailure<*> -> {
                    val result = response.data as CampaignError
                    processFailure(result, request)
                    null
                }
                is ResultSuccess<*> -> {
                    val result = response.data as CampaignPayload
                    result
                }
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag fetchCampaignPayload() : " }
        }
        return null
    }

    private fun processFailure(error: CampaignError, request: CampaignRequest) {
        sdkInstance.logger.log { "$tag processFailure() : Error: $error" }
        when {
            error.hasParsingException && request.campaignContext != null -> {
                InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance)
                    .updateStatForCampaign(
                        request.campaignContext,
                        currentISOTime(),
                        DELIVERY_STAGE_MANDATORY_PARAM_MISSING
                    )
            }

            error.code == ERROR_CODE_UPDATE_STATE -> processError(error.message, request.campaignId)
            error.code != ERROR_CODE_PERSONALISATION_FAILURE && error.code != 200 && request.campaignContext != null -> {
                InAppInstanceProvider.getDeliveryLoggerForInstance(sdkInstance)
                    .updateStatForCampaign(
                        request.campaignContext,
                        currentISOTime(),
                        DELIVERY_STAGE_API_FAILURE
                    )
            }
        }
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY)
    fun onLogout() {
        sdkInstance.logger.log { "$tag onLogout() : " }
        uploadStats()
        clearDataAndUpdateCache()
    }

    fun updateCache() {
        sdkInstance.logger.log { "$tag updateCache() : Updating cache" }
        InAppInstanceProvider.getCacheForInstance(sdkInstance).updateCache(this)
    }

    @WorkerThread
    fun fetchTestCampaignPayload(
        campaignId: String,
        deviceType: DeviceType
    ): NetworkResult? {
        sdkInstance.logger.log { "$tag fetchTestCampaignPayload() : Fetching in-app test campaign payload." }
        try {
            if (!isModuleEnabled()) return null
            return fetchTestCampaign(CampaignRequest(baseRequest(), campaignId, deviceType))
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag fetchTestCampaignPayload() : " }
        }
        return null
    }

    private val syncObj = Any()

    fun uploadStats() {
        try {
            sdkInstance.logger.log { "$tag uploadStats() : " }
            if (!isModuleEnabled() || !sdkInstance.remoteConfig.inAppConfig.isStatsEnabled) return
            synchronized(syncObj) {
                while (true) {
                    var isSuccess = true
                    val statList = getStats(STATISTICS_BATCH_SIZE)
                    if (statList.isNullOrEmpty()) {
                        sdkInstance.logger.log { "$tag uploadStats() : Not pending batches" }
                        return
                    }
                    for (model in statList) {
                        val response = uploadStats(StatsUploadRequest(baseRequest(), model))

                        if (response is ResultFailure<*>) {
                            isSuccess = false
                            break
                        }
                        deleteStatById(model)
                    }
                    if (!isSuccess) break
                }
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag uploadStats() : " }
        }
    }

    fun isModuleEnabled(): Boolean {
        val isModuleEnabled =
            getSdkStatus().isEnabled &&
                sdkInstance.remoteConfig.isAppEnabled &&
                sdkInstance.remoteConfig.moduleStatus.isInAppEnabled &&
                isStorageAndAPICallEnabled()
        sdkInstance.logger.log { "$tag isModuleEnabled() : $isModuleEnabled" }
        return isModuleEnabled
    }

    private fun processError(errorResponse: String, campaignId: String) {
        try {
            sdkInstance.logger.log { "$tag processError() : Campaign id: $campaignId, error response: $errorResponse" }
            if (errorResponse.isBlank()) return
            val errorJson = JSONObject(errorResponse)
            if (ERROR_CODE_GLOBAL_CONTROL_GROUP == errorJson.optString(PARAM_ERROR_CODE, "")) {
                updateCampaignStateForControlGroup(campaignId)
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag processError() : " }
        }
    }

    private fun updateCampaignStateForControlGroup(campaignId: String) {
        sdkInstance.logger.log { "$tag updateCampaignStateForControlGroup() : Updating campaign state for id: $campaignId" }
        val campaign = getCampaignById(campaignId) ?: return
        val updatedState = CampaignState(
            campaign.state.showCount + 1,
            currentSeconds(),
            campaign.state.isClicked
        )
        updateCampaignState(updatedState, campaignId)
        updateCache()
    }

    fun getCampaignsForEvent(eventName: String): List<InAppCampaign> {
        try {
            val mapper = PayloadMapper()
            val triggerCampaigns = mapper.entityToCampaign(localRepository.getTriggerCampaigns())
            if (triggerCampaigns.isEmpty()) return emptyList()
            val filteredTriggerCampaigns = mutableListOf<InAppCampaign>()
            for (triggerCampaign in triggerCampaigns) {
                triggerCampaign.campaignMeta.trigger?.let { trigger ->
                    for (triggerCondition in trigger.primaryConditions) {
                        if (triggerCondition.eventName == eventName) {
                            filteredTriggerCampaigns.add(triggerCampaign)
                            break
                        }
                    }
                }
            }
            return filteredTriggerCampaigns
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getCampaignsForEvent() : " }
        }
        return emptyList()
    }

    fun getPrimaryTriggerEvents(): Set<String> {
        try {
            val triggerCampaigns = PayloadMapper().entityToCampaign(getTriggerCampaigns())
            if (triggerCampaigns.isEmpty()) return emptySet()
            val triggerEvents = mutableSetOf<String>()
            for (triggerCampaign in triggerCampaigns) {
                triggerCampaign.campaignMeta.trigger?.let { trigger ->
                    triggerEvents.addAll(
                        trigger.primaryConditions.map { triggerCondition ->
                            triggerCondition.eventName
                        }
                    )
                }
            }
            sdkInstance.logger.log { "$tag getPrimaryTriggerEvents() : Trigger Events: ${triggerEvents.joinToString()}" }
            return triggerEvents
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getPrimaryTriggerEvents() : " }
        }
        return emptySet()
    }

    /**
     * @since 6.7.0
     */
    fun clearDataAndUpdateCache() {
        sdkInstance.logger.log { "$tag clearDataAndUpdateCache() : " }
        clearData()
        updateCache()
    }

    internal fun syncAndResetData() {
        sdkInstance.logger.log { "$tag syncAndResetData() : " }
        uploadStats()
        clearDataAndUpdateCache()
    }
}