/*
 * Copyright (c) 2014-2023 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.repository

import com.moengage.core.internal.utils.isNullOrBlank
import com.moengage.inapp.internal.model.meta.Trigger
import com.moengage.inapp.internal.model.meta.TriggerCondition
import org.json.JSONArray
import org.json.JSONObject

/**
 * Convert the [Trigger] to [JSONObject]
 *
 * @param trigger instance of [Trigger]
 * @return [JSONObject] - jsonObject for the provided trigger object
 * @since 7.1.0
 */
internal fun triggerToJson(trigger: Trigger?): JSONObject? {
    if (trigger == null) return null
    val triggerJson = JSONObject()

    // Primary Condition Json
    val primaryConditionFilters = JSONArray()
    for (primaryCondition in trigger.primaryConditions) {
        primaryConditionFilters.put(triggerConditionToJson(primaryCondition))
    }
    val primaryConditionFilter = JSONObject()
        .put(FILTERS, primaryConditionFilters)
    val primaryConditionIncludedFilter = JSONObject()
        .put(INCLUDED_FILTERS, primaryConditionFilter)

    // Trigger Json
    triggerJson.put(PRIMARY_CONDITION, primaryConditionIncludedFilter)
    return triggerJson
}

/**
 * Convert the [TriggerCondition] to [JSONObject]
 *
 * @param triggerCondition instance of [TriggerCondition]
 * @return [JSONObject] - jsonObject for the provided triggerCondition object
 * @since 7.1.0
 */
internal fun triggerConditionToJson(triggerCondition: TriggerCondition): JSONObject {
    val triggerJson = JSONObject()
    triggerJson.put(ACTION_NAME, triggerCondition.eventName)
    triggerJson.put(
        ATTRIBUTES,
        if (!triggerCondition.attributes.isNullOrBlank()) {
            triggerCondition.attributes
        } else {
            JSONObject()
        }
    )
    return triggerJson
}