/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.repository

import androidx.annotation.VisibleForTesting
import com.moengage.core.internal.utils.MoEUtils
import com.moengage.core.internal.utils.currentSeconds
import com.moengage.core.internal.utils.jsonArrayToStringSet
import com.moengage.core.internal.utils.secondsFromIsoString
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.exceptions.ParseException
import com.moengage.inapp.internal.model.CampaignEntity
import com.moengage.inapp.internal.model.enums.CampaignSubType
import com.moengage.inapp.internal.model.enums.InAppType
import com.moengage.inapp.internal.model.meta.CampaignMeta
import com.moengage.inapp.internal.model.meta.CampaignState
import com.moengage.inapp.internal.model.meta.DeliveryControl
import com.moengage.inapp.internal.model.meta.DisplayControl
import com.moengage.inapp.internal.model.meta.FrequencyCapping
import com.moengage.inapp.internal.model.meta.InAppCampaign
import com.moengage.inapp.internal.model.meta.Rules
import com.moengage.inapp.internal.model.meta.Trigger
import com.moengage.inapp.internal.model.meta.TriggerCondition
import com.moengage.inapp.internal.screenOrientationFromJson
import com.moengage.inapp.model.CampaignContext
import com.moengage.inapp.model.enums.InAppPosition
import org.json.JSONObject

/**
 * @author Umang Chamaria
 */

private const val CAMPAIGN_ID: String = "campaign_id"
private const val CAMPAIGN_NAME = "campaign_name"

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal const val EXPIRY_TIME = "expiry_time"

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal const val LAST_UPDATED_TIME = "updated_time"

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal const val DISPLAY_CONTROL = "display"
private const val TEMPLATE_TYPE = "template_type"

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal const val DELIVERY_CONTROL = "delivery"

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal const val TRIGGER = "trigger"

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal const val CAMPAIGN_CONTEXT = "campaign_context"
private const val INAPP_TYPE = "inapp_type"
private const val ORIENTATIONS = "orientations"
private const val STATUS = "status"

// Display Control
private const val RULES = "rules"

// Rules
private const val SCREEN_NAME = "screen_name"
private const val CONTEXTS = "contexts"

// Campaign Context
private const val CID = "cid"

// Delivery Control
private const val PRIORITY = "priority"

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal const val FC_META = "fc_meta"
private const val PERSISTENT = "persistent"

// Frequency Meta
private const val IGNORE_GLOBAL_DELAY = "ignore_global_delay"
private const val MAX_COUNT = "count"
private const val MINIMUM_DELAY = "delay"

// Trigger
internal const val PRIMARY_CONDITION = "primary_condition"
internal const val ACTION_NAME = "action_name"
internal const val INCLUDED_FILTERS = "included_filters"
internal const val FILTERS = "filters"
internal const val ATTRIBUTES = "attributes"

// state
private const val SHOW_COUNT = "show_count"
private const val LAST_SHOW_TIME = "last_show_time"
private const val IS_CLICKED = "is_clicked"

private const val CAMPAIGN_SUB_TYPE = "campaign_sub_type"
private const val DELAY = "delay"

// For Non-Intrusive Nudges
private const val POSITION = "position"

internal class PayloadMapper {

    fun jsonToCampaignMeta(metaJson: JSONObject): CampaignMeta {
        return CampaignMeta(
            metaJson.getString(CAMPAIGN_ID),
            metaJson.getString(CAMPAIGN_NAME),
            secondsFromIsoString(metaJson.getString(EXPIRY_TIME)),
            secondsFromIsoString(metaJson.getString(LAST_UPDATED_TIME)),
            jsonToDisplayControl(metaJson.optJSONObject(DISPLAY_CONTROL)),
            metaJson.getString(TEMPLATE_TYPE),
            jsonToDeliveryControl(metaJson.getJSONObject(DELIVERY_CONTROL)),
            jsonToTrigger(metaJson.optJSONObject(TRIGGER)),
            jsonToCampaignContext(metaJson.optJSONObject(CAMPAIGN_CONTEXT)),
            if (metaJson.has(INAPP_TYPE)) {
                InAppType.valueOf(
                    metaJson.getString(INAPP_TYPE).uppercase()
                )
            } else {
                null
            },
            if (metaJson.has(ORIENTATIONS)) {
                screenOrientationFromJson(
                    metaJson.getJSONArray(ORIENTATIONS)
                )
            } else {
                null
            },
            CampaignSubType.valueOf(
                metaJson.optString(CAMPAIGN_SUB_TYPE, CampaignSubType.GENERAL.toString())
                    .uppercase()
            ),
            if (metaJson.has(POSITION)) {
                InAppPosition.valueOf(metaJson.getString(POSITION).trim().uppercase())
            } else {
                null
            }
        )
    }

    fun jsonToDisplayControl(displayJson: JSONObject?): DisplayControl {
        if (displayJson == null) return DisplayControl(Rules(null, emptySet()), -1L)
        return DisplayControl(
            jsonToRules(displayJson.optJSONObject(RULES)),
            displayJson.optLong(DELAY, -1L)
        )
    }

    fun jsonToRules(rulesJson: JSONObject?): Rules {
        if (rulesJson == null) return Rules(null, emptySet())
        return Rules(
            rulesJson.optString(SCREEN_NAME, null),
            jsonArrayToStringSet(rulesJson.optJSONArray(CONTEXTS))
        )
    }

    fun jsonToDeliveryControl(deliveryJson: JSONObject): DeliveryControl {
        return DeliveryControl(
            deliveryJson.getLong(PRIORITY),
            jsonToFrequencyCapping(deliveryJson.getJSONObject(FC_META))
        )
    }

    fun jsonToFrequencyCapping(frequencyJson: JSONObject): FrequencyCapping {
        return FrequencyCapping(
            frequencyJson.getBoolean(IGNORE_GLOBAL_DELAY),
            frequencyJson.getLong(MAX_COUNT),
            frequencyJson.getLong(MINIMUM_DELAY)
        )
    }

    /**
     * Convert the triggerJson to the [Trigger] object instance
     *
     * @param triggerJson json object for [Trigger]
     * @return [Trigger] instance of Trigger or null if required attribute missing
     * @since 7.1.0
     */
    fun jsonToTrigger(triggerJson: JSONObject?): Trigger? {
        if (triggerJson == null || !triggerJson.has(PRIMARY_CONDITION)) return null
        val primaryCondition = triggerJson.getJSONObject(PRIMARY_CONDITION)

        return if (primaryCondition.has(INCLUDED_FILTERS) &&
            primaryCondition.getJSONObject(INCLUDED_FILTERS).has(FILTERS)
        ) {
            val filters = primaryCondition.getJSONObject(INCLUDED_FILTERS).getJSONArray(FILTERS)
            val triggerConditions = mutableListOf<TriggerCondition>()
            for (filterIndex in 0 until filters.length()) {
                jsonToTriggerCondition(filters.getJSONObject(filterIndex))?.let { triggerCondition ->
                    triggerConditions.add(triggerCondition)
                }
            }

            if (triggerConditions.isNotEmpty()) {
                Trigger(triggerConditions)
            } else {
                null
            }
        } else {
            null
        }
    }

    /**
     * Convert the triggerConditionJson to the [TriggerCondition] object instance
     *
     * @param triggerJson json object for [TriggerCondition]
     * @return [TriggerCondition] - instance of TriggerCondition or null if required attribute missing
     * @since 7.1.0
     */
    fun jsonToTriggerCondition(triggerJson: JSONObject): TriggerCondition? {
        return if (triggerJson.has(ACTION_NAME) &&
            !triggerJson.getString(ACTION_NAME).isNullOrBlank()
        ) {
            TriggerCondition(
                triggerJson.getString(ACTION_NAME),
                triggerJson.optJSONObject(ATTRIBUTES)
            )
        } else {
            null
        }
    }

    fun jsonToCampaignContext(contextJson: JSONObject?): CampaignContext? {
        return if (contextJson == null) {
            null
        } else {
            CampaignContext(
                contextJson.getString(CID),
                contextJson,
                MoEUtils.jsonToMap(contextJson)
            )
        }
    }

    fun campaignStateFromJson(stateJson: JSONObject): CampaignState {
        return CampaignState(
            stateJson.optLong(SHOW_COUNT, 0),
            stateJson.optLong(LAST_SHOW_TIME, 0),
            stateJson.optBoolean(IS_CLICKED, false)
        )
    }

    fun campaignStateToJson(state: CampaignState): JSONObject {
        val json = JSONObject()
        json.put(SHOW_COUNT, state.showCount)
            .put(LAST_SHOW_TIME, state.lastShowTime)
            .put(IS_CLICKED, state.isClicked)
        return json
    }

    fun jsonToCampaignEntity(campaignJson: JSONObject): CampaignEntity {
        validateMandatoryCampaignParams(campaignJson)
        return CampaignEntity(
            id = -1L,
            campaignId = campaignJson.getString(CAMPAIGN_ID),
            campaignType = getCampaignType(campaignJson),
            status = campaignJson.getString(STATUS),
            templateType = campaignJson.getString(TEMPLATE_TYPE),
            state = CampaignState(0, 0, false),
            priority = getPriorityForCampaign(campaignJson),
            lastUpdatedTime = secondsFromIsoString(campaignJson.getString(LAST_UPDATED_TIME)),
            deletionTime = deletionTimeForCampaign(campaignJson),
            lastReceivedTime = currentSeconds(),
            metaPayload = campaignJson.toString()
        )
    }

    fun deletionTimeForCampaign(campaignJson: JSONObject): Long {
        return (currentSeconds() + InAppConstants.SIXTY_DAYS_SECONDS).coerceAtLeast(
            secondsFromIsoString(campaignJson.getString(EXPIRY_TIME))
        )
    }

    fun getPriorityForCampaign(campaignJson: JSONObject): Long {
        return campaignJson.getJSONObject(DELIVERY_CONTROL).getLong(PRIORITY)
    }

    fun getCampaignType(campaignJson: JSONObject): String {
        return if (campaignJson.optJSONObject(TRIGGER) != null) {
            InAppConstants
                .IN_APP_CAMPAIGN_TYPE_SMART
        } else {
            InAppConstants.IN_APP_CAMPAIGN_TYPE_GENERAL
        }
    }

    fun entityToCampaign(entities: List<CampaignEntity>): List<InAppCampaign> {
        val campaigns = mutableListOf<InAppCampaign>()
        for (entity in entities) {
            campaigns.add(campaignEntityToCampaign(entity))
        }
        return campaigns
    }

    fun campaignEntityToCampaign(entity: CampaignEntity): InAppCampaign {
        return InAppCampaign(
            campaignType = entity.campaignType,
            status = entity.status,
            deletionTime = entity.deletionTime,
            campaignMeta = jsonToCampaignMeta(JSONObject(entity.metaPayload)),
            campaignState = entity.state
        )
    }

    /**
     * Validates mandatory params for the campaign present in meta call
     *
     * @param campaignJson [JSONObject]
     * @throws ParseException In case the param is not valid.
     */
    @Throws(ParseException::class)
    fun validateMandatoryCampaignParams(campaignJson: JSONObject) {
        if (campaignJson.getString(TEMPLATE_TYPE)
            .equals(InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE) && !campaignJson.has(
                    POSITION
                )
        ) {
            throw ParseException("position is a mandatory param for non-intrusive nudges")
        }
    }
}