/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.repository.local

import android.content.ContentValues
import android.content.Context
import android.database.Cursor
import com.moengage.core.LogLevel
import com.moengage.core.internal.CoreInternalHelper
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.model.SdkStatus
import com.moengage.core.internal.model.database.DataAccessor
import com.moengage.core.internal.model.database.QueryParams
import com.moengage.core.internal.model.database.WhereClause
import com.moengage.core.internal.model.network.BaseRequest
import com.moengage.core.internal.storage.database.contract.BASE_COLUMN_NAME_ID
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_CAMPAIGN_ID
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_CAMPAIGN_STATUS
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_CAMPAIGN_TYPE
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_DELETION_TIME
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_LAST_UPDATED_TIME
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_PRIORITY
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_TEMPLATE_TYPE
import com.moengage.core.internal.storage.database.contract.PROJECTION_INAPP_STATS
import com.moengage.core.internal.storage.database.contract.PROJECTION_INAPP_V3
import com.moengage.core.internal.storage.database.contract.TABLE_NAME_INAPP_STATS
import com.moengage.core.internal.storage.database.contract.TABLE_NAME_INAPP_V3
import com.moengage.core.internal.storage.preference.DEFAULT_LAST_INAPP_SHOW_TIME
import com.moengage.core.internal.storage.preference.KEY_INAPP_LAST_SHOWN_TIME
import com.moengage.core.internal.storage.preference.KEY_PUSH_PERMISSION_REQUEST_COUNT
import com.moengage.core.internal.utils.currentSeconds
import com.moengage.core.internal.utils.getBaseRequest
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.model.CampaignEntity
import com.moengage.inapp.internal.model.InAppGlobalState
import com.moengage.inapp.internal.model.StatModel
import com.moengage.inapp.internal.model.meta.CampaignState
import com.moengage.inapp.internal.model.meta.InAppCampaign
import com.moengage.inapp.internal.repository.InAppFileManager
import kotlin.collections.set

/**
 * @author Arshiya Khanum
 */
private const val DEFAULT_SORT_ORDER =
    "$INAPP_V3_COLUMN_NAME_PRIORITY DESC, $INAPP_V3_COLUMN_NAME_LAST_UPDATED_TIME DESC"

private const val KEY_LAST_SYNC_TIME_IN_APP = "inapp_last_sync_time"
private const val KEY_GLOBAL_DELAY_IN_APP = "in_app_global_delay"
private const val KEY_MINIMUM_SYNC_DELAY_IN_APP = "inapp_api_sync_delay"
private const val KEY_LAST_HTML_ASSETS_DELETE_TIME = "inapp_html_assets_delete_time"

internal class LocalRepositoryImpl(
    private val context: Context,
    private val dataAccessor: DataAccessor,
    private val sdkInstance: SdkInstance
) : LocalRepository {

    private val tag = "${InAppConstants.MODULE_TAG}LocalRepositoryImpl"
    private val marshaller = Marshaller(context, sdkInstance)

    override fun storeApiSyncInterval(syncInterval: Long) {
        dataAccessor.preference.putLong(KEY_MINIMUM_SYNC_DELAY_IN_APP, syncInterval)
    }

    override fun getApiSyncInterval(): Long {
        return dataAccessor.preference.getLong(
            KEY_MINIMUM_SYNC_DELAY_IN_APP,
            InAppConstants.DEFAULT_API_SYNC_INTERVAL
        )
    }

    override fun storeLastApiSyncTime(nextSyncTime: Long) {
        dataAccessor.preference.putLong(KEY_LAST_SYNC_TIME_IN_APP, nextSyncTime)
    }

    override fun getLastSyncTime(): Long {
        return dataAccessor.preference.getLong(
            KEY_LAST_SYNC_TIME_IN_APP,
            InAppConstants.DEFAULT_LAST_SYNC_TIME
        )
    }

    override fun storeGlobalDelay(globalDelay: Long) {
        dataAccessor.preference.putLong(KEY_GLOBAL_DELAY_IN_APP, globalDelay)
    }

    override fun getGlobalState(): InAppGlobalState {
        val lastShowTime =
            dataAccessor.preference.getLong(KEY_INAPP_LAST_SHOWN_TIME, DEFAULT_LAST_INAPP_SHOW_TIME)
        return InAppGlobalState(
            dataAccessor.preference.getLong(
                KEY_GLOBAL_DELAY_IN_APP,
                InAppConstants.DEFAULT_GLOBAL_DELAY_IN_APP
            ),
            lastShowTime,
            currentSeconds()
        )
    }

    override fun addOrUpdateInApp(newCampaigns: List<CampaignEntity>) {
        try {
            val storedCampaigns = getStoredCampaigns().toMutableMap()
            if (storedCampaigns.isEmpty()) {
                val contentValues = ArrayList<ContentValues>(newCampaigns.size)
                for (campaign in newCampaigns) {
                    contentValues.add(marshaller.campaignEntityToContentValues(campaign))
                }
                dataAccessor.dbAdapter.bulkInsert(TABLE_NAME_INAPP_V3, contentValues)
                return
            }
            for (campaign in newCampaigns) {
                val inAppCampaign = storedCampaigns[campaign.campaignId]
                if (inAppCampaign != null) {
                    campaign.id = inAppCampaign.id
                    campaign.state = inAppCampaign.state
                    updateCampaign(campaign)
                    storedCampaigns.remove(inAppCampaign.campaignId)
                } else {
                    saveCampaign(campaign)
                }
            }
            for (campaign in storedCampaigns.values) {
                updateCampaignStatus(
                    campaign.campaignId,
                    InAppConstants.IN_APP_CAMPAIGN_STATUS_IN_ACTIVE
                )
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag addOrUpdateInApp() : " }
        }
    }

    override fun deleteExpiredCampaigns() {
        sdkInstance.logger.log { "$tag deleteExpiredCampaigns() : " }
        InAppFileManager(context, sdkInstance).deleteImagesForCampaignIds(
            campaignsEligibleForDeletion(currentSeconds().toString())
        )
        deleteExpiredCampaignsFromDb(currentSeconds())
    }

    override fun clearData() {
        clearLastSyncTime()
        deleteAllCampaigns()
        deleteImagesForCampaigns()
        deleteAllStats()
    }

    /**
     * Returns all active general campaigns with template type POP_UP and FULL_SCREEN if any,
     * else empty.
     *
     * @return [List] of [InAppCampaign]
     */
    override
    fun getGeneralCampaigns(): List<CampaignEntity> {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter
                .query(
                    TABLE_NAME_INAPP_V3,
                    QueryParams(
                        projection = PROJECTION_INAPP_V3,
                        whereClause = WhereClause(
                            INAPP_V3_COLUMN_NAME_CAMPAIGN_STATUS + " = ? " +
                                " AND " +
                                INAPP_V3_COLUMN_NAME_CAMPAIGN_TYPE + " = ? " +
                                " AND " +
                                INAPP_V3_COLUMN_NAME_TEMPLATE_TYPE + " IN ( ?, ? ) ",
                            arrayOf(
                                InAppConstants.IN_APP_CAMPAIGN_STATUS_ACTIVE,
                                InAppConstants.IN_APP_CAMPAIGN_TYPE_GENERAL,
                                InAppConstants.IN_APP_TEMPLATE_TYPE_POP_UP,
                                InAppConstants.IN_APP_TEMPLATE_TYPE_FULL_SCREEN
                            )
                        ),
                        orderBy = DEFAULT_SORT_ORDER
                    )
                )
            return marshaller.campaignsFromCursor(cursor)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getGeneralCampaigns() : " }
        } finally {
            cursor?.close()
        }
        return emptyList()
    }

    override fun getSelfHandledCampaign(): List<CampaignEntity> {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INAPP_V3,
                QueryParams(
                    projection = PROJECTION_INAPP_V3,
                    whereClause = WhereClause(
                        INAPP_V3_COLUMN_NAME_CAMPAIGN_STATUS + " = ? " +
                            " AND " +
                            INAPP_V3_COLUMN_NAME_CAMPAIGN_TYPE + " = ? " +
                            " AND " +
                            INAPP_V3_COLUMN_NAME_TEMPLATE_TYPE + " = ? ",
                        arrayOf(
                            InAppConstants.IN_APP_CAMPAIGN_STATUS_ACTIVE,
                            InAppConstants.IN_APP_CAMPAIGN_TYPE_GENERAL,
                            InAppConstants.IN_APP_TEMPLATE_TYPE_SELF_HANDLED
                        )
                    ),
                    orderBy = DEFAULT_SORT_ORDER
                )
            )
            return marshaller.campaignsFromCursor(cursor)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag selfHandledCampaigns() : " }
        } finally {
            cursor?.close()
        }
        return emptyList()
    }

    override fun baseRequest(): BaseRequest {
        return getBaseRequest(context, sdkInstance)
    }

    override fun updateCampaignState(state: CampaignState, campaignId: String): Int {
        var updateCount = -1
        try {
            updateCount = dataAccessor.dbAdapter.update(
                TABLE_NAME_INAPP_V3,
                marshaller.campaignStateToContentValues(state),
                WhereClause(
                    "$INAPP_V3_COLUMN_NAME_CAMPAIGN_ID = ? ",
                    arrayOf(campaignId)
                )
            )
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag updateStateForCampaign() : " }
        }
        return updateCount
    }

    override fun updateLastShowTime(time: Long) {
        dataAccessor.preference.putLong(KEY_INAPP_LAST_SHOWN_TIME, time)
    }

    /**
     * Fetches all active campaigns irrespective of type and alignment.
     *
     * @return [List] of [InAppCampaign]
     */
    override fun getAllActiveCampaigns(): List<CampaignEntity> {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INAPP_V3,
                QueryParams(
                    projection = PROJECTION_INAPP_V3,
                    whereClause = WhereClause(
                        "$INAPP_V3_COLUMN_NAME_CAMPAIGN_STATUS = ? ",
                        arrayOf(InAppConstants.IN_APP_CAMPAIGN_STATUS_ACTIVE)
                    ),
                    orderBy = DEFAULT_SORT_ORDER
                )
            )
            return marshaller.campaignsFromCursor(cursor)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getAllActiveCampaigns() : " }
        } finally {
            cursor?.close()
        }
        return emptyList()
    }

    /**
     * Gets campaign for the given id else null.
     *
     * @param campaignId id for which campaign should be returned.
     * @return instance of [InAppCampaign]
     */
    override fun getCampaignById(campaignId: String): CampaignEntity? {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INAPP_V3,
                QueryParams(
                    PROJECTION_INAPP_V3,
                    WhereClause(
                        "$INAPP_V3_COLUMN_NAME_CAMPAIGN_ID = ? ",
                        arrayOf(campaignId)
                    )
                )
            )
            if (cursor != null && cursor.moveToFirst()) {
                return marshaller.entityFromCursor(cursor)
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getCampaignById() : " }
        } finally {
            cursor?.close()
        }
        return null
    }

    override fun getStats(batchSize: Int): List<StatModel> {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INAPP_STATS,
                QueryParams(
                    projection = PROJECTION_INAPP_STATS,
                    whereClause = null,
                    limit = batchSize
                )
            )
            if (cursor == null || !cursor.moveToFirst() || cursor.count == 0) return emptyList()
            val statsList = ArrayList<StatModel>(cursor.count)
            do {
                try {
                    val statModel = marshaller.statFromCursor(cursor)
                    statsList.add(statModel)
                } catch (t: Throwable) {
                    sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getStats() : " }
                }
            } while (cursor.moveToNext())
            return statsList
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getStats() : " }
        } finally {
            cursor?.close()
        }
        return emptyList()
    }

    override fun deleteStatById(stat: StatModel): Int {
        try {
            return dataAccessor.dbAdapter.delete(
                TABLE_NAME_INAPP_STATS,
                WhereClause(
                    "$BASE_COLUMN_NAME_ID = ? ",
                    arrayOf(stat._id.toString())
                )
            )
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag deleteStatById() : " }
        }
        return -1
    }

    override fun getSdkStatus(): SdkStatus {
        return CoreInternalHelper.getSdkStatus(context, sdkInstance)
    }

    fun saveCampaign(entity: CampaignEntity): Long {
        return dataAccessor.dbAdapter.insert(
            TABLE_NAME_INAPP_V3,
            marshaller.campaignEntityToContentValues(entity)
        )
    }

    private fun updateCampaign(entity: CampaignEntity): Int {
        return dataAccessor.dbAdapter.update(
            TABLE_NAME_INAPP_V3,
            marshaller.campaignEntityToContentValues(entity),
            WhereClause(
                "$BASE_COLUMN_NAME_ID = ?",
                arrayOf(entity.id.toString())
            )
        )
    }

    /**
     * Returns of all the stored campaigns.
     *
     * @return all stored campaigns.
     */
    fun getStoredCampaigns(): Map<String, CampaignEntity> {
        var cursor: Cursor? = null
        try {
            val campaignsMap: MutableMap<String, CampaignEntity> = HashMap()
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INAPP_V3,
                QueryParams(
                    PROJECTION_INAPP_V3,
                    null
                )
            )
            if (cursor == null || !cursor.moveToFirst()) return emptyMap()
            do {
                try {
                    val campaign = marshaller.entityFromCursor(cursor)
                    campaignsMap[campaign.campaignId] = campaign
                } catch (t: Throwable) {
                    sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getStoredCampaigns() : " }
                }
            } while (cursor.moveToNext())
            return campaignsMap
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getStoredCampaigns() : " }
        } finally {
            cursor?.close()
        }
        return emptyMap()
    }

    /**
     * All campaign ids eligible for deletion.
     *
     * @return [Set] of [String]
     */
    fun campaignsEligibleForDeletion(timeInSecs: String): Set<String> {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INAPP_V3,
                QueryParams(
                    arrayOf(INAPP_V3_COLUMN_NAME_CAMPAIGN_ID),
                    WhereClause(
                        "$INAPP_V3_COLUMN_NAME_DELETION_TIME < ? ",
                        arrayOf(timeInSecs)
                    )
                )
            )
            return marshaller.campaignIdsFromCursor(cursor)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag campaignsEligibleForDeletion() : " }
        } finally {
            cursor?.close()
        }
        return emptySet()
    }

    /**
     * Get a set of all campaign ids.
     *
     * @return [Set] of [String]
     */
    fun getAllCampaignIds(): Set<String> {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INAPP_V3,
                QueryParams(
                    arrayOf(INAPP_V3_COLUMN_NAME_CAMPAIGN_ID),
                    null
                )
            )
            return marshaller.campaignIdsFromCursor(cursor)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getAllCampaignIds() : " }
        } finally {
            cursor?.close()
        }
        return emptySet()
    }

    fun clearLastSyncTime() {
        dataAccessor.preference.removeKey(KEY_LAST_SYNC_TIME_IN_APP)
    }

    private fun deleteImagesForCampaigns() {
        InAppFileManager(context, sdkInstance).deleteImagesForCampaignIds(
            getAllCampaignIds()
        )
    }

    fun deleteAllStats(): Int {
        return dataAccessor.dbAdapter.delete(TABLE_NAME_INAPP_STATS, null)
    }

    /**
     * Returns all active campaigns of TYPE [InAppConstants.IN_APP_CAMPAIGN_TYPE_SMART]  if
     * any, else null.
     *
     * @return [List] of [InAppCampaign]
     */
    override fun getTriggerCampaigns(): List<CampaignEntity> {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INAPP_V3,
                QueryParams(
                    projection = PROJECTION_INAPP_V3,
                    whereClause = WhereClause(
                        "$INAPP_V3_COLUMN_NAME_CAMPAIGN_STATUS = ?  AND $INAPP_V3_COLUMN_NAME_CAMPAIGN_TYPE = ? ",
                        arrayOf(
                            InAppConstants.IN_APP_CAMPAIGN_STATUS_ACTIVE,
                            InAppConstants.IN_APP_CAMPAIGN_TYPE_SMART
                        )
                    ),
                    orderBy = DEFAULT_SORT_ORDER
                )
            )
            return marshaller.campaignsFromCursor(cursor)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getTriggerCampaigns() : " }
        } finally {
            cursor?.close()
        }
        return emptyList()
    }

    fun deleteExpiredCampaignsFromDb(timeInSeconds: Long): Int {
        try {
            return dataAccessor.dbAdapter.delete(
                TABLE_NAME_INAPP_V3,
                WhereClause(
                    "$INAPP_V3_COLUMN_NAME_DELETION_TIME < ? ",
                    arrayOf(timeInSeconds.toString())
                )
            )
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag deleteExpiredCampaignsFromDb() :" }
        }
        return -1
    }

    fun deleteAllCampaigns(): Int {
        return dataAccessor.dbAdapter.delete(TABLE_NAME_INAPP_V3, null)
    }

    override fun writeStats(statModel: StatModel): Long {
        var rowId: Long = -1
        try {
            sdkInstance.logger.log { "$tag writeStats(): will write in-app stats to storage." }
            rowId = dataAccessor.dbAdapter.insert(
                TABLE_NAME_INAPP_STATS,
                marshaller.statToContentValues(statModel)
            )
            sdkInstance.logger.log { "$tag writeStats(): saved : $rowId , stats: $statModel" }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag writeStats() : " }
        }

        return rowId
    }

    /**
     * Fetches all stored campaigns irrespective of type and alignment.
     *
     * @return [List] of [InAppCampaign]
     */
    override fun getAllCampaigns(): List<CampaignEntity> {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INAPP_V3,
                QueryParams(
                    projection = PROJECTION_INAPP_V3,
                    whereClause = null,
                    orderBy = DEFAULT_SORT_ORDER
                )
            )
            return marshaller.campaignsFromCursor(cursor)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag getAllCampaigns() : " }
        } finally {
            cursor?.close()
        }
        return emptyList()
    }

    override fun storeHtmlAssetsDeleteTime(deleteTime: Long) {
        dataAccessor.preference.putLong(
            KEY_LAST_HTML_ASSETS_DELETE_TIME,
            deleteTime
        )
    }

    override fun getLastHtmlAssetsDeleteTime(): Long {
        return dataAccessor.preference.getLong(
            KEY_LAST_HTML_ASSETS_DELETE_TIME,
            0
        )
    }

    private fun updateCampaignStatus(campaignId: String, status: String): Int {
        var updateCount = -1
        try {
            updateCount = dataAccessor.dbAdapter.update(
                TABLE_NAME_INAPP_V3,
                marshaller.campaignStatusToContentValues(status),
                WhereClause(
                    "$INAPP_V3_COLUMN_NAME_CAMPAIGN_ID = ? ",
                    arrayOf(campaignId)
                )
            )
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag updateStateForCampaign() : " }
        }
        return updateCount
    }

    override fun getPushPermissionRequestCount(): Int {
        sdkInstance.logger.log { "$tag getPushPermissionRequestCount() : " }
        return dataAccessor.preference.getInt(KEY_PUSH_PERMISSION_REQUEST_COUNT, 0)
    }

    override fun isStorageAndAPICallEnabled(): Boolean {
        return CoreInternalHelper.isStorageAndAPICallEnabled(context, sdkInstance)
    }

    /**
     * Returns all active campaigns for [InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE]
     * template type.
     *
     * @return [List] of [CampaignEntity]
     */
    override fun getNonIntrusiveNudgeCampaigns(): List<CampaignEntity> {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INAPP_V3,
                QueryParams(
                    projection = PROJECTION_INAPP_V3,
                    whereClause = WhereClause(
                        INAPP_V3_COLUMN_NAME_CAMPAIGN_STATUS + " = ? " +
                            " AND " +
                            INAPP_V3_COLUMN_NAME_CAMPAIGN_TYPE + " = ? " +
                            " AND " +
                            INAPP_V3_COLUMN_NAME_TEMPLATE_TYPE + " = ? ",
                        arrayOf(
                            InAppConstants.IN_APP_CAMPAIGN_STATUS_ACTIVE,
                            InAppConstants.IN_APP_CAMPAIGN_TYPE_GENERAL,
                            InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE
                        )
                    ),
                    orderBy = DEFAULT_SORT_ORDER
                )
            )
            return marshaller.campaignsFromCursor(cursor)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag selfHandledCampaigns() : " }
        } finally {
            cursor?.close()
        }
        return emptyList()
    }
}