/*
 * Copyright (c) 2014-2023 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.repository.local

import android.content.ContentValues
import android.content.Context
import android.database.Cursor
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.storage.database.contract.BASE_COLUMN_INDEX_ID
import com.moengage.core.internal.storage.database.contract.BASE_COLUMN_NAME_ID
import com.moengage.core.internal.storage.database.contract.INAPP_STATS_COLUMN_INDEX_PAYLOAD
import com.moengage.core.internal.storage.database.contract.INAPP_STATS_COLUMN_INDEX_REQUEST_ID
import com.moengage.core.internal.storage.database.contract.INAPP_STATS_COLUMN_INDEX_TIMESTAMP
import com.moengage.core.internal.storage.database.contract.INAPP_STATS_COLUMN_NAME_PAYLOAD
import com.moengage.core.internal.storage.database.contract.INAPP_STATS_COLUMN_NAME_REQUEST_ID
import com.moengage.core.internal.storage.database.contract.INAPP_STATS_COLUMN_NAME_TIMESTAMP
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_INDEX_CAMPAIGN_ID
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_INDEX_CAMPAIGN_META
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_INDEX_CAMPAIGN_STATE
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_INDEX_CAMPAIGN_STATUS
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_INDEX_CAMPAIGN_TYPE
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_INDEX_DELETION_TIME
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_INDEX_LAST_RECEIVED_TIME
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_INDEX_LAST_UPDATED_TIME
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_INDEX_PRIORITY
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_INDEX_TEMPLATE_TYPE
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_CAMPAIGN_ID
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_CAMPAIGN_META
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_CAMPAIGN_STATE
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_CAMPAIGN_STATUS
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_CAMPAIGN_TYPE
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_DELETION_TIME
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_LAST_RECEIVED_TIME
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_LAST_UPDATED_TIME
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_PRIORITY
import com.moengage.core.internal.storage.database.contract.INAPP_V3_COLUMN_NAME_TEMPLATE_TYPE
import com.moengage.core.internal.storage.decryptValueIfRequired
import com.moengage.core.internal.storage.encryptValueIfRequired
import com.moengage.inapp.internal.model.CampaignEntity
import com.moengage.inapp.internal.model.StatModel
import com.moengage.inapp.internal.model.meta.CampaignState
import com.moengage.inapp.internal.repository.PayloadMapper
import org.json.JSONException
import org.json.JSONObject

/**
 * @author Umang Chamaria
 */
internal class Marshaller(private val context: Context, private val sdkInstance: SdkInstance) {

    fun campaignEntityToContentValues(entity: CampaignEntity): ContentValues {
        val contentValue = ContentValues()
        if (entity.id != -1L) {
            contentValue.put(BASE_COLUMN_NAME_ID, entity.id)
        }
        contentValue.put(INAPP_V3_COLUMN_NAME_CAMPAIGN_ID, entity.campaignId)
        contentValue.put(INAPP_V3_COLUMN_NAME_CAMPAIGN_TYPE, entity.campaignType)
        contentValue.put(INAPP_V3_COLUMN_NAME_CAMPAIGN_STATUS, entity.status)
        contentValue.put(
            INAPP_V3_COLUMN_NAME_CAMPAIGN_STATE,
            PayloadMapper().campaignStateToJson(entity.state).toString()
        )
        contentValue.put(INAPP_V3_COLUMN_NAME_PRIORITY, entity.priority)
        contentValue.put(INAPP_V3_COLUMN_NAME_LAST_UPDATED_TIME, entity.lastUpdatedTime)
        contentValue.put(INAPP_V3_COLUMN_NAME_TEMPLATE_TYPE, entity.templateType)
        contentValue.put(INAPP_V3_COLUMN_NAME_DELETION_TIME, entity.deletionTime)
        contentValue.put(INAPP_V3_COLUMN_NAME_LAST_RECEIVED_TIME, entity.lastReceivedTime)
        contentValue.put(
            INAPP_V3_COLUMN_NAME_CAMPAIGN_META,
            encryptValueIfRequired(context, sdkInstance, entity.metaPayload)
        )
        return contentValue
    }

    @Throws(JSONException::class)
    fun entityFromCursor(cursor: Cursor): CampaignEntity {
        return CampaignEntity(
            id = cursor.getLong(BASE_COLUMN_INDEX_ID),
            campaignId = cursor.getString(INAPP_V3_COLUMN_INDEX_CAMPAIGN_ID),
            campaignType = cursor.getString(INAPP_V3_COLUMN_INDEX_CAMPAIGN_TYPE),
            status = cursor.getString(INAPP_V3_COLUMN_INDEX_CAMPAIGN_STATUS),
            // empty string is required to handle migration from v2 SDK
            templateType = cursor.getString(INAPP_V3_COLUMN_INDEX_TEMPLATE_TYPE) ?: "",
            state = PayloadMapper().campaignStateFromJson(
                JSONObject(
                    cursor.getString(INAPP_V3_COLUMN_INDEX_CAMPAIGN_STATE)
                )
            ),
            priority = cursor.getLong(INAPP_V3_COLUMN_INDEX_PRIORITY),
            lastUpdatedTime = cursor.getLong(INAPP_V3_COLUMN_INDEX_LAST_UPDATED_TIME),
            deletionTime = cursor.getLong(INAPP_V3_COLUMN_INDEX_DELETION_TIME),
            lastReceivedTime = cursor.getLong(INAPP_V3_COLUMN_INDEX_LAST_RECEIVED_TIME),
            metaPayload = decryptValueIfRequired(
                context,
                sdkInstance,
                cursor.getString(
                    INAPP_V3_COLUMN_INDEX_CAMPAIGN_META
                )
            )
        )
    }

    fun campaignIdsFromCursor(cursor: Cursor?): Set<String> {
        val campaignIds = mutableSetOf<String>()
        if (cursor == null || !cursor.moveToFirst()) return campaignIds
        do {
            campaignIds.add(cursor.getString(0))
        } while (cursor.moveToNext())
        return campaignIds
    }

    fun campaignStateToContentValues(state: CampaignState): ContentValues {
        val contentValue = ContentValues()
        contentValue.put(
            INAPP_V3_COLUMN_NAME_CAMPAIGN_STATE,
            PayloadMapper().campaignStateToJson(state).toString()
        )
        return contentValue
    }

    fun campaignsFromCursor(cursor: Cursor?): List<CampaignEntity> {
        if (cursor == null || !cursor.moveToFirst()) return emptyList()
        val campaigns = mutableListOf<CampaignEntity>()
        do {
            campaigns.add(entityFromCursor(cursor))
        } while (cursor.moveToNext())
        return campaigns
    }

    @Throws(JSONException::class)
    fun statFromCursor(cursor: Cursor): StatModel {
        return StatModel(
            cursor.getLong(BASE_COLUMN_INDEX_ID),
            cursor.getLong(INAPP_STATS_COLUMN_INDEX_TIMESTAMP),
            cursor.getString(INAPP_STATS_COLUMN_INDEX_REQUEST_ID),
            JSONObject(
                decryptValueIfRequired(
                    context,
                    sdkInstance,
                    cursor.getString(INAPP_STATS_COLUMN_INDEX_PAYLOAD)
                )
            )
        )
    }

    fun statToContentValues(stat: StatModel): ContentValues {
        val values = ContentValues()
        if (stat._id != -1L) {
            values.put(BASE_COLUMN_NAME_ID, stat._id)
        }
        values.put(INAPP_STATS_COLUMN_NAME_TIMESTAMP, stat.timestamp)
        values.put(INAPP_STATS_COLUMN_NAME_REQUEST_ID, stat.requestId)
        values.put(
            INAPP_STATS_COLUMN_NAME_PAYLOAD,
            encryptValueIfRequired(context, sdkInstance, stat.statsJson.toString())
        )
        return values
    }

    fun campaignStatusToContentValues(status: String): ContentValues {
        val contentValue = ContentValues()
        contentValue.put(INAPP_V3_COLUMN_NAME_CAMPAIGN_STATUS, status)
        return contentValue
    }
}