/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.inapp.internal.repository.remote

import com.moengage.core.LogLevel
import com.moengage.core.internal.REQUEST_ATTR_QUERY_PARAMS
import com.moengage.core.internal.authorization.AuthorizationHandler
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.rest.NetworkResponse
import com.moengage.core.internal.rest.RequestType
import com.moengage.core.internal.rest.ResponseFailure
import com.moengage.core.internal.rest.RestClient
import com.moengage.core.internal.utils.DEFAULT_NETWORK_FAILURE_ERROR_CODE
import com.moengage.core.internal.utils.DEFAULT_NETWORK_FAILURE_MESSAGE
import com.moengage.core.internal.utils.JsonBuilder
import com.moengage.core.internal.utils.getBaseRequestBuilder
import com.moengage.core.internal.utils.getBaseUriBuilder
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.model.network.CampaignRequest
import com.moengage.inapp.internal.model.network.InAppMetaRequest
import com.moengage.inapp.internal.model.network.StatsUploadRequest
import org.json.JSONArray
import org.json.JSONObject

/**
 * @author Umang Chamaria
 */

private const val BASE_PATH = "v3/campaigns/inapp/"
private const val PATH_PARAM_LIVE = "live"
private const val PATH_PARAM_TEST = "test"
private const val PATH_PARAM_STATS = "live/stats"
private const val QUERY_PARAM_SDK_VERSION = "sdk_ver"
private const val QUERY_PARAM_OS = "os"
private const val QUERY_PARAM_UNIQUE_ID = "unique_id"
private const val HEADER_KEY = "MOE-INAPP-BATCH-ID"
private const val TRIGGER_EVENT = "event"
private const val EVENT_NAME = "name"
private const val TIME = "time"
private const val ATTRIBUTES = "attributes"
private const val SCREEN_NAME = "screen_name"
private const val CONTEXTS = "contexts"
private const val STATS = "stats"
private const val REQUEST_ATTR_CAMPAIGN_CONTEXT = "campaign_context"
private const val QUERY_PARAM_DEVICE_TYPE = "device_type"
private const val QUERY_PARAM_INAPP_VERSION = "inapp_ver"
private const val QUERY_PARAM_PUSH_OPT_IN_STATUS = "push_opt_in_status"

internal class ApiManager(
    private val sdkInstance: SdkInstance,
    private val authorizationHandler: AuthorizationHandler
) {

    private val tag = InAppConstants.MODULE_TAG + "ApiManager"

    fun fetchCampaignMeta(requestMeta: InAppMetaRequest): NetworkResponse {
        try {
            val uriBuilder = getBaseUriBuilder(sdkInstance)
                .appendEncodedPath(BASE_PATH)
                .appendEncodedPath(PATH_PARAM_LIVE)
                .appendQueryParameter(QUERY_PARAM_UNIQUE_ID, requestMeta.uniqueId)
                .appendQueryParameter(QUERY_PARAM_SDK_VERSION, requestMeta.sdkVersion.toString())
                .appendQueryParameter(QUERY_PARAM_OS, requestMeta.platform)
                .appendQueryParameter(QUERY_PARAM_DEVICE_TYPE, requestMeta.deviceType.toString())
                .appendQueryParameter(QUERY_PARAM_INAPP_VERSION, requestMeta.inAppVersion)
                .appendQueryParameter(
                    QUERY_PARAM_PUSH_OPT_IN_STATUS,
                    requestMeta.pushOptInStatus.toString()
                )
            val requestBody = JSONObject()
            requestBody.put(
                REQUEST_ATTR_QUERY_PARAMS,
                requestMeta.defaultParams.build()
            )

            val request = getBaseRequestBuilder(
                uriBuilder.build(),
                RequestType.POST,
                sdkInstance,
                authorizationHandler,
                requestMeta.networkDataEncryptionKey
            ).addBody(requestBody)
                .build()
            val restClient = RestClient(request, sdkInstance)
            return restClient.executeRequest()
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag fetchCampaignMeta() : " }
        }
        return ResponseFailure(DEFAULT_NETWORK_FAILURE_ERROR_CODE, DEFAULT_NETWORK_FAILURE_MESSAGE)
    }

    fun fetchCampaignPayload(campaignRequest: CampaignRequest): NetworkResponse {
        try {
            val uriBuilder = getBaseUriBuilder(sdkInstance)
                .appendEncodedPath(BASE_PATH)
                .appendEncodedPath(PATH_PARAM_LIVE)
                .appendEncodedPath(campaignRequest.campaignId)
                .appendQueryParameter(QUERY_PARAM_UNIQUE_ID, campaignRequest.uniqueId)
                .appendQueryParameter(
                    QUERY_PARAM_SDK_VERSION,
                    campaignRequest.sdkVersion.toString()
                )
                .appendQueryParameter(QUERY_PARAM_OS, campaignRequest.platform)
                .appendQueryParameter(
                    QUERY_PARAM_DEVICE_TYPE,
                    campaignRequest.deviceType.toString()
                )
                .appendQueryParameter(QUERY_PARAM_INAPP_VERSION, campaignRequest.inAppVersion)
            val requestBodyBuilder = JsonBuilder()
            if (campaignRequest.triggerMeta != null) {
                val eventBuilder = JsonBuilder()
                eventBuilder.putString(EVENT_NAME, campaignRequest.triggerMeta.eventName)
                    .putString(TIME, campaignRequest.triggerMeta.timeStamp)
                    .putJsonObject(ATTRIBUTES, campaignRequest.triggerMeta.attributes)
                requestBodyBuilder.putJsonObject(TRIGGER_EVENT, eventBuilder.build())
            }
            requestBodyBuilder.putJsonObject(
                REQUEST_ATTR_QUERY_PARAMS,
                campaignRequest.defaultParams.build()
            )
            if (!campaignRequest.screenName.isNullOrBlank()) {
                requestBodyBuilder.putString(SCREEN_NAME, campaignRequest.screenName)
            }
            if (!campaignRequest.contextList.isNullOrEmpty()) {
                val contextArray = JSONArray()
                for (context in campaignRequest.contextList) {
                    contextArray.put(context)
                }
                requestBodyBuilder.putJsonArray(CONTEXTS, contextArray)
            }
            requestBodyBuilder.putJsonObject(
                REQUEST_ATTR_CAMPAIGN_CONTEXT,
                campaignRequest.campaignContext.payload
            )
            val request = getBaseRequestBuilder(
                uriBuilder.build(),
                RequestType.POST,
                sdkInstance,
                authorizationHandler,
                campaignRequest.networkDataEncryptionKey
            ).addBody(requestBodyBuilder.build()).build()
            val restClient =
                RestClient(request, sdkInstance)
            return restClient.executeRequest()
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag fetchCampaignPayload() : " }
        }
        return ResponseFailure(DEFAULT_NETWORK_FAILURE_ERROR_CODE, DEFAULT_NETWORK_FAILURE_MESSAGE)
    }

    fun fetchTestCampaign(campaignRequest: CampaignRequest): NetworkResponse {
        try {
            val uriBuilder = getBaseUriBuilder(sdkInstance)
                .appendEncodedPath(BASE_PATH)
                .appendEncodedPath(PATH_PARAM_TEST)
                .appendEncodedPath(campaignRequest.campaignId)
                .appendQueryParameter(
                    QUERY_PARAM_SDK_VERSION,
                    campaignRequest.sdkVersion.toString()
                )
                .appendQueryParameter(QUERY_PARAM_OS, campaignRequest.platform)
                .appendQueryParameter(QUERY_PARAM_UNIQUE_ID, campaignRequest.uniqueId)
                .appendQueryParameter(
                    QUERY_PARAM_DEVICE_TYPE,
                    campaignRequest.deviceType.toString()
                )
                .appendQueryParameter(QUERY_PARAM_INAPP_VERSION, campaignRequest.inAppVersion)
            val requestBuilder = getBaseRequestBuilder(
                uriBuilder.build(),
                RequestType.GET,
                sdkInstance,
                authorizationHandler,
                campaignRequest.networkDataEncryptionKey
            )
            val restClient =
                RestClient(requestBuilder.build(), sdkInstance)
            return restClient.executeRequest()
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag fetchTestCampaign() : " }
        }
        return ResponseFailure(DEFAULT_NETWORK_FAILURE_ERROR_CODE, DEFAULT_NETWORK_FAILURE_MESSAGE)
    }

    fun uploadStats(request: StatsUploadRequest): NetworkResponse {
        try {
            sdkInstance.logger.log { "$tag uploadStats() : ${request.stat.statsJson}" }
            val uriBuilder = getBaseUriBuilder(sdkInstance)
                .appendEncodedPath(BASE_PATH)
                .appendEncodedPath(PATH_PARAM_STATS)
                .appendQueryParameter(QUERY_PARAM_SDK_VERSION, request.sdkVersion.toString())
                .appendQueryParameter(QUERY_PARAM_OS, request.platform)
                .appendQueryParameter(QUERY_PARAM_UNIQUE_ID, request.uniqueId)
                .appendQueryParameter(QUERY_PARAM_INAPP_VERSION, request.inAppVersion)

            val requestBody = JSONObject()
            requestBody.put(STATS, request.stat.statsJson)
            requestBody.put(REQUEST_ATTR_QUERY_PARAMS, request.defaultParams.build())

            val requestBuilder = getBaseRequestBuilder(
                uriBuilder.build(),
                RequestType.POST,
                sdkInstance,
                authorizationHandler,
                request.networkDataEncryptionKey,
                true
            ).addBody(requestBody)
                .addHeader(HEADER_KEY, request.stat.requestId)
            val restClient =
                RestClient(requestBuilder.build(), sdkInstance)
            return restClient.executeRequest()
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag uploadStats() : " }
        }
        return ResponseFailure(DEFAULT_NETWORK_FAILURE_ERROR_CODE, DEFAULT_NETWORK_FAILURE_MESSAGE)
    }
}