/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.repository.remote

import androidx.annotation.VisibleForTesting
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.NetworkResult
import com.moengage.core.internal.model.ResultFailure
import com.moengage.core.internal.model.ResultSuccess
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.rest.NetworkResponse
import com.moengage.core.internal.rest.ResponseFailure
import com.moengage.core.internal.rest.ResponseSuccess
import com.moengage.core.internal.utils.DEFAULT_NETWORK_FAILURE_ERROR_CODE
import com.moengage.core.internal.utils.logJsonArray
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.model.CampaignEntity
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.HtmlCampaignPayload
import com.moengage.inapp.internal.model.NativeCampaignPayload
import com.moengage.inapp.internal.model.enums.InAppType
import com.moengage.inapp.internal.model.network.CampaignError
import com.moengage.inapp.internal.model.network.MetaResponse
import com.moengage.inapp.internal.repository.PayloadMapper
import org.json.JSONObject

/**
 * @author Umang Chamaria
 * Date: 2021/07/22
 */

/**
 * Response attribute which denotes the delay to be set for two consecutive inapp messages
 */
private const val GLOBAL_MINIMUM_DELAY = "min_delay_btw_inapps"

/**
 * Response attribute which denotes the delay before which /inapps/live API calls shouldn't be
 * made.
 */
private const val SYNC_INTERVAL = "sync_interval"

/**
 * Response attribute which contains campaign array
 */
private const val CAMPAIGNS = "campaigns"

private const val CAMPAIGN_TYPE = "campaign_type"

private const val ERROR_MESSAGE_NO_INTERNET_CONNECTION = "No Internet Connection.\n Please " +
    "connect to internet and try again."
private const val ERROR_MESSAGE_COULD_NOT_REACH_MOENGAGE_SERVER =
    "Could not reach MoEngage Server.\n Please try again or contact MoEngage Support."

/**
 * Response attribute for error description
 */
private const val ERROR_DESCRIPTION = "description"

internal class Parser(private val sdkInstance: SdkInstance) {

    private val tag = InAppConstants.MODULE_TAG + "Parser"

    fun campaignFromResponse(response: NetworkResponse): NetworkResult {
        return when (response) {
            is ResponseFailure -> ResultFailure(
                CampaignError(
                    response.errorCode,
                    response.errorMessage,
                    false
                )
            )

            is ResponseSuccess -> {
                try {
                    val responseBody = response.data
                    val responseJson = JSONObject(responseBody)
                    val campaignPayload: CampaignPayload =
                        when (InAppType.valueOf(responseJson.getString(ResponseParser.INAPP_TYPE))) {
                            InAppType.HTML -> {
                                htmlCampaignFromJson(responseJson)
                            }

                            InAppType.NATIVE -> {
                                nativeCampaignFromJson(responseJson)
                            }
                        }
                    ResultSuccess(campaignPayload)
                } catch (t: Throwable) {
                    sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag campaignFromResponse() : " }
                    ResultFailure(CampaignError(200, response.data, true))
                }
            }
        }
    }

    private fun htmlCampaignFromJson(responseJson: JSONObject): HtmlCampaignPayload {
        return ResponseParser().htmlCampaignFromJson(responseJson)
    }

    private fun nativeCampaignFromJson(responseJson: JSONObject): NativeCampaignPayload {
        val responseParser = ResponseParser()
        return if (InAppConstants.IN_APP_TEMPLATE_TYPE_SELF_HANDLED ==
            responseJson.getString(ResponseParser.TEMPLATE_TYPE)
        ) {
            responseParser.selfHandledCampaignFromJson(responseJson)
        } else {
            responseParser.campaignPayloadFromResponse(responseJson)
        }
    }

    fun parseStatsUploadResponse(response: NetworkResponse): NetworkResult {
        return when (response) {
            is ResponseSuccess -> ResultSuccess(true)
            is ResponseFailure -> ResultFailure<Any>()
        }
    }

    fun parseCampaignMeta(response: NetworkResponse): NetworkResult {
        return when (response) {
            is ResponseFailure -> ResultFailure<Any>()
            is ResponseSuccess -> ResultSuccess(metaResponseFromJson(JSONObject(response.data)))
        }
    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal fun metaResponseFromJson(responseJson: JSONObject): MetaResponse {
        return MetaResponse(
            campaigns = campaignsFromResponse(responseJson),
            syncInterval = responseJson.optLong(SYNC_INTERVAL, -1),
            globalDelay = responseJson.getLong(GLOBAL_MINIMUM_DELAY)
        )
    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal fun campaignsFromResponse(responseJson: JSONObject): List<CampaignEntity> {
        try {
            if (!responseJson.has(CAMPAIGNS)) return emptyList()
            val campaignArray = responseJson.getJSONArray(CAMPAIGNS)
            if (campaignArray.length() == 0) return emptyList()
            logJsonArray(tag, campaignArray)
            val campaigns = mutableListOf<CampaignEntity>()
            val mapper = PayloadMapper()
            for (i in 0 until campaignArray.length()) {
                try {
                    val campaignJson = campaignArray.getJSONObject(i)
                    val campaign = mapper.jsonToCampaignEntity(campaignJson)
                    campaigns.add(campaign)
                } catch (t: Throwable) {
                    sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag campaignsFromResponse() : " }
                }
            }
            return campaigns
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag campaignsFromResponse() : " }
        }
        return emptyList()
    }

    fun parseTestCampaignResponse(response: NetworkResponse): NetworkResult {
        return when (response) {
            is ResponseFailure -> {
                when (response.errorCode) {
                    DEFAULT_NETWORK_FAILURE_ERROR_CODE -> ResultFailure(
                        ERROR_MESSAGE_NO_INTERNET_CONNECTION
                    )

                    in 500..599 -> ResultFailure(
                        ERROR_MESSAGE_COULD_NOT_REACH_MOENGAGE_SERVER
                    )

                    in 400..499 -> ResultFailure(
                        JSONObject(response.errorMessage).getString(ERROR_DESCRIPTION)
                    )

                    else -> ResultFailure(ERROR_MESSAGE_NO_INTERNET_CONNECTION)
                }
            }

            is ResponseSuccess -> {
                val responseBody = response.data
                val responseJson = JSONObject(responseBody)
                val campaignPayload: CampaignPayload =
                    when (InAppType.valueOf(responseJson.getString(ResponseParser.INAPP_TYPE))) {
                        InAppType.HTML -> {
                            htmlCampaignFromJson(responseJson)
                        }

                        InAppType.NATIVE -> {
                            nativeCampaignFromJson(responseJson)
                        }
                    }
                ResultSuccess(campaignPayload)
            }
        }
    }
}