/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.repository.remote;

import static com.moengage.inapp.internal.model.enums.ViewType.CUSTOM_RATING;
import static com.moengage.inapp.internal.model.enums.ViewType.FEEDBACK_TEXT;

import android.view.ViewGroup;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.moengage.core.LogLevel;
import com.moengage.core.internal.logger.Logger;
import com.moengage.core.internal.utils.CoreUtils;
import com.moengage.core.internal.utils.MoEUtils;
import com.moengage.inapp.R;
import com.moengage.inapp.internal.InAppConstants;
import com.moengage.inapp.internal.exceptions.ParseException;
import com.moengage.inapp.internal.model.Animation;
import com.moengage.inapp.internal.model.Background;
import com.moengage.inapp.internal.model.Border;
import com.moengage.inapp.internal.model.CampaignPayload;
import com.moengage.inapp.internal.model.Color;
import com.moengage.inapp.internal.model.Font;
import com.moengage.inapp.internal.model.HtmlCampaignPayload;
import com.moengage.inapp.internal.model.HtmlMeta;
import com.moengage.inapp.internal.model.InAppComponent;
import com.moengage.inapp.internal.model.InAppContainer;
import com.moengage.inapp.internal.model.InAppWidget;
import com.moengage.inapp.internal.model.Margin;
import com.moengage.inapp.internal.model.NativeCampaignPayload;
import com.moengage.inapp.internal.model.Padding;
import com.moengage.inapp.internal.model.Widget;
import com.moengage.inapp.internal.model.actions.CallAction;
import com.moengage.inapp.internal.model.actions.Condition;
import com.moengage.inapp.internal.model.actions.ConditionAction;
import com.moengage.inapp.internal.model.actions.CopyAction;
import com.moengage.inapp.internal.model.actions.DismissAction;
import com.moengage.inapp.internal.model.actions.NavigateToSettingsAction;
import com.moengage.inapp.internal.model.actions.RatingChangeAction;
import com.moengage.inapp.internal.model.actions.SetTextAction;
import com.moengage.inapp.internal.model.actions.ShareAction;
import com.moengage.inapp.internal.model.actions.SmsAction;
import com.moengage.inapp.internal.model.actions.TrackAction;
import com.moengage.inapp.internal.model.actions.UserInputAction;
import com.moengage.inapp.internal.model.customrating.CustomRatingComponent;
import com.moengage.inapp.internal.model.customrating.RatingIcon;
import com.moengage.inapp.internal.model.customrating.RatingIconState;
import com.moengage.inapp.internal.model.enums.ClosePosition;
import com.moengage.inapp.internal.model.enums.DataTrackType;
import com.moengage.inapp.internal.model.enums.DisplaySize;
import com.moengage.inapp.internal.model.enums.InAppType;
import com.moengage.inapp.internal.model.enums.Orientation;
import com.moengage.inapp.internal.model.enums.RatingType;
import com.moengage.inapp.internal.model.enums.TemplateAlignment;
import com.moengage.inapp.internal.model.enums.UserInputType;
import com.moengage.inapp.internal.model.enums.ViewType;
import com.moengage.inapp.internal.model.enums.ViewVisibility;
import com.moengage.inapp.internal.model.enums.WidgetType;
import com.moengage.inapp.internal.model.style.ButtonStyle;
import com.moengage.inapp.internal.model.style.CloseStyle;
import com.moengage.inapp.internal.model.style.ContainerStyle;
import com.moengage.inapp.internal.model.style.CustomRatingStyle;
import com.moengage.inapp.internal.model.style.ImageStyle;
import com.moengage.inapp.internal.model.style.InAppStyle;
import com.moengage.inapp.internal.model.style.RatingIconStyle;
import com.moengage.inapp.internal.model.style.RatingStyle;
import com.moengage.inapp.internal.model.style.TextStyle;
import com.moengage.inapp.model.CampaignContext;
import com.moengage.inapp.model.actions.Action;
import com.moengage.inapp.model.actions.CustomAction;
import com.moengage.inapp.model.actions.NavigationAction;
import com.moengage.inapp.model.actions.RequestNotificationAction;
import com.moengage.inapp.model.enums.ActionType;
import com.moengage.inapp.model.enums.InAppPosition;
import com.moengage.inapp.model.enums.NavigationType;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * @author Umang Chamaria
 */
@VisibleForTesting
public class ResponseParser {

  private static final String TAG = InAppConstants.MODULE_TAG + "ResponseParser";

  NativeCampaignPayload selfHandledCampaignFromJson(JSONObject responseJson)
      throws JSONException {
    return new NativeCampaignPayload(
        responseJson.getString(CAMPAIGN_ID),
        responseJson.getString(CAMPAIGN_NAME),
        TemplateAlignment.setValue(
            responseJson.optString(TEMPLATE_ALIGNMENT, TemplateAlignment.CENTER.toString())
                .trim()
                .toUpperCase()),
        responseJson.getString(TEMPLATE_TYPE),
        responseJson.optLong(DISMISS_INTERVAL, -1),
        responseJson,
        responseJson.getString(PAYLOAD),
        CampaignContext.fromJson(responseJson.getJSONObject(CAMPAIGN_CONTEXT)),
        InAppType.valueOf(responseJson.getString(INAPP_TYPE)),
        com.moengage.inapp.internal.UtilsKt.screenOrientationFromJson(responseJson.getJSONArray(ORIENTATIONS))
    );
  }

  public NativeCampaignPayload campaignPayloadFromResponse(JSONObject responseJson)
          throws JSONException, ParseException {
    NativeCampaignPayload nativeCampaignPayload = new NativeCampaignPayload(
            responseJson.getString(CAMPAIGN_ID),
            responseJson.getString(CAMPAIGN_NAME),
            containerFromJson(responseJson, getJsonFromReferencePath(responseJson,
                    responseJson.getJSONObject(PRIMARY_CONTAINER)
                            .getString(REFERENCE_PATH)), true),
            responseJson.getString(TEMPLATE_TYPE),
            TemplateAlignment.setValue(
                    responseJson.optString(TEMPLATE_ALIGNMENT, TemplateAlignment.CENTER.toString())
                            .trim()
                            .toUpperCase()),
            responseJson.optLong(DISMISS_INTERVAL, -1),
            responseJson,
            CampaignContext.fromJson(responseJson.getJSONObject(CAMPAIGN_CONTEXT)),
            InAppType.valueOf(responseJson.getString(INAPP_TYPE)),
            com.moengage.inapp.internal.UtilsKt.screenOrientationFromJson(
                    responseJson.getJSONArray(ORIENTATIONS)),
            getPositionFromJson(responseJson)
    );
    validateMandatoryParams(nativeCampaignPayload);
    return nativeCampaignPayload;
  }

  private InAppContainer containerFromJson(JSONObject responseJSON, JSONObject containerJSON,
      boolean isPrimary) throws JSONException, ParseException {
    InAppStyle style = styleFromJson(responseJSON,
        getJsonFromReferencePath(responseJSON, containerJSON.getJSONObject(STYLE)
            .getString(REFERENCE_PATH)), WidgetType.CONTAINER, null);
    if (style == null) throw new ParseException("Style could not be parsed.");
    return new InAppContainer(containerJSON.getInt(ID), style,
      Orientation.setValue(containerJSON.getString(POSITION).trim().toUpperCase()),
      isPrimary,
      widgetForContainer(responseJSON, containerJSON.getJSONArray(WIDGETS)));
  }

  private ArrayList<Widget> widgetForContainer(JSONObject responseJSON, JSONArray widgetArray)
    throws JSONException, ParseException {
    ArrayList<Widget> widgetList = new ArrayList<>();
    for (int index = 0; index < widgetArray.length(); index++) {
      JSONObject widgetJSON = widgetArray.getJSONObject(index);
      WidgetType widgetType =
          WidgetType.setValue(widgetJSON.getString(TYPE).trim().toUpperCase());
      if (widgetType == WidgetType.WIDGET) {
        InAppWidget inAppWidget = widgetFromJson(responseJSON,
            getJsonFromReferencePath(responseJSON,
                widgetJSON.getString(REFERENCE_PATH)));
        widgetList.add(new Widget(widgetType, inAppWidget));
      } else if (widgetType == WidgetType.CONTAINER) {
        InAppContainer inAppContainer = containerFromJson(responseJSON,
            getJsonFromReferencePath(responseJSON,
                widgetJSON.getString(REFERENCE_PATH)), false);
        widgetList.add(new Widget(widgetType, inAppContainer));
      }
    }
    return widgetList;
  }

	@VisibleForTesting
	public InAppWidget widgetFromJson(JSONObject responseJson, JSONObject widgetJson)
    throws JSONException, ParseException {
    ViewType viewType =
        ViewType.setValue(widgetJson.getString(TYPE).trim().toUpperCase());
    return new InAppWidget(widgetJson.getInt(ID),
        viewType,
        getComponentFromJson(responseJson, getJsonFromReferencePath(responseJson,
            widgetJson.getJSONObject(COMPONENT)
                .getString(REFERENCE_PATH)), viewType),
        actionsForWidget(responseJson, widgetJson));
  }

  private List<Action> actionsForWidget(JSONObject responseJSON, JSONObject widgetJSON)
      throws JSONException {
    if (!widgetJSON.has(WIDGET_ACTION)) return null;
    return actionListFromJson(widgetJSON.getJSONObject(WIDGET_ACTION), responseJSON);
  }

  private List<Action> actionListFromJson(JSONObject actionsJson, JSONObject responseJSON)
      throws JSONException {
    ArrayList<Action> actionList = new ArrayList<>();
    for (Iterator<String> actionIterator = actionsJson.keys(); actionIterator.hasNext(); ) {
      Action action = actionFromJson(responseJSON, getJsonFromReferencePath(responseJSON,
          actionsJson.getJSONObject(actionIterator.next())
              .getString(REFERENCE_PATH)));
      if (action != null) actionList.add(action);
    }
    return actionList;
  }

  private Action actionFromJson(JSONObject responseJson, JSONObject actionJson) {
    try {
      ActionType actionType = ActionType.valueOf(
          actionJson.getString(ACTION_TYPE).trim().toUpperCase());
      switch (actionType) {
        case DISMISS:
          return new DismissAction(actionType);
        case TRACK_DATA:
          return trackActionFromJson(actionType, responseJson, actionJson);
        case NAVIGATE:
          return navigationActionFromJson(actionType, responseJson, actionJson);
        case SHARE:
          return shareActionFromJson(actionType, responseJson, actionJson);
        case COPY_TEXT:
          return copyActionFromJson(actionType, responseJson, actionJson);
        case CALL:
          return callActionFromJson(actionType, responseJson, actionJson);
        case SMS:
          return smsActionFromJson(actionType, responseJson, actionJson);
        case CUSTOM_ACTION:
          return customActionFromJson(actionType, responseJson, actionJson);
        case CONDITION_ACTION:
          return conditionActionFromJson(actionType, responseJson, actionJson);
        case USER_INPUT:
          return userInputActionJson(actionType, responseJson, actionJson);
        case REQUEST_NOTIFICATION_PERMISSION:
          return new RequestNotificationAction(actionType, -1);
        case NAVIGATE_SETTINGS_NOTIFICATIONS:
          return new NavigateToSettingsAction(actionType);
        case RATING_CHANGE:
          return ratingChangeActionFromJson(actionType, actionJson, responseJson);
        case SET_TEXT:
          return setTextActionFromJson(actionType, actionJson, responseJson);
      }
    } catch (Exception e) {
      Logger.print(LogLevel.ERROR, e, () -> TAG + " actionFromJson() ");
    }
    return null;
  }

  private TrackAction trackActionFromJson(ActionType actionType, JSONObject responseJson,
      JSONObject actionJson) throws JSONException {
    return new TrackAction(
        actionType,
        DataTrackType.setValue(
            actionJson.getString(TRACK_TYPE).trim().toUpperCase()),
        actionJson.has(VALUE) ? getStringFromPath(responseJson,
            actionJson.getJSONObject(VALUE).getString(REFERENCE_PATH)) : null,
        getStringFromPath(responseJson,
            actionJson.getJSONObject(NAME).getString(REFERENCE_PATH)),
        dataMapForAction(responseJson, actionJson)
    );
  }

  private NavigationAction navigationActionFromJson(ActionType actionType, JSONObject responseJson,
      JSONObject actionJson) throws JSONException {
    return new NavigationAction(
        actionType,
        NavigationType.valueOf(
            actionJson.getString(NAVIGATION_TYPE).trim().toUpperCase()),
        getStringFromPath(responseJson,
            actionJson.getJSONObject(VALUE).getString(REFERENCE_PATH)),
        dataMapForAction(responseJson, actionJson)
    );
  }

  private CopyAction copyActionFromJson(ActionType actionType, JSONObject responseJson,
      JSONObject actionJson) throws JSONException {
    return new CopyAction(
        actionType,
        actionJson.has(MESSAGE) ? getStringFromPath(responseJson,
            actionJson.getJSONObject(MESSAGE).getString(REFERENCE_PATH))
            : null,
        getStringFromPath(responseJson,
            actionJson.getJSONObject(VALUE).getString(REFERENCE_PATH))
    );
  }

  private ShareAction shareActionFromJson(ActionType actionType, JSONObject responseJson,
      JSONObject actionJson) throws JSONException {
    return new ShareAction(
        actionType,
        getStringFromPath(responseJson,
            actionJson.getJSONObject(VALUE).getString(REFERENCE_PATH))
    );
  }

  private CustomAction customActionFromJson(ActionType actionType, JSONObject responseJson,
      JSONObject actionJson) throws JSONException {
    return new CustomAction(
        actionType
        , dataMapForAction(responseJson, actionJson));
  }

  private CallAction callActionFromJson(ActionType actionType, JSONObject responseJson,
      JSONObject actionJson) throws JSONException {
    return new CallAction(actionType, getStringFromPath(responseJson,
        actionJson.getJSONObject(VALUE).getString(REFERENCE_PATH)));
  }

  private SmsAction smsActionFromJson(ActionType actionType, JSONObject responseJson,
      JSONObject actionJson) throws JSONException {
    return new SmsAction(actionType, getStringFromPath(responseJson,
        actionJson.getJSONObject(VALUE).getString(REFERENCE_PATH)),
        getStringFromPath(responseJson,
            actionJson.getJSONObject(MESSAGE)
                .getString(REFERENCE_PATH)));
  }

  private ConditionAction conditionActionFromJson(ActionType actionType, JSONObject responseJson,
      JSONObject actionJson) throws JSONException, ParseException {
    if (!actionJson.has(CONDITIONS)) {
      throw new ParseException("Mandatory key \"conditions\" missing.");
    }
    InAppWidget widget = widgetFromJson(responseJson, getJsonFromReferencePath(responseJson,
        actionJson.getJSONObject(WIDGET_ID).getString(REFERENCE_PATH)));
    JSONArray conditionsArray = actionJson.getJSONArray(CONDITIONS);
    List<Condition> conditionList = new ArrayList<>();
    for (int i = 0; i < conditionsArray.length(); i++) {
      JSONObject conditionJson = conditionsArray.getJSONObject(i);
      conditionList.add(new Condition(conditionJson.getJSONObject(ATTRIBUTE),
          actionListFromReferences(conditionJson.getJSONObject(WIDGET_ACTIONS), responseJson)));
    }
    return new ConditionAction(actionType, conditionList, widget.id);
  }

  private UserInputAction userInputActionJson(ActionType actionType, JSONObject responseJson,
      JSONObject actionJson) throws JSONException, ParseException {
    InAppWidget widget = widgetFromJson(responseJson, getJsonFromReferencePath(responseJson,
        actionJson.getJSONObject(WIDGET_ID).getString(REFERENCE_PATH)));

    return new UserInputAction(actionType,
        UserInputType.valueOf(actionJson.getString(INPUT_TYPE).trim().toUpperCase()),
        widget.id,
        actionListFromReferences(actionJson.getJSONObject(WIDGET_ACTIONS), responseJson)
    );
  }

  private List<Action> actionListFromReferences(JSONObject actionListJson, JSONObject responseJson)
      throws JSONException {
    List<Action> actionList = new ArrayList<>();
    if (actionListJson.length() == 0) return actionList;
    Iterator<String> actionKeys = actionListJson.keys();
    while (actionKeys.hasNext()){
      Action action = actionFromJson(responseJson,
          getJsonFromReferencePath(responseJson,
              actionListJson.getJSONObject(actionKeys.next()).getString(REFERENCE_PATH)));
      if (action != null){
        actionList.add(action);
      }
    }
    return actionList;
  }

  private InAppComponent getComponentFromJson(JSONObject responseJson, JSONObject componentJson,
      ViewType viewType) throws JSONException, ParseException {
    InAppStyle style = styleFromJson(responseJson, getJsonFromReferencePath(responseJson,
        componentJson.getJSONObject(STYLE)
            .getString(REFERENCE_PATH)), WidgetType.WIDGET, viewType);
    if (style == null) throw new ParseException("Style could not be parsed.");
		if (viewType != ViewType.RATING && viewType != CUSTOM_RATING && viewType != FEEDBACK_TEXT && !componentJson.has(CONTENT)) {
			throw new ParseException("Mandatory param content missing");
		}
    if (viewType == CUSTOM_RATING) {
      return new CustomRatingComponent(
        style,
        ratingIconListFromJson(responseJson, (CustomRatingStyle) style, componentJson)
      );
    } else {
      return new InAppComponent(
        componentJson.has(CONTENT) ? getStringFromPath(responseJson,
          componentJson.getJSONObject(CONTENT).getString(REFERENCE_PATH)) : null,
        style
      );
    }
  }

  private InAppStyle styleFromJson(JSONObject responseJson,
                                   JSONObject styleJson,
                                   WidgetType widgetType, ViewType viewType)
    throws JSONException, ParseException {
    InAppStyle baseStyle = baseStyleFromJson(styleJson);
    switch (widgetType) {
      case CONTAINER:
        return containerStyleFromJson(responseJson, styleJson, baseStyle);
      case WIDGET:
        switch (viewType) {
					case TEXT:
						return textStyleFromJson(responseJson, styleJson, baseStyle, styleJson.optInt(MAX_LINES, -1));
					case FEEDBACK_TEXT:
						return textStyleFromJson(responseJson, styleJson, baseStyle, styleJson.optInt(MAX_LINES, 1));
          case IMAGE:
            return imageStyleFromJson(responseJson, styleJson, baseStyle);
          case BUTTON:
            return buttonStyleFromJson(responseJson, styleJson, baseStyle);
          case RATING:
            return ratingStyleFromJson(responseJson, styleJson, baseStyle);
          case CLOSE_BUTTON:
            return closeStyleFromJson(responseJson, styleJson, baseStyle);
          case VIDEO:
            return baseStyle;
          case CUSTOM_RATING:
            return customRatingStyleFromJson(styleJson, baseStyle);
				}
        break;
    }
    return null;
  }

  private ContainerStyle containerStyleFromJson(JSONObject responseJson,
      JSONObject styleJson, InAppStyle baseStyle) throws JSONException {
    return new ContainerStyle(
        baseStyle,
        borderFromJson(styleJson),
        backgroundFromJson(styleJson, responseJson),
        animationFromJson(styleJson),
        styleJson.has(DISPLAY_SIZE)?
          DisplaySize.valueOf(styleJson.getString(DISPLAY_SIZE).trim().toUpperCase()):
          null
    );
  }

	private TextStyle textStyleFromJson(JSONObject responseJson, JSONObject styleJson,
																			InAppStyle baseStyle, int maxLines) throws JSONException {
		return new TextStyle(
			baseStyle,
			fontFromJson(styleJson),
			backgroundFromJson(styleJson, responseJson),
			borderFromJson(styleJson),
			ViewVisibility.valueOf(
				styleJson.optString(
					INITIAL_STATE, ViewVisibility.VISIBLE.toString()
				).toUpperCase()
			),
			maxLines
		);
	}

  private ImageStyle imageStyleFromJson(JSONObject responseJson, JSONObject styleJson,
      InAppStyle baseStyle) throws JSONException {
    return new ImageStyle(
        baseStyle,
        borderFromJson(styleJson),
        styleJson.getDouble(REAL_HEIGHT),
        styleJson.getDouble(REAL_WIDTH)
    );
  }

  private ButtonStyle buttonStyleFromJson(JSONObject responseJson, JSONObject styleJson,
      InAppStyle baseStyle) throws JSONException {
    return new ButtonStyle(
        baseStyle,
        fontFromJson(styleJson),
        backgroundFromJson(styleJson, responseJson),
        borderFromJson(styleJson),
        styleJson.getInt(MINIMUM_HEIGHT)
    );
  }

  private RatingStyle ratingStyleFromJson(JSONObject responseJson, JSONObject styleJson,
      InAppStyle baseStyle) throws JSONException, ParseException {
    if (!styleJson.has(RATING_STYLE)) {
      throw new ParseException("Mandatory key \"rating_style\" missing.");
    }
    JSONObject ratingStyle = styleJson.getJSONObject(RATING_STYLE);
    return new RatingStyle(
        baseStyle,
        borderFromJson(styleJson),
        colorFromJson(ratingStyle.getJSONObject(COLOR)),
        ratingStyle.getInt(NUMBER_OF_STARS),
        ratingStyle.getBoolean(HALF_STEP_ALLOWED),
        styleJson.getDouble(REAL_HEIGHT)
    );
  }

  private CloseStyle closeStyleFromJson(JSONObject responseJson, JSONObject styleJson,
      InAppStyle baseStyle) throws JSONException {
    return new CloseStyle(
        baseStyle,
        styleJson.has(WIDGET_FLOAT)?
            ClosePosition.setValue(styleJson.getString(WIDGET_FLOAT).trim().toUpperCase()):
            ClosePosition.RIGHT
    );
  }

  private Font fontFromJson(JSONObject styleJSONObject) throws JSONException {
    JSONObject fontJson = styleJSONObject.getJSONObject(FONT);
    return new Font(
        fontJson.optString(FONT_NAME),
        fontJson.getInt(SIZE),
        fontJson.has(COLOR) ? colorFromJson(fontJson.getJSONObject(
            COLOR)) : new Color(0, 0, 0, 1)
    );
  }

  private Animation animationFromJson(JSONObject styleJSONObject) throws JSONException {
    if (!styleJSONObject.has(ANIMATION)) return null;

    JSONObject animationJson = styleJSONObject.getJSONObject(ANIMATION);
    return new Animation(animationJson.has(ENTRY) ? getEntryAnimation(
        animationJson.getString(ENTRY)) : -1,
        animationJson.has(EXIT) ? getExitAnimation(
            animationJson.getString(EXIT)) : -1);
  }

  private Background backgroundFromJson(JSONObject styleJSONObject, JSONObject responseJSON)
      throws JSONException {
    if (!styleJSONObject.has(BACKGROUND)) return null;
    JSONObject backgroundJson = styleJSONObject.getJSONObject(BACKGROUND);
    return new Background(
        backgroundJson.has(COLOR) ? colorFromJson(backgroundJson.getJSONObject(
            COLOR)) : null,
        contentFromBackground(backgroundJson, responseJSON)
    );
  }

  private Border borderFromJson(JSONObject styleJSONObject) throws JSONException {
    if (styleJSONObject.has(BORDER)) {
      JSONObject borderJson = styleJSONObject.getJSONObject(BORDER);
      return new Border(borderJson.has(COLOR) ?
          colorFromJson(borderJson.getJSONObject(COLOR)) : null,
          borderJson.optDouble(RADIUS, 0.0),
          borderJson.optDouble(WIDTH, 0.0));
    } else {
      return null;
    }
  }

  private Margin marginFromJson(JSONObject styleJSONObject) throws JSONException {
    if (styleJSONObject.has(MARGIN)) {
      JSONObject marginJSON = styleJSONObject.getJSONObject(MARGIN);
      return new Margin(marginJSON.optDouble(LEFT, 0.0),
          marginJSON.optDouble(RIGHT, 0.0),
          marginJSON.optDouble(TOP, 0.0),
          marginJSON.optDouble(BOTTOM, 0.0));
    } else {
      return new Margin(0.0, 0.0, 0.0, 0.0);
    }
  }

  private Padding paddingFromJson(JSONObject styleJson) throws JSONException {
    if (styleJson.has(PADDING)) {
      JSONObject marginJSON = styleJson.getJSONObject(PADDING);
      return new Padding(marginJSON.optDouble(LEFT, 0.0),
          marginJSON.optDouble(RIGHT, 0.0),
          marginJSON.optDouble(TOP, 0.0),
          marginJSON.optDouble(BOTTOM, 0.0));
    } else {
      return new Padding(0.0, 0.0, 0.0, 0.0);
    }
  }

  @Nullable private String contentFromBackground(JSONObject backgroundJSON, JSONObject responseJSON)
      throws JSONException {
    String contentString = null;
    if (backgroundJSON.has(IMAGE)) {
      contentString = getStringFromPath(responseJSON,
          backgroundJSON.getJSONObject(IMAGE)
              .getString(REFERENCE_PATH));
    }
    if (CoreUtils.isNullOrEmpty(contentString)) contentString = null;
    return contentString;
  }

  private String getStringFromPath(JSONObject responseJSON, String contentPath)
      throws JSONException {
    String[] pathPieces = contentPath.split("/");
    JSONObject campaignPayload = responseJSON;
    for (int i = 1; i < pathPieces.length - 1; i++)
      campaignPayload = campaignPayload.getJSONObject(pathPieces[i]);
    return campaignPayload.getString(pathPieces[pathPieces.length - 1]);
  }

  private JSONObject getJsonFromReferencePath(JSONObject responseJSON, String contentPath)
      throws JSONException {
    String[] pathPieces = contentPath.split("/");
    JSONObject campaignPayload = responseJSON;
    for (int i = 1; i < pathPieces.length; i++)
      campaignPayload = campaignPayload.getJSONObject(pathPieces[i]);
    return campaignPayload;
  }

  @VisibleForTesting
  public int getEntryAnimation(String entryAnimation) {
    switch (entryAnimation) {
      case IN_APP_ANIMATION_SLIDE_UP:
        return R.anim.slide_up_in;
      case IN_APP_ANIMATION_SLIDE_DOWN:
        return R.anim.slide_down_in;
      case IN_APP_ANIMATION_SLIDE_LEFT:
        return R.anim.slide_right_in;
      case IN_APP_ANIMATION_SLIDE_RIGHT:
        return R.anim.slide_left_in;
      case IN_APP_ANIMATION_FADE_IN:
        return R.anim.fade_in;
      default:
        return -1;
    }
  }

  @VisibleForTesting
  public int getExitAnimation(String exitAnimation) {
    switch (exitAnimation) {
      case IN_APP_ANIMATION_SLIDE_UP:
        return R.anim.slide_up_out;
      case IN_APP_ANIMATION_SLIDE_DOWN:
        return R.anim.slide_down_out;
      case IN_APP_ANIMATION_SLIDE_LEFT:
        return R.anim.slide_left_out;
      case IN_APP_ANIMATION_SLIDE_RIGHT:
        return R.anim.slide_right_out;
      case IN_APP_ANIMATION_FADE_OUT:
        return R.anim.fade_out;
      default:
        return -1;
    }
  }

  private Map<String, Object> dataMapForAction(JSONObject responseJson, JSONObject actionJson)
      throws JSONException {
    return actionJson.has(ATTR_DATA_MAP) ? MoEUtils.jsonToMap(
        getJsonFromReferencePath(responseJson,
            actionJson.getJSONObject(ATTR_DATA_MAP)
                .getString(REFERENCE_PATH)))
        : new HashMap<String, Object>();
  }

  private Color colorFromJson(JSONObject colorJson) throws JSONException {
    return new Color(
        colorJson.getInt(COLOR_RED),
        colorJson.getInt(COLOR_GREEN),
        colorJson.getInt(COLOR_BLUE),
        (float) colorJson.getDouble(COLOR_ALPHA)
    );
  }


  HtmlCampaignPayload htmlCampaignFromJson(JSONObject responseJson)
      throws JSONException, ParseException {
    HtmlCampaignPayload htmlCampaignPayload = new HtmlCampaignPayload(
        responseJson.getString(CAMPAIGN_ID),
        responseJson.getString(CAMPAIGN_NAME),
        responseJson.getString(TEMPLATE_TYPE),
        responseJson.optLong(DISMISS_INTERVAL, -1),
        responseJson,
        CampaignContext.fromJson(responseJson.getJSONObject(CAMPAIGN_CONTEXT)),
        InAppType.valueOf(responseJson.getString(INAPP_TYPE)),
        com.moengage.inapp.internal.UtilsKt.screenOrientationFromJson(
            responseJson.getJSONArray(ORIENTATIONS)),
        responseJson.has(HTML_META)? htmlMetaFromJson(responseJson.getJSONObject(HTML_META)): null,
        responseJson.getString(PAYLOAD));
    validateMandatoryParams(htmlCampaignPayload);
    return htmlCampaignPayload;
  }

  private HtmlMeta htmlMetaFromJson(JSONObject jsonObject) throws JSONException {
    JSONObject payloadJson = jsonObject.optJSONObject(ASSETS);
    if (payloadJson == null || payloadJson.length() == 0) {
      return new HtmlMeta(new HashMap<>());
    }
    Map<String, String> map = new HashMap<>(payloadJson.length());
    Iterator<String> iterator = payloadJson.keys();
    try {
      while (iterator.hasNext()) {
        String key = iterator.next();
        map.put(key, payloadJson.getString(key));
      }
    } catch (Exception e) {
      Logger.print(LogLevel.ERROR, e, () -> TAG + " htmlMetaFromJson() ");
    }
    return new HtmlMeta(map);
  }

  private void validateMandatoryParams(CampaignPayload campaignPayload) throws ParseException {
    if (CoreUtils.isNullOrEmpty(campaignPayload.getTemplateType())) {
      throw new ParseException("mandatory key \"template_type\" cannot be empty.");
    }

    if (campaignPayload.getSupportedOrientations().isEmpty()) {
      throw new ParseException("mandatory key \"orientations\" cannot be empty.");
    }

    if (campaignPayload.getInAppType() == InAppType.HTML && CoreUtils.isNullOrEmpty(
        ((HtmlCampaignPayload) campaignPayload).getHtmlPayload())) {
      throw new ParseException("mandatory key \"payload\" cannot be empty.");
    }
  }

  private InAppPosition getPositionFromJson(JSONObject responseJson) throws JSONException, ParseException {
    if (responseJson.getString(TEMPLATE_TYPE).equals(InAppConstants.IN_APP_TEMPLATE_TYPE_NON_INTRUSIVE)) {
      if (!responseJson.has(POSITION)) {
        throw new ParseException("mandatory key \"position\" cannot be empty");
      }
      return InAppPosition.valueOf(responseJson.getString(POSITION).trim().toUpperCase());
    } else {
      return InAppPosition.ANY;
    }
  }

  private RatingChangeAction ratingChangeActionFromJson(ActionType actionType, JSONObject actionJson, JSONObject responseJson) throws JSONException {
    return new RatingChangeAction(
      actionType,
      actionListFromReferences(actionJson.getJSONObject(WIDGET_ACTIONS), responseJson)
    );
  }

  private SetTextAction setTextActionFromJson(ActionType actionType, JSONObject actionJson,
                                              JSONObject responseJson) throws JSONException, ParseException {
    InAppWidget widget = widgetFromJson(responseJson, getJsonFromReferencePath(responseJson,
      actionJson.getJSONObject(WIDGET_ID).getString(REFERENCE_PATH)));
    return new SetTextAction(
      actionType,
      widget.id
    );
  }

  private CustomRatingStyle customRatingStyleFromJson(JSONObject styleJson,
                                                      InAppStyle baseStyle) throws JSONException, ParseException {
    if (!styleJson.has(RATING_STYLE)) {
      throw new ParseException("Mandatory key \""+ RATING_STYLE +"\" missing.");
    }
    JSONObject ratingStyle = styleJson.getJSONObject(RATING_STYLE);
    if (!ratingStyle.has(NUMBER_OF_RATINGS)) {
      throw new ParseException("Mandatory key \"" + NUMBER_OF_RATINGS+ "\" missing.");
    }
    if (!ratingStyle.has(RATING_TYPE)) {
      throw new ParseException("Mandatory key \"" + RATING_TYPE+ "\" missing.");
    }
    return new CustomRatingStyle(
      baseStyle,
      borderFromJson(styleJson),
      styleJson.getDouble(REAL_HEIGHT),
      ratingStyle.getInt(NUMBER_OF_RATINGS),
      RatingType.valueOf(ratingStyle.getString(RATING_TYPE).toUpperCase())
    );
  }


  private Map<Integer, RatingIcon> ratingIconListFromJson(JSONObject responseJson, CustomRatingStyle style,
																													JSONObject componentJson) throws ParseException, JSONException {
    if (!componentJson.has(RATING_ICONS)) {
      throw new ParseException("Mandatory param \"" + RATING_ICONS + "\" missing");
    }
    JSONObject ratingIconsJson = componentJson.getJSONObject(RATING_ICONS);
    Map<Integer, RatingIcon> ratingIconMap = new HashMap<Integer, RatingIcon>();
    int numberOfRatings = style.getNumberOfRatings();
    for (int i = 1; i <= numberOfRatings; i++) {
			ratingIconMap.put(i, ratingIconFromJson(style, responseJson,
        ratingIconsJson.getJSONObject(String.valueOf(i))));
    }
    return ratingIconMap;
  }

  private RatingIcon ratingIconFromJson(CustomRatingStyle style,
                                        JSONObject responseJson,
                                        JSONObject ratingIconJson) throws JSONException, ParseException {
    if (!ratingIconJson.has(VALUE)) {
      throw new ParseException("Mandatory param \"" + VALUE + "\" missing");
    }
    if (!ratingIconJson.has(DESCRIPTION)) {
      throw new ParseException("Mandatory param \"" + DESCRIPTION + "\" missing");
    }
    if (!ratingIconJson.has(SELECTED_STATE)) {
      throw new ParseException("Mandatory param \"" + SELECTED_STATE + "\" missing");
    }
    if (!ratingIconJson.has(UNSELECTED_STATE)) {
      throw new ParseException("Mandatory param \"" + UNSELECTED_STATE + "\" missing");
    }
    return new RatingIcon(
      ratingIconJson.getInt(VALUE),
      getStringFromPath(responseJson,
        ratingIconJson.getJSONObject(DESCRIPTION).getString(REFERENCE_PATH)),
      ratingIconStateFromJson(style, responseJson, ratingIconJson.getJSONObject(SELECTED_STATE)),
      ratingIconStateFromJson(style, responseJson, ratingIconJson.getJSONObject(UNSELECTED_STATE))
    );
  }

  private RatingIconState ratingIconStateFromJson(CustomRatingStyle style,
                                                  JSONObject responseJson,
                                                  JSONObject ratingIconStateJson) throws JSONException, ParseException {
		if (!ratingIconStateJson.has(STYLE)) {
			throw new ParseException("Mandatory param \"" + STYLE + "\" missing");
		}
		return new RatingIconState(
			ratingIconStyleFromJson(
				getJsonFromReferencePath(responseJson, ratingIconStateJson.getJSONObject(STYLE)
					.getString(REFERENCE_PATH)), responseJson
			),
			style.getRatingType() != RatingType.STAR ? getIconFromJson(responseJson, ratingIconStateJson) : ""
		);
	}

  private InAppStyle baseStyleFromJson(JSONObject styleJson) throws JSONException {
    return new InAppStyle(
      styleJson.optDouble(HEIGHT, ViewGroup.LayoutParams.WRAP_CONTENT),
      styleJson.getDouble(WIDTH),
      marginFromJson(styleJson),
      paddingFromJson(styleJson),
      styleJson.getBoolean(DISPLAY)
    );
  }

	private RatingIconStyle ratingIconStyleFromJson(JSONObject styleJSONObject,
																									JSONObject responseJSON) throws JSONException, ParseException {
		Background background = backgroundFromJson(styleJSONObject, responseJSON);
		Border border = borderFromJson(styleJSONObject);
		if (background == null) {
			throw new ParseException("Mandatory param \"" + BACKGROUND + "\" missing");
		}
		if (border == null) {
			throw new ParseException("Mandatory param \"" + BORDER + "\" missing");
		}
		return new RatingIconStyle(
			background,
			border
		);
	}

	private String getIconFromJson(JSONObject responseJson, JSONObject ratingIconStateJson) throws JSONException {
		return getStringFromPath(responseJson,
			ratingIconStateJson.getString(ICON));
	}
  /**
   * Response attribute which denotes the delay to be set for two consecutive inapp messages
   */
  private static final String GLOBAL_MINIMUM_DELAY = "min_delay_btw_inapps";
  /**
   * Response attribute which denotes the delay before which /inapps/live API calls shouldn't be
   * made.
   */
  private static final String SYNC_INTERVAL = "sync_interval";
  /**
   * Response attribute which contains campaign array
   */
  private static final String CAMPAIGNS = "campaigns";

  // meta call response attributes
  /**
   * Denotes the campaign ID for the In-App message
   */
  private static final String CAMPAIGN_ID = "campaign_id";

  private static final String CAMPAIGN_TYPE = "campaign_type";

  private static final String IN_APP_ANIMATION_SLIDE_UP = "SLIDE_UP";
  private static final String IN_APP_ANIMATION_SLIDE_DOWN = "SLIDE_DOWN";
  private static final String IN_APP_ANIMATION_SLIDE_LEFT = "SLIDE_LEFT";
  private static final String IN_APP_ANIMATION_SLIDE_RIGHT = "SLIDE_RIGHT";
  @VisibleForTesting
  public static final String IN_APP_ANIMATION_FADE_IN = "FADE_IN";
  @VisibleForTesting
  public static final String IN_APP_ANIMATION_FADE_OUT = "FADE_OUT";

  private static final String WIDGETS = "widgets";

  private static final String PAYLOAD = "payload";

  private static final String PRIMARY_CONTAINER = "primary_container";

  private static final String PRIMARY_WIDGET = "primary_widget";

  private static final String MARGIN = "margin";

  private static final String BORDER = "border";

  private static final String BACKGROUND = "background";

  private static final String ANIMATION = "animation";

  private static final String HEIGHT = "height";

  private static final String WIDTH = "width";

  private static final String LEFT = "left";
  private static final String RIGHT = "right";
  private static final String TOP = "top";
  private static final String BOTTOM = "bottom";

  private static final String CONTENT = "content";

  private static final String ENTRY = "entry";
  private static final String EXIT = "exit";

  private static final String RADIUS = "radius";

  private static final String FONT = "font";
  private static final String SIZE = "size";
  private static final String FONT_NAME = "font_name";

  private static final String STYLE = "style";

  private static final String ACTION_TYPE = "action_type";

  private static final String TRACK_TYPE = "track_type";

  private static final String VALUE = "value";

  private static final String REFERENCE_PATH = "_ref";

  private static final String WIDGET_FLOAT = "float";

  private static final String ID = "id";

  private static final String TYPE = "type";

  private static final String COMPONENT = "component";

  private static final String POSITION = "position";

  private static final String MINIMUM_HEIGHT = "min_height";

  private static final String NAVIGATION_TYPE = "navigation_type";

  private static final String NAME = "name";

  private static final String MESSAGE = "message";

  private static final String ATTR_DATA_MAP = "data_map";

  private static final String CAMPAIGN_NAME = "campaign_name";

  private static final String TEMPLATE_ALIGNMENT = "template_alignment";

  private static final String REAL_HEIGHT = "realHeight";

  private static final String REAL_WIDTH = "realWidth";

  private static final String WIDGET_ACTION = "action";
  private static final String WIDGET_ACTIONS = "actions";

  private static final String DISPLAY = "display";

  private static final String PADDING = "padding";

  private static final String IMAGE = "image";

  // color properties
  private static final String COLOR = "color";
  private static final String COLOR_ALPHA = "a";
  private static final String COLOR_RED = "r";
  private static final String COLOR_GREEN = "g";
  private static final String COLOR_BLUE = "b";

  static final String TEMPLATE_TYPE = "template_type";

  private static final String DISMISS_INTERVAL = "dismiss_interval";

  private static final String WIDGET_ID = "widget_id";

  // rating styles
  private static final String RATING_STYLE = "rating_style";
  private static final String NUMBER_OF_STARS = "number_of_stars";
  private static final String HALF_STEP_ALLOWED = "half_step_allowed";

  private static final String INPUT_TYPE = "input_type";

  private static final String CONDITIONS = "conditions";
  private static final String ATTRIBUTE = "attribute";

  private static final String ERROR = "error";

  private static final String ERROR_MESSAGE_NO_INTERNET_CONNECTION = "No Internet Connection.\n"
      + "Please connect to internet and try again.";
  private static final String ERROR_MESSAGE_COULD_NOT_REACH_MOENGAGE_SERVER = "Could not reach "
      + "MoEngage Server.\n Please try again or contact MoEngage Support.";

  private static final String CAMPAIGN_CONTEXT = "campaign_context";
  public static final String INAPP_TYPE = "inapp_type";
  private static final String ORIENTATIONS = "orientations";
  private static final String HTML_META = "html_meta";
  private static final String ASSETS = "assets";

  private static final String DISPLAY_SIZE = "display_size";

  private static final String NUMBER_OF_RATINGS = "number_of_ratings";

  private static final String RATING_TYPE = "rating_type";

  private static final String RATING_ICONS = "rating_icons";
  private static final String ICON = "icon";
  private static final String DESCRIPTION = "description";
  private static final String SELECTED_STATE = "selected_state";
  private static final String UNSELECTED_STATE = "unselected_state";

  private static final String INITIAL_STATE = "initial_state";

  private static final String MAX_LINES = "maxLines";

}
