/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.tasks

import android.content.Context
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.currentSeconds
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.InAppInstanceProvider
import com.moengage.inapp.internal.model.enums.InAppType
import com.moengage.inapp.internal.repository.InAppFileManager
import com.moengage.inapp.internal.repository.PayloadMapper

/**
 * @author Arshiya Khanum
 */

private const val DEFAULT_HTML_ASSETS_DELETE_INTERVAL = 900L

internal class AppOpenHandler(private val context: Context, private val sdkInstance: SdkInstance) {

    private val tag = "${InAppConstants.MODULE_TAG}AppOpenJob"
    private val repository = InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance)

    fun onAppOpen() {
        try {
            sdkInstance.logger.log { "$tag onAppOpen() : Processing app open" }
            val currentTime = currentSeconds()
            if (shouldClearHtmlAssets(currentTime)) {
                clearHtmlAssetsCache()
                repository.storeHtmlAssetsDeleteTime(currentTime)
            }
            InAppInstanceProvider.getControllerForInstance(sdkInstance).syncMeta(context)
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag onAppOpen() : " }
        }
    }

    private fun clearHtmlAssetsCache() {
        sdkInstance.logger.log { "$tag clearHtmlAssetsCache() : clearing html assets" }
        val campaignIds = PayloadMapper().entityToCampaign(repository.getAllCampaigns())
            .filter { it.campaignMeta.inAppType == InAppType.HTML }
            .map { it.campaignMeta.campaignId }
            .toSet()
        InAppFileManager(context, sdkInstance).deleteHtmlAssetsForCampaignIds(campaignIds)
    }

    private fun shouldClearHtmlAssets(currentTime: Long): Boolean =
        (repository.getLastHtmlAssetsDeleteTime() + DEFAULT_HTML_ASSETS_DELETE_INTERVAL < currentTime)
}